<?php
namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Consultation;
use App\Models\Medication;
use App\Models\Patient;
use App\Models\Prescription;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class PrescriptionController extends Controller
{
    // Display a listing of prescriptions
    public function index(Request $request)
    {
        $query = Prescription::with([
            'patient',
            'medication',
            'prescriber',
            'consultation.clinic', // Include consultation and nested clinic
        ])->latest();

        if ($search = $request->input('search')) {
            $query->whereHas('patient', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%");
            })->orWhereHas('medication', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%");
            });
        }

        if ($prescriber = $request->input('prescriber')) {
            $query->where('prescribed_by', $prescriber);
        }

        $prescriptions = $query->paginate(10)
            ->withQueryString();

        $prescribers = User::whereIn('role', [UserRole::Doctor, UserRole::Pharmacist])
            ->get();

        return Inertia::render('Prescriptions/Index', [
            'prescriptions' => $prescriptions,
            'filters'       => $request->only(['search', 'prescriber']),
            'prescribers'   => $prescribers,
        ]);
    }

    public function create(Consultation $consultation)
    {
        // Eager load relationships you want to include
        $consultation->load([
            'patient',
            'doctor',
            'clinic',
            'medicationUsages',
            'dispensations',
            'triage',
            // add any other relationships you need
        ]);

        return Inertia::render('Prescriptions/Create', [
            'consultation'     => $consultation,
            'patient'          => $consultation->patient, // already loaded but included for convenience
            'medications'      => Medication::all(),
            'routes'           => ['Oral', 'IV', 'IM', 'Subcutaneous', 'Topical'],
            'foodRelations'    => ['Before Food', 'After Food', 'With Food', 'Any Time'],
            'frequencyOptions' => [
                '1'   => 'Once daily',
                '2'   => 'Twice daily',
                '3'   => 'Three times daily',
                '0.5' => 'Every other day',
                '1/7' => 'Once weekly',
            ],
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'consultation_id'             => 'required|exists:consultations,id',
            'patient_id'                  => 'required|exists:patients,id',
            'medications'                 => 'required|array|min:1',
            'medications.*.medication_id' => 'nullable',
            'medications.*.name'          => 'required_without:medications.*.medication_id|string|max:255',
            'medications.*.dosage'        => 'required|string|max:50',
            'medications.*.frequency'     => 'required|string|max:50',
            'medications.*.duration'      => 'required|string|max:50',
            'medications.*.quantity'      => 'required|integer|min:1',
            'medications.*.route'         => 'required|string|max:50',
            'medications.*.food_relation' => 'required|string|max:50',
            'medications.*.notes'         => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $prescriptions = [];
            foreach ($validated['medications'] as $medication) {
                // Check if the medication_id exists and is numeric (existing medication)
                if (! empty($medication['medication_id']) && is_numeric($medication['medication_id'])) {
                    // Existing medication
                    $medicationModel = Medication::findOrFail($medication['medication_id']);
                } else {
                    // New medication - use either the name from the input or the medication_id as name
                    $medicationName = $medication['name'] ?? $medication['medication_id'] ?? 'Unknown Medication';

                    // Capitalize the medication name
                    $medicationName = ucwords(strtolower($medicationName));

                    $medicationModel = Medication::create([
                        'name'         => $medicationName,
                        'dosage'       => $medication['dosage'],
                        'form'         => 'Unknown',
                        'unit'         => 'Unknown',
                        'instructions' => $medication['notes'] ?? 'No specific instructions',
                    ]);
                }

                // Create prescription
                $prescriptions[] = Prescription::create([
                    'consultation_id' => $validated['consultation_id'],
                    'patient_id'      => $validated['patient_id'],
                    'prescribed_by'   => Auth::id(),
                    'medication_id'   => $medicationModel->id,
                    'dosage'          => $medication['dosage'],
                    'frequency'       => $medication['frequency'],
                    'duration'        => $medication['duration'],
                    'quantity'        => $medication['quantity'],
                    'route'           => $medication['route'],
                    'food_relation'   => $medication['food_relation'],
                    'notes'           => $medication['notes'] ?? null,
                ]);
            }

            DB::commit();

            return redirect()
                ->route('consultations.show', $validated['consultation_id'])
                ->with('success', count($prescriptions) . ' medications prescribed successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create prescription: ' . $e->getMessage());
        }
    }

    // Display the specified prescription
    public function show(Prescription $prescription)
    {
        return Inertia::render('Prescriptions/Show', [
            'prescription' => $prescription->load(['patient', 'medication', 'prescriber', 'consultation']),
        ]);
    }

    // Show the form for editing the prescription
    public function edit(Prescription $prescription)
    {
        return Inertia::render('Prescriptions/Edit', [
            'prescription'  => $prescription,
            'medications'   => Medication::all(),
            'routes'        => ['Oral', 'IV', 'IM', 'Subcutaneous', 'Topical'],
            'foodRelations' => ['Before Food', 'After Food', 'With Food', 'Any Time'],
        ]);
    }

    // Update the specified prescription
    public function update(Request $request, Prescription $prescription)
    {
        $validated = $request->validate([
            'medication_id' => 'required|exists:medications,id',
            'dosage'        => 'required|string|max:50',
            'frequency'     => 'required|string|max:50',
            'duration'      => 'required|string|max:50',
            'quantity'      => 'required|integer|min:1',
            'route'         => 'required|string|max:50',
            'food_relation' => 'required|string|max:50',
            'notes'         => 'nullable|string',
        ]);

        $prescription->update($validated);

        return redirect()->route('consultations.show', $prescription->consultation_id)
            ->with('success', 'Prescription updated successfully.');
    }

    // Remove the specified prescription
    public function destroy(Prescription $prescription)
    {
        $consultation_id = $prescription->consultation_id;
        $prescription->delete();

        return redirect()->back()
            ->with('success', 'Prescription deleted successfully.');
    }

    // Get prescriptions for a specific patient
    public function patientPrescriptions(Patient $patient)
    {
        $prescriptions = Prescription::where('patient_id', $patient->id)
            ->with(['medication', 'prescriber'])
            ->latest()
            ->get();

        return Inertia::render('Prescriptions/PatientIndex', [
            'prescriptions' => $prescriptions,
            'patient'       => $patient,
        ]);
    }
}
