<?php
namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Consultation;
use App\Models\RadiologyOrder;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class RadiologyOrderController extends Controller
{
    public function create($consultationId)
    {
        $consultation = Consultation::with(['patient:id,first_name,surname', 'doctor:id,name'])->findOrFail($consultationId);

        return Inertia::render('RadiologyOrders/Create', [
            'consultationId' => $consultation->id,
            'patient'        => $consultation->patient,
            'consultation'   => $consultation->only([
                'id',
                'consultation_date',
                'presenting_complaint',
                'clinical_notes',
                'diagnosis',
                'treatment_plan',
                'doctor',
            ]),
            'radiologyTypes' => $this->getAvailableRadiologyTypes(),
            'radiologists'   => User::where('role', UserRole::Radiologist)
                ->get(['id', 'name']),
        ]);
    }

    public function store(Request $request, $consultationId)
    {
        $consultation = Consultation::findOrFail($consultationId);

        $validated = $request->validate([
            'radiology_type' => 'required|string|max:255',
            'status'         => 'required|in:requested,in_progress,completed',
            'report_notes'   => 'nullable|string',
            'result_file'    => 'nullable|string',
            'radiologist_id' => 'nullable|exists:users,id',
        ]);

        RadiologyOrder::create([
            'patient_id'      => $consultation->patient_id,
            'consultation_id' => $consultation->id,
            'radiology_type'  => $validated['radiology_type'],
            'status'          => $validated['status'],
            'report_notes'    => $validated['report_notes'] ?? null,
            'result_file'     => $validated['result_file'] ?? null,
            'ordered_by'      => Auth::id(),
            'radiologist_id'  => $validated['radiologist_id'] ?? null,
        ]);

        return redirect()->route('consultations.show', $consultationId)->with('success', 'Radiology order created.');
    }

    public function edit(RadiologyOrder $radiologyOrder)
    {
        $radiologyOrder->load(['patient', 'consultation']);

        return Inertia::render('RadiologyOrders/Edit', [
            'order'          => $radiologyOrder,
            'radiologyTypes' => $this->getAvailableRadiologyTypes(),
            'radiologists'   => User::where('role', 'radiologist')->get(['id', 'first_name', 'surname']),
        ]);
    }

    public function update(Request $request, RadiologyOrder $radiologyOrder)
    {
        $validated = $request->validate([
            'radiology_type' => 'required|string|max:255',
            'status'         => 'required|in:requested,in_progress,completed',
            'report_notes'   => 'nullable|string',
            'result_file'    => 'nullable|string',
            'radiologist_id' => 'nullable|exists:users,id',
        ]);

        $radiologyOrder->update($validated);

        return redirect()->route('consultations.show', $radiologyOrder->consultation_id)->with('success', 'Radiology order updated.');
    }

    private function getAvailableRadiologyTypes()
    {
        return [
            'Chest X-Ray',
            'CT Brain',
            'MRI Spine',
            'Ultrasound Abdomen',
            'X-Ray Limb',
            'Mammogram',
            'CT Abdomen',
        ];
    }
}
