<?php

namespace App\Http\Controllers;

use App\Models\Consultation;
use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class ReferralsReportController extends Controller
{
    /** Canonical referral flag -> printable label (prevents column name typos). */
    private const REFERRAL_FLAGS = [
        'refer_for_radiology' => 'Radiology',
        'refer_for_lab'       => 'Lab',
        'refer_to_specialist' => 'Specialist',
        'refer_to_casualty'   => 'Casualty',   // ✅ correct column
        'refer_out_patient'   => 'Out-Patient',
    ];

    /**
     * Monthly Referrals report (clinics limited to those the authed user can access).
     * - Required: year, month
     * - Optional: clinic_id, search
     * - Shows Diagnosis/Referral Notes only for doctor/nurse/superadmin.
     */
    public function monthly(Request $request)
    {
        $user = Auth::user();
        abort_if(!$user, 401);

        $year  = (int) $request->query('year');
        $month = (int) $request->query('month');
        abort_if(!$year || !$month, 422, 'Year and month are required.');

        $clinicId = $request->query('clinic_id');
        $search   = $request->query('search');

        // Which clinics can this user see?
        $clinicIds = [];
        if (method_exists($user, 'accessibleClinicIds')) {
            $clinicIds = (array) $user->accessibleClinicIds();
        } else {
            try {
                $ids = $user->accessibleClinics()->pluck('clinics.id')->all();
            } catch (\Throwable $e) {
                $ids = [];
            }
            if (!empty($user->clinic_id) && !in_array((int) $user->clinic_id, $ids, true)) {
                $ids[] = (int) $user->clinic_id;
            }
            $clinicIds = array_values(array_unique(array_map('intval', $ids)));
        }

        // If a clinic_id is provided, intersect with accessible clinics.
        if (!empty($clinicId)) {
            $clinicIds = array_values(array_intersect($clinicIds, [(int) $clinicId]));
        }

        // Always enforce clinic scope; if none remain, force-empty result set.
        $clinicIdsFilter = !empty($clinicIds) ? $clinicIds : [-1];

        // Who can see diagnosis/notes?
        $role              = $user->role?->value ?? (string) $user->role;
        $includeSensitive  = in_array($role, ['doctor','nurse','superadmin'], true);

        $start        = Carbon::create($year, $month, 1)->startOfMonth();
        $end          = (clone $start)->endOfMonth();
        $periodLabel  = $start->isoFormat('MMMM YYYY');
        $generatedAt  = now()->format('Y-m-d H:i:s');

        $rows = Consultation::query()
            ->with([
                'patient:id,first_name,surname,gender,employee_number,parent_patient_id,relationship,company_id',
                'patient.company:id,name',
                'clinic:id,name,city',
            ])
            // Month filter: prefer consultation_date, fallback to created_at
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('consultation_date', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->whereNull('consultation_date')
                         ->whereBetween('created_at', [$start, $end]);
                  });
            })
            // Any referral flags (centralized, no typos)
            ->where(function ($q) {
                $first = true;
                foreach (self::REFERRAL_FLAGS as $col => $label) {
                    if ($first) { $q->where($col, 1); $first = false; }
                    else        { $q->orWhere($col, 1); }
                }
            })
            // Enforce clinic scope
            ->whereIn('clinic_id', $clinicIdsFilter)
            // Search (patient/company/diagnosis/notes/specialist)
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->orWhereHas('patient', function ($p) use ($search) {
                            $p->where('first_name', 'like', "%{$search}%")
                              ->orWhere('surname', 'like', "%{$search}%")
                              ->orWhere('employee_number', 'like', "%{$search}%");
                        })
                        ->orWhereHas('patient.company', fn($c) =>
                            $c->where('name', 'like', "%{$search}%")
                        )
                        ->orWhere('diagnosis', 'like', "%{$search}%")
                        ->orWhere('referral_notes', 'like', "%{$search}%")
                        ->orWhere('specialist_name', 'like', "%{$search}%");
                });
            })
            ->orderBy('consultation_date')
            ->orderBy('created_at')
            ->get();

        // Preload account holders for dependents
        $parentIds   = $rows->pluck('patient.parent_patient_id')->filter()->unique()->values();
        $parentsById = $parentIds->isNotEmpty()
            ? Patient::whereIn('id', $parentIds)->get(['id','first_name','surname','employee_number'])->keyBy('id')
            : collect();

        // Group by clinic label
        $byClinic = $rows->groupBy(function ($c) {
            $clinic = $c->clinic;
            if (!$clinic) return 'Unknown Clinic';
            $name = (string) ($clinic->name ?? '');
            $city = (string) ($clinic->city ?? '');
            return trim($name . ($city ? " ({$city})" : ''));
        })->sortKeys();

        // Header filters line
        $filters = ["Period: {$periodLabel}"];
        if (!empty($clinicId)) $filters[] = "Clinic ID: {$clinicId}";
        if (!empty($search))   $filters[] = "Search: {$search}";
        $filtersLine = implode(' | ', $filters);

        $fileName = sprintf('Referrals_%d_%02d.xlsx', $year, $month);

        return response()->streamDownload(function () use ($rows, $byClinic, $parentsById, $generatedAt, $filtersLine, $includeSensitive) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            // ===== All Referrals =====
            $writer->getCurrentSheet()->setName($this->safeSheetName('All Referrals'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['REFERRALS — ALL CLINICS'],
                ['Generated At', $generatedAt],
                ['Filters', $filtersLine],
            ]);
            $this->writeReferralsTable($writer, $rows, $headerStyle, $wrapStyle, $parentsById, true, $includeSensitive);
            $this->writeReferralsTotals($writer, $rows);

            // ===== Per-Clinic Sheets =====
            $used = ['All Referrals' => true];
            foreach ($byClinic as $clinicLabel => $dataset) {
                $writer->addNewSheetAndMakeItCurrent();
                $sheetName = $this->uniqueSheetName($used, $this->safeSheetName($clinicLabel));
                $writer->getCurrentSheet()->setName($sheetName);

                $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                    ["REFERRALS — {$clinicLabel}"],
                    ['Generated At', $generatedAt],
                ]);
                // Omit Clinic column (implied by sheet)
                $this->writeReferralsTable($writer, $dataset, $headerStyle, $wrapStyle, $parentsById, false, $includeSensitive);
                $this->writeReferralsTotals($writer, $dataset);
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /** Table writer. If $includeClinic true, append the Clinic column. */
    private function writeReferralsTable(
        XLSXWriter $writer,
        $dataset,
        Style $headerStyle,
        Style $wrapStyle,
        $parentsById,
        bool $includeClinic,
        bool $includeSensitive
    ): void {
        // Headers (Diagnosis/Notes only for doctor/nurse/superadmin)
        $headers = [
            'Date',
            'Patient FirstName',
            'Patient Surname',
            'Gender',
            'Category',           // Employee | Dependent
            'Account Holder',     // for dependents
            'Holder Employee #',  // for dependents
            'Relationship',       // for dependents
            'Referral Type(s)',   // Radiology • Lab • Specialist • Casualty • Out-Patient
            'Specialist Name',
        ];
        if ($includeSensitive) {
            $headers[] = 'Diagnosis';
            $headers[] = 'Referral Notes';
        }
        if ($includeClinic) {
            $headers[] = 'Clinic';
        }

        $writer->addRow(Row::fromValues($headers, $headerStyle));

        // Column widths
        $opts = $writer->getOptions();
        $setW = function (string $label, int $width) use ($headers, $opts) {
            $idx = array_search($label, $headers, true);
            if ($idx !== false) $opts->setColumnWidth($width, $idx + 1);
        };
        $setW('Date', 14);
        $setW('Patient FirstName', 24);
        $setW('Patient Surname', 24);
        $setW('Gender', 10);
        $setW('Category', 14);
        $setW('Account Holder', 28);
        $setW('Holder Employee #', 18);
        $setW('Relationship', 16);
        $setW('Referral Type(s)', 26);
        $setW('Specialist Name', 24);
        if ($includeSensitive) { $setW('Diagnosis', 40); $setW('Referral Notes', 40); }
        if ($includeClinic)    { $setW('Clinic', 24); }

        foreach ($dataset as $c) {
            $patient     = $c->patient;
            $isDependent = !empty(optional($patient)->parent_patient_id);

            $holderName = '';
            $holderEmp  = '';
            $relationship = (string) ($patient->relationship ?? '');

            if ($isDependent) {
                $parentId = $patient->parent_patient_id;
                $holder   = $parentId ? $parentsById->get($parentId) : null;
                $holderName = $holder ? trim(($holder->first_name ?? '') . ' ' . ($holder->surname ?? '')) : '';
                $holderEmp  = $holder->employee_number ?? '';
            }

            // Referral types string (from constants)
            $types = collect(self::REFERRAL_FLAGS)
                ->map(fn($label, $col) => $c->{$col} ? $label : null)
                ->filter()
                ->implode(' • ');

            $clinic  = $this->resolveClinicLabelFromConsultation($c);
            $dateCell = optional($c->consultation_date)->format('Y-m-d')
                ?? (optional($c->created_at)->format('Y-m-d') ?? '');

            $row = [
                $dateCell,
                (string) ($patient->first_name ?? ''),
                (string) ($patient->surname ?? ''),
                (string) ($patient->gender ?? ''),
                $isDependent ? 'Dependent' : 'Employee',
                $holderName,
                $holderEmp,
                $relationship,
                $types,
                (string) ($c->specialist_name ?? ''),
            ];

            if ($includeSensitive) {
                $row[] = (string) ($c->diagnosis ?? '');
                $row[] = (string) ($c->referral_notes ?? '');
            }
            if ($includeClinic) {
                $row[] = $clinic;
            }

            $writer->addRow(Row::fromValues($row, $wrapStyle));
        }
    }

    /** Totals footer: overall + by referral type + by category. */
    private function writeReferralsTotals(XLSXWriter $writer, $dataset): void
    {
        $writer->addRow(Row::fromValues([]));

        $employees  = $dataset->filter(fn($c) => empty(optional($c->patient)->parent_patient_id))->count();
        $dependents = $dataset->filter(fn($c) => !empty(optional($c->patient)->parent_patient_id))->count();

        $writer->addRow(Row::fromValues(['Total Referrals', $dataset->count()]));
        $writer->addRow(Row::fromValues(['Employees', $employees]));
        $writer->addRow(Row::fromValues(['Dependents', $dependents]));

        // Counts by referral type (driven by constants)
        $writer->addRow(Row::fromValues([]));
        $writer->addRow(Row::fromValues(['Breakdown by Referral Type'], (new Style())->setFontBold()));
        foreach (self::REFERRAL_FLAGS as $col => $label) {
            $writer->addRow(Row::fromValues([$label, (int) $dataset->where($col, 1)->count()]));
        }
    }

    /** "Clinic Name (City)" from Consultation relation. */
    private function resolveClinicLabelFromConsultation($consultation): string
    {
        $c = $consultation->clinic;
        if (!$c) return '';
        $name = (string) ($c->name ?? '');
        $city = (string) ($c->city ?? '');
        return trim($name . ($city ? " ({$city})" : ''));
    }

    /* ---------------- helpers ---------------- */

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        // Excel forbids: : \ / ? * [ ]
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Ensure unique sheet name within a workbook. */
    private function uniqueSheetName(array &$used, string $base): string
    {
        $name = $base;
        $i = 2;
        while (isset($used[$name])) {
            $suffix = " ({$i})";
            $name   = mb_substr($base, 0, 31 - mb_strlen($suffix)) . $suffix;
            $i++;
        }
        $used[$name] = true;
        return $name;
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }
}
