<?php
namespace App\Http\Controllers;

use App\Models\Clinic;
use App\Models\Consultation;
use App\Models\Patient;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ReportsController extends Controller
{

    public function monthlyConsultations(Request $request)
    {
        $clinicId = $request->query('clinic_id');
        $year     = $request->query('year');

        // Qualify columns to avoid ambiguity
        $dateExpr = 'COALESCE(consultations.consultation_date, consultations.created_at)';

        $query = DB::table('consultations')
            ->join('patients', 'patients.id', '=', 'consultations.patient_id')
            ->selectRaw('DATE_FORMAT(' . $dateExpr . ', "%Y-%m") as month')
            ->selectRaw('SUM(CASE WHEN patients.parent_patient_id IS NULL THEN 1 ELSE 0 END) AS employees')
            ->selectRaw('SUM(CASE WHEN patients.parent_patient_id IS NOT NULL THEN 1 ELSE 0 END) AS dependents')
            ->selectRaw('COUNT(*) AS total')
            ->when($year, fn($q) => $q->whereYear(DB::raw($dateExpr), $year))
            ->when($clinicId, fn($q) => $q->where('consultations.clinic_id', $clinicId))
        // Group by the raw expression to be friendly with ONLY_FULL_GROUP_BY
            ->groupByRaw('DATE_FORMAT(' . $dateExpr . ', "%Y-%m")')
            ->orderBy('month', 'asc');

        $data = $query->get();

        return response()->json($data);
    }

    public function getClinics()
    {
        $clinics = Clinic::all();
        return response()->json($clinics);
    }

    public function getYears()
    {
        $years = DB::table('consultations')
            ->selectRaw('YEAR(consultation_date) as year')
            ->groupBy('year')
            ->orderBy('year', 'desc')
            ->get();

        return response()->json($years);
    }

    public function genderDistribution()
    {
        $genderDistribution = Patient::getGenderDistribution();
        return response()->json($genderDistribution);
    }

    public function employeeVsDependents()
    {
        $data = Patient::getEmployeeVsDependentsCount();
        return response()->json($data);
    }

    public function referralsPerMonth(Request $request)
    {
        $year     = $request->input('year', date('Y'));
        $clinicId = $request->input('clinic_id');

        $data = Consultation::getReferralsPerMonth($year, $clinicId);

        return response()->json($data);
    }

    public function ageRangeDistribution()
    {
        $data = Patient::getPatientsByAgeRange();

        return response()->json($data);
    }

    public function sickLeavePerMonth(Request $request)
    {
        // Keep your year param (used only when a clinic is selected)
        $year     = $request->input('year', date('Y'));
        $clinicId = $request->input('clinic_id');

        // If NO clinic selected -> return ALL clinics, aggregated across ALL years
        if (empty($clinicId)) {
            // Build counts per month using consultation_date, falling back to created_at
            $rows = \App\Models\Consultation::query()
                ->where(function ($q) {
                    // Treat any truthy/positive value as "has sick leave"
                    // adjust to ->where('sick_leave', true) if your column is boolean
                    $q->whereNotNull('sick_leave')
                        ->where('sick_leave', '!=', 0);
                })
                ->selectRaw("
                MONTH(COALESCE(consultation_date, created_at)) AS m,
                COUNT(*) AS c
            ")
                ->groupBy('m')
                ->orderBy('m')
                ->get()
                ->keyBy('m');

            // Normalize to 12 months (1..12) with zeros if missing
            $result = collect(range(1, 12))->map(function ($m) use ($rows) {
                return [
                    // You can return numeric month index or any label you prefer.
                    // The frontend parser already accepts numeric 1..12.
                    'month' => $m,
                    'count' => (int) ($rows[$m]->c ?? 0),
                ];
            });

            return response()->json($result->all());
        }

        // Clinic selected -> keep the existing behavior (specific clinic + year)
        $data = \App\Models\Consultation::getSickLeaveIssuedPerMonth($year, $clinicId);

        return response()->json($data);
    }

    public function injuryOnDutyCases(Request $request)
    {
        // Default year to current if not provided or invalid
        $year = (int) ($request->input('year') ?: date('Y'));

        // Treat missing/blank/"all" as null (=> no filter)
        $rawCompany = $request->input('company_id');
        $rawClinic  = $request->input('clinic_id');

        $companyId = ($request->filled('company_id') && $rawCompany !== '' && $rawCompany !== 'all')
        ? (int) $rawCompany
        : null;

        $clinicId = ($request->filled('clinic_id') && $rawClinic !== '' && $rawClinic !== 'all')
        ? (int) $rawClinic
        : null;

        // If both are null => return for ALL companies & clinics (model should aggregate across all)
        // If one is set => model will filter by that dimension only
        $data = \App\Models\Consultation::getInjuryOnDutyCasesPerCompany($year, $companyId, $clinicId);

        return response()->json($data);
    }

    public function index(Request $request)
    {
        $search      = $request->input('search');
        $startDate   = $request->input('start_date');
        $endDate     = $request->input('end_date');
        $clinicId    = $request->input('clinic_id');
        $doctorId    = $request->input('doctor_id');
        $year        = $request->input('year', date('Y'));
        $month       = $request->input('month');
        $period      = $request->input('period');       // daily, weekly, monthly, quarterly
        $patientType = $request->input('patient_type'); // all, dependents, non-dependents

        $consultations = Consultation::with([
            'clinic:id,name',
            'doctor:id,name',
            'patient',
        ])
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('diagnosis', 'like', "%{$search}%")
                        ->orWhere('treatment_plan', 'like', "%{$search}%")
                        ->orWhereHas('clinic', fn($q) => $q->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('doctor', fn($q) => $q->where('name', 'like', "%{$search}%"))
                        ->orWhereHas('patient', fn($q) => $q->where('first_name', 'like', "%{$search}%")
                                ->orWhere('surname', 'like', "%{$search}%"));
                });
            })
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                $query->whereBetween('consultation_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($year && ! $startDate && ! $endDate, function ($query) use ($year) {
                $query->whereYear('consultation_date', $year);
            })
            ->when($month, function ($query) use ($year, $month) {
                $query->whereYear('consultation_date', $year)
                    ->whereMonth('consultation_date', $month);
            })
            ->when($period, function ($query) use ($period, $year, $month) {
                switch ($period) {
                    case 'quarterly':
                        if ($month) {
                            $quarter = ceil($month / 3);
                            $query->whereBetween('consultation_date', [
                                Carbon::create($year, ($quarter * 3) - 2, 1)->startOfDay(),
                                Carbon::create($year, $quarter * 3, 1)->endOfMonth()->endOfDay(),
                            ]);
                        } else {
                            // Default to current quarter if no month selected
                            $currentQuarter = ceil(date('n') / 3);
                            $query->whereBetween('consultation_date', [
                                Carbon::create($year, ($currentQuarter * 3) - 2, 1)->startOfDay(),
                                Carbon::create($year, $currentQuarter * 3, 1)->endOfMonth()->endOfDay(),
                            ]);
                        }
                        break;

                    case 'weekly':
                        $query->whereBetween('consultation_date', [
                            Carbon::now()->startOfWeek(),
                            Carbon::now()->endOfWeek(),
                        ]);
                        break;

                    case 'daily':
                        $query->whereDate('consultation_date', Carbon::today());
                        break;
                }
            })
            ->when($clinicId, function ($query) use ($clinicId) {
                $query->where('clinic_id', $clinicId);
            })
            ->when($doctorId, function ($query) use ($doctorId) {
                $query->where('doctor_id', $doctorId);
            })
            ->when($patientType, function ($query) use ($patientType) {
                if ($patientType === 'dependents') {
                    $query->whereHas('patient', function ($q) {
                        $q->whereNotNull('parent_patient_id');
                    });
                } elseif ($patientType === 'non-dependents') {
                    $query->whereHas('patient', function ($q) {
                        $q->whereNull('parent_patient_id');
                    });
                }
            })
            ->latest('consultation_date')
            ->paginate(10)
            ->appends($request->all());

        // Get clinics and doctors for dropdowns
        $clinics = Clinic::select('id', 'name')->orderBy('name')->get();
        $doctors = User::where('role', 'doctor')
            ->select('id', 'name')
            ->orderBy('name')
            ->get();

        // Generate years for dropdown (last 10 years + next 2 years)
        $currentYear = (int) date('Y');
        $years       = range($currentYear, $currentYear - 5);

        return Inertia::render('Reports/Index', [
            'consultations' => $consultations,
            'clinics'       => $clinics,
            'doctors'       => $doctors,
            'years'         => $years,
            'patientTypes'  => [
                ['value' => 'all', 'label' => 'All Patients'],
                ['value' => 'dependents', 'label' => 'Dependents Only'],
                ['value' => 'non-dependents', 'label' => 'Non-Dependents Only'],
            ],
            'periods'       => [
                ['value' => 'daily', 'label' => 'Daily'],
                ['value' => 'weekly', 'label' => 'Weekly'],
                ['value' => 'monthly', 'label' => 'Monthly'],
                ['value' => 'quarterly', 'label' => 'Quarterly'],
            ],
            'filters'       => $request->only([
                'search',
                'start_date',
                'end_date',
                'clinic_id',
                'doctor_id',
                'year',
                'month',
                'period',
                'patient_type',
            ]),
        ]);
    }
}
