<?php
namespace App\Http\Controllers;

use App\Models\CentralStore;
use App\Models\Clinic;
use App\Models\ClinicMedicationStock;
use App\Models\ClinicStockReceipt;
use App\Models\Log as AuditLog;
use App\Models\StockTransfer;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class StockTransferController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, CentralStore $centralStore)
    {
        $clinicId      = $request->input('clinic_id');
        $batchNumber   = $request->input('batch_number');
        $startDate     = $request->input('start_date');
        $endDate       = $request->input('end_date');
        $transferredBy = $request->input('transferred_by');

        $stockTransfers = StockTransfer::with([
            'fromCentralStore:id,name',
            'toClinic:id,name',
            'medicationBatch:id,id,batch_number,medication_id',
            'medicationBatch.medication:id,name,dosage,form,unit,instructions',
            'transferredBy:id,name',
        ])
            ->where('from_central_store_id', $centralStore->id)
            ->when($clinicId, fn($query) =>
                $query->where('to_clinic_id', $clinicId)
            )
            ->when($batchNumber, function ($query) use ($batchNumber) {
                $query->whereHas('medicationBatch', fn($q) =>
                    $q->where('batch_number', 'like', "%{$batchNumber}%")
                );
            })
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                $query->whereBetween('transfer_date', [
                    Carbon::parse($startDate)->startOfDay(),
                    Carbon::parse($endDate)->endOfDay(),
                ]);
            })
            ->when($transferredBy, fn($query) =>
                $query->where('transferred_by', $transferredBy)
            )
            ->latest() // ✅ Sort by created_at
            ->paginate(10)
            ->appends($request->all());

        $clinics = Clinic::select('id', 'name')->orderBy('name')->get();
        $users   = User::select('id', 'name')->orderBy('name')->get();

        return Inertia::render('CentralStores/TransferHistory', [
            'stockTransfers' => $stockTransfers,
            'clinics'        => $clinics,
            'users'          => $users,
            'centralStore'   => $centralStore,
            'filters'        => $request->only([
                'clinic_id',
                'batch_number',
                'start_date',
                'end_date',
                'transferred_by',
            ]),
        ]);
    }

    public function approveStockTransfer(Request $request, Clinic $clinic, $stockTransferId)
    {
        try {
            $validated = $request->validate([
                'notes'    => 'nullable|string|max:255',
                'quantity' => 'required|integer|min:1',
            ]);

            DB::beginTransaction();

            // Find stock transfer and make sure it belongs to this clinic
            $stockTransfer = StockTransfer::where('id', $stockTransferId)
                ->where('to_clinic_id', $clinic->id)
                ->firstOrFail();

            $oldQuantity = $stockTransfer->quantity;
            $newQuantity = $validated['quantity'];

            // Update transfer with status + quantity
            $stockTransfer->update([
                'status'      => 'Approved',
                'approved_by' => Auth::id(),
                'approved_at' => now(),
                'quantity'    => $newQuantity,
            ]);

            // Update or create receipt
            $clinicStockReceipt = ClinicStockReceipt::where('stock_transfer_id', $stockTransfer->id)->firstOrFail();
            $clinicStockReceipt->update([
                'approved_by' => Auth::id(),
                'approved_at' => now(),
                'notes'       => $validated['notes'] ?? 'Approved',
            ]);

            // Add stock to clinic
            ClinicMedicationStock::updateOrCreate(
                [
                    'clinic_id'           => $stockTransfer->to_clinic_id,
                    'medication_batch_id' => $stockTransfer->medication_batch_id,
                ],
                [
                    'quantity' => DB::raw("quantity + {$newQuantity}"),
                ]
            );

            // Prepare log description
            $description = "Approved stock transfer of {$newQuantity} units to clinic {$clinic->name}";
            if ($oldQuantity !== $newQuantity) {
                $description .= " (original requested: {$oldQuantity}, approved: {$newQuantity})";
            }

            // Create log
            AuditLog::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()?->name,
                'action'        => 'approve',
                'loggable_type' => StockTransfer::class,
                'loggable_id'   => $stockTransfer->id,
                'description'   => $description,
                'old_values'    => array_merge($stockTransfer->getOriginal(), ['quantity' => $oldQuantity]),
                'new_values'    => $stockTransfer->getChanges(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->userAgent(),
            ]);

            DB::commit();

            return redirect()
                ->back()
                ->with('success', 'Stock transfer approved successfully.');

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', 'Stock transfer not found or does not belong to this clinic.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Stock transfer approval failed: ' . $e->getMessage());

            return redirect()
                ->back()
                ->with('error', 'An unexpected error occurred: ' . $e->getMessage());
        }
    }

}
