<?php

namespace App\Http\Controllers;

use App\Models\Patient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class SystemCorrectionController extends Controller
{
    /**
     * List employees (parent_patient_id = null) whose gender is NULL or ''.
     * Supports simple search & pagination.
     */
    public function employeesWithoutGender(Request $request)
    {
        // $this->authorize('viewAny', Patient::class); // optional if you have policies

        $search   = $request->input('search');
        $perPage  = (int) ($request->input('per_page') ?: 50);

        $query = Patient::query()
            ->accessibleByUser(Auth::user()) // uses your model scope
            ->whereNull('parent_patient_id') // employees only
            ->where(function ($q) {
                $q->whereNull('gender')
                  ->orWhere('gender', '')
                  ->orWhereRaw("LOWER(TRIM(gender)) IN ('unknown','n/a','na','unspecified')");
            })
            ->with(['company:id,name'])
            ->orderBy('surname')
            ->orderBy('first_name');

        if ($search) {
            $like = '%' . trim($search) . '%';
            $query->where(function ($q) use ($like) {
                $q->where('first_name', 'like', $like)
                  ->orWhere('middle_name', 'like', $like)
                  ->orWhere('surname', 'like', $like)
                  ->orWhere('employee_number', 'like', $like)
                  ->orWhere('id_number', 'like', $like);
            });
        }

        $employees = $query->paginate($perPage)->withQueryString();

        return Inertia::render('SystemCorrections/EmployeesWithoutGender', [
            'employees' => $employees->through(function (Patient $p) {
                return [
                    'id'               => $p->id,
                    'full_name'        => $p->full_name,
                    'employee_number'  => $p->employee_number,
                    'company'          => $p->company?->name,
                    'gender'           => $p->gender,
                    'id_number'        => $p->id_number,
                ];
            }),
            'filters'  => [
                'search'  => $search,
                'per_page'=> $perPage,
            ],
        ]);
    }

    /**
     * Bulk set gender for selected employees.
     * Accepts payload like:
     * { male_ids: [1,2], female_ids: [3,4] }
     */
    public function bulkAssignGender(Request $request)
    {
        $data = $request->validate([
            'male_ids'   => ['array'],
            'male_ids.*' => ['integer', 'distinct'],
            'female_ids'   => ['array'],
            'female_ids.*' => ['integer', 'distinct'],
        ]);

        $maleIds   = array_map('intval', $data['male_ids']   ?? []);
        $femaleIds = array_map('intval', $data['female_ids'] ?? []);

        // Prevent overlap
        $overlap = array_intersect($maleIds, $femaleIds);
        if (!empty($overlap)) {
            return back()->withErrors(['gender' => 'Some patients are selected as both Male and Female. Please fix selections.']);
        }

        // Limit updates strictly to employees currently without valid gender, and user-accessible
        $baseScope = Patient::query()
            ->accessibleByUser(Auth::user())
            ->whereNull('parent_patient_id')
            ->where(function ($q) {
                $q->whereNull('gender')
                  ->orWhere('gender', '')
                  ->orWhereRaw("LOWER(TRIM(gender)) IN ('unknown','n/a','na','unspecified')");
            });

        DB::transaction(function () use ($maleIds, $femaleIds, $baseScope) {
            if (!empty($maleIds)) {
                (clone $baseScope)->whereIn('id', $maleIds)->update(['gender' => 'Male']);
            }
            if (!empty($femaleIds)) {
                (clone $baseScope)->whereIn('id', $femaleIds)->update(['gender' => 'Female']);
            }
        });

        $updatedCount = count($maleIds) + count($femaleIds);

        return back()->with('success', "Updated gender for {$updatedCount} employee(s).");
    }
}
