<?php

namespace App\Http\Controllers;

use App\Models\Patient;
use Carbon\Carbon;
use Illuminate\Http\Request;

// OpenSpout v4
use OpenSpout\Common\Entity\Row;
use OpenSpout\Common\Entity\Style\Style;
use OpenSpout\Writer\XLSX\Writer as XLSXWriter;

class SystemDataController extends Controller
{
    /**
     * Full system export:
     *  - Sheet 1: Account Holders (no parent_patient_id) + Company
     *  - Sheet 2: Dependents + Holder full name, employee #, company
     *  - Sheet 3: Grouped (each holder row, then their dependents; blank line between holders)
     */
    public function all(Request $request)
    {
        // Load everyone with company + parent(+company)
        $all = Patient::query()
            ->with([
                'company:id,name',
                'parent:id,first_name,middle_name,surname,employee_number,company_id',
                'parent.company:id,name',
            ])
            ->orderBy('surname')
            ->orderBy('first_name')
            ->get();

        $holders    = $all->whereNull('parent_patient_id')->values();
        $dependents = $all->whereNotNull('parent_patient_id')->values();

        // Convenient maps
        $parentsById = $holders->keyBy('id');

        $generatedAt = now()->format('Y-m-d H:i:s');
        $fileName    = 'System_Patients_Export.xlsx';

        return response()->streamDownload(function () use ($holders, $dependents, $parentsById, $generatedAt) {
            $writer = new XLSXWriter();
            $writer->openToFile('php://output');

            // Basic styles
            $titleStyle  = (new Style())->setFontBold();
            $labelStyle  = (new Style())->setFontBold();
            $headerStyle = (new Style())->setFontBold();
            $wrapStyle   = (new Style())->setShouldWrapText(true);

            /* ---------------- Sheet 1: Account Holders ---------------- */
            $writer->getCurrentSheet()->setName($this->safeSheetName('Account Holders'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['SYSTEM EXPORT — ACCOUNT HOLDERS'],
                ['Generated At', $generatedAt],
            ]);

            $headers1 = [
                'First Name','Middle Name','Surname','Full Name','Gender','Date of Birth','Age',
                'Employee Number','Company','ID Number','Email','Phone',
                'Home Address','Work Area','Suburb',
                'Occupation','Marital Status',
                'Medical Aid Provider','Medical Aid Number',
                'Allergies','Is Smoker',
                'Emergency Contact Name','Emergency Contact Relation','Emergency Contact Phone',
                'Created At','Updated At',
            ];
            $this->setColumnWidthsByHeader($writer, $headers1, [
                'First Name'=>18,'Middle Name'=>18,'Surname'=>18,'Full Name'=>26,
                'Gender'=>10,'Date of Birth'=>14,'Age'=>6,
                'Employee Number'=>16,'Company'=>26,'ID Number'=>18,'Email'=>26,'Phone'=>16,
                'Home Address'=>26,'Work Area'=>18,'Suburb'=>18,
                'Occupation'=>18,'Marital Status'=>14,
                'Medical Aid Provider'=>20,'Medical Aid Number'=>18,
                'Allergies'=>20,'Is Smoker'=>10,
                'Emergency Contact Name'=>22,'Emergency Contact Relation'=>22,'Emergency Contact Phone'=>18,
                'Created At'=>18,'Updated At'=>18,
            ]);
            $writer->addRow(Row::fromValues($headers1, $headerStyle));

            foreach ($holders as $p) {
                $writer->addRow(Row::fromValues([
                    (string) ($p->first_name ?? ''),
                    (string) ($p->middle_name ?? ''),
                    (string) ($p->surname ?? ''),
                    (string) ($p->full_name ?? trim(($p->first_name ?? '') . ' ' . ($p->middle_name ?? '') . ' ' . ($p->surname ?? ''))),
                    (string) ($p->gender ?? ''),
                    $this->fmtDate($p->date_of_birth),
                    $this->age($p->date_of_birth),
                    (string) ($p->employee_number ?? ''),
                    (string) optional($p->company)->name ?? '',
                    (string) ($p->id_number ?? ''),
                    (string) ($p->email ?? ''),
                    (string) ($p->phone ?? ''),
                    (string) ($p->home_address ?? ''),
                    (string) ($p->work_area ?? ''),
                    (string) ($p->suburb ?? ''),
                    (string) ($p->occupation ?? ''),
                    (string) ($p->marital_status ?? ''),
                    (string) ($p->medical_aid_provider ?? ''),
                    (string) ($p->medical_aid_number ?? ''),
                    (string) ($p->allergies ?? ''),
                    $this->yn($p->is_smoker),
                    (string) ($p->emergency_contact_name ?? ''),
                    (string) ($p->emergency_contact_relation ?? ''),
                    (string) ($p->emergency_contact_phone ?? ''),
                    $this->fmtDateTime($p->created_at),
                    $this->fmtDateTime($p->updated_at),
                ], $wrapStyle));
            }

            /* ---------------- Sheet 2: Dependents ---------------- */
            $writer->addNewSheetAndMakeItCurrent();
            $writer->getCurrentSheet()->setName($this->safeSheetName('Dependents'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['SYSTEM EXPORT — DEPENDENTS'],
                ['Generated At', $generatedAt],
            ]);

            $headers2 = [
                'First Name','Middle Name','Surname','Full Name','Gender','Date of Birth','Age',
                'Relationship',
                'Account Holder Full Name','Holder Employee #','Holder Company',
                'ID Number','Email','Phone',
                'Home Address','Work Area','Suburb',
                'Occupation','Marital Status',
                'Medical Aid Provider','Medical Aid Number',
                'Allergies','Is Smoker',
                'Emergency Contact Name','Emergency Contact Relation','Emergency Contact Phone',
                'Created At','Updated At',
            ];
            $this->setColumnWidthsByHeader($writer, $headers2, [
                'Account Holder Full Name'=>28,'Holder Employee #'=>18,'Holder Company'=>26,
                'Relationship'=>16,
            ]);
            $writer->addRow(Row::fromValues($headers2, $headerStyle));

            foreach ($dependents as $p) {
                $holder   = $parentsById->get((int) $p->parent_patient_id);
                $holderNm = $holder ? $this->fullName($holder) : '';
                $holderEmp= $holder->employee_number ?? '';
                $holderCo = optional($holder?->company)->name ?? '';

                $writer->addRow(Row::fromValues([
                    (string) ($p->first_name ?? ''),
                    (string) ($p->middle_name ?? ''),
                    (string) ($p->surname ?? ''),
                    (string) ($p->full_name ?? $this->fullName($p)),
                    (string) ($p->gender ?? ''),
                    $this->fmtDate($p->date_of_birth),
                    $this->age($p->date_of_birth),
                    (string) ($p->relationship ?? ''),
                    $holderNm,
                    (string) $holderEmp,
                    (string) $holderCo,
                    (string) ($p->id_number ?? ''),
                    (string) ($p->email ?? ''),
                    (string) ($p->phone ?? ''),
                    (string) ($p->home_address ?? ''),
                    (string) ($p->work_area ?? ''),
                    (string) ($p->suburb ?? ''),
                    (string) ($p->occupation ?? ''),
                    (string) ($p->marital_status ?? ''),
                    (string) ($p->medical_aid_provider ?? ''),
                    (string) ($p->medical_aid_number ?? ''),
                    (string) ($p->allergies ?? ''),
                    $this->yn($p->is_smoker),
                    (string) ($p->emergency_contact_name ?? ''),
                    (string) ($p->emergency_contact_relation ?? ''),
                    (string) ($p->emergency_contact_phone ?? ''),
                    $this->fmtDateTime($p->created_at),
                    $this->fmtDateTime($p->updated_at),
                ], $wrapStyle));
            }

            /* ---------------- Sheet 3: Grouped (Holder + Dependents) ---------------- */
            $writer->addNewSheetAndMakeItCurrent();
            $writer->getCurrentSheet()->setName($this->safeSheetName('Grouped'));
            $this->writeSheetHeader($writer, $titleStyle, $labelStyle, [
                ['SYSTEM EXPORT — GROUPED BY ACCOUNT HOLDER'],
                ['Generated At', $generatedAt],
            ]);

            // One consistent, rich schema so both holder & dependents can sit in the same table
            $headers3 = [
                'Category', // Employee | Dependent
                'First Name','Middle Name','Surname','Full Name','Gender','Date of Birth','Age',
                'Employee Number','Company',
                'Relationship',
                'Holder Full Name','Holder Employee #','Holder Company',
                'ID Number','Email','Phone',
                'Home Address','Work Area','Suburb',
                'Occupation','Marital Status',
                'Medical Aid Provider','Medical Aid Number',
                'Allergies','Is Smoker',
                'Emergency Contact Name','Emergency Contact Relation','Emergency Contact Phone',
                'Created At','Updated At',
            ];
            $this->setColumnWidthsByHeader($writer, $headers3, [
                'Category'=>12,'Full Name'=>26,'Company'=>26,
                'Holder Full Name'=>28,'Holder Employee #'=>18,'Holder Company'=>26,
            ]);
            $writer->addRow(Row::fromValues($headers3, $headerStyle));

            // Build dependents by parent_id for quick grouping
            $depsByParent = $dependents->groupBy('parent_patient_id');

            foreach ($holders as $h) {
                // Holder row
                $writer->addRow(Row::fromValues([
                    'Employee',
                    (string) ($h->first_name ?? ''), (string) ($h->middle_name ?? ''), (string) ($h->surname ?? ''),
                    (string) ($h->full_name ?? $this->fullName($h)),
                    (string) ($h->gender ?? ''),
                    $this->fmtDate($h->date_of_birth),
                    $this->age($h->date_of_birth),
                    (string) ($h->employee_number ?? ''),
                    (string) optional($h->company)->name ?? '',
                    '',           // relationship
                    '', '', '',   // holder details (not applicable for holder)
                    (string) ($h->id_number ?? ''),
                    (string) ($h->email ?? ''),
                    (string) ($h->phone ?? ''),
                    (string) ($h->home_address ?? ''),
                    (string) ($h->work_area ?? ''),
                    (string) ($h->suburb ?? ''),
                    (string) ($h->occupation ?? ''),
                    (string) ($h->marital_status ?? ''),
                    (string) ($h->medical_aid_provider ?? ''),
                    (string) ($h->medical_aid_number ?? ''),
                    (string) ($h->allergies ?? ''),
                    $this->yn($h->is_smoker),
                    (string) ($h->emergency_contact_name ?? ''),
                    (string) ($h->emergency_contact_relation ?? ''),
                    (string) ($h->emergency_contact_phone ?? ''),
                    $this->fmtDateTime($h->created_at),
                    $this->fmtDateTime($h->updated_at),
                ], $wrapStyle));

                // Dependents of this holder
                $deps = $depsByParent->get($h->id) ?? collect();
                foreach ($deps as $d) {
                    $writer->addRow(Row::fromValues([
                        'Dependent',
                        (string) ($d->first_name ?? ''), (string) ($d->middle_name ?? ''), (string) ($d->surname ?? ''),
                        (string) ($d->full_name ?? $this->fullName($d)),
                        (string) ($d->gender ?? ''),
                        $this->fmtDate($d->date_of_birth),
                        $this->age($d->date_of_birth),
                        '', // employee #
                        '', // company (dependents inherit via holder; kept below)
                        (string) ($d->relationship ?? ''),
                        $this->fullName($h),
                        (string) ($h->employee_number ?? ''),
                        (string) (optional($h->company)->name ?? ''),
                        (string) ($d->id_number ?? ''),
                        (string) ($d->email ?? ''),
                        (string) ($d->phone ?? ''),
                        (string) ($d->home_address ?? ''),
                        (string) ($d->work_area ?? ''),
                        (string) ($d->suburb ?? ''),
                        (string) ($d->occupation ?? ''),
                        (string) ($d->marital_status ?? ''),
                        (string) ($d->medical_aid_provider ?? ''),
                        (string) ($d->medical_aid_number ?? ''),
                        (string) ($d->allergies ?? ''),
                        $this->yn($d->is_smoker),
                        (string) ($d->emergency_contact_name ?? ''),
                        (string) ($d->emergency_contact_relation ?? ''),
                        (string) ($d->emergency_contact_phone ?? ''),
                        $this->fmtDateTime($d->created_at),
                        $this->fmtDateTime($d->updated_at),
                    ], $wrapStyle));
                }

                // Spacer line between holders
                $writer->addRow(Row::fromValues([]));
            }

            $writer->close();
        }, $fileName, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /* ---------------- helpers ---------------- */

    private function fullName($p): string
    {
        $parts = array_filter([
            $p->first_name ?? null,
            $p->middle_name ?? null,
            $p->surname ?? null,
        ], fn($v) => filled($v));
        return trim(implode(' ', $parts));
    }

    private function age($dob): ?int
    {
        if (empty($dob)) return null;
        try {
            $d = $dob instanceof \DateTimeInterface ? Carbon::instance($dob) : Carbon::parse($dob);
            return $d->age;
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function fmtDate($d): string
    {
        if (empty($d)) return '';
        try {
            $dt = $d instanceof \DateTimeInterface ? Carbon::instance($d) : Carbon::parse($d);
            return $dt->format('Y-m-d');
        } catch (\Throwable $e) {
            return '';
        }
    }

    private function fmtDateTime($d): string
    {
        if (empty($d)) return '';
        try {
            $dt = $d instanceof \DateTimeInterface ? Carbon::instance($d) : Carbon::parse($d);
            return $dt->format('Y-m-d H:i');
        } catch (\Throwable $e) {
            return '';
        }
    }

    /** Excel-safe sheet name (strip invalid chars and trim to 31 chars). */
    private function safeSheetName(string $name): string
    {
        $name = preg_replace('/[:\\\\\\/\\?\\*\\[\\]]+/', ' ', $name) ?? $name;
        $name = trim($name);
        if ($name === '') $name = 'Sheet';
        return mb_substr($name, 0, 31);
    }

    /** Helper: set column widths by header label map. */
    private function setColumnWidthsByHeader(XLSXWriter $writer, array $headers, array $map): void
    {
        $opts = $writer->getOptions();
        foreach ($map as $label => $width) {
            $idx = array_search($label, $headers, true);
            if ($idx !== false) {
                $opts->setColumnWidth((int) $width, $idx + 1); // 1-based
            }
        }
    }

    /**
     * Write a simple header block. Each $row can be:
     *  - ['Single centered title']  -> bold line
     *  - ['Label', 'Value']         -> key/value line
     *  - null                        -> skipped
     */
    private function writeSheetHeader(
        XLSXWriter $writer,
        Style $titleStyle,
        Style $labelStyle,
        array $lines
    ): void {
        foreach ($lines as $row) {
            if ($row === null) continue;
            if (count($row) === 1) {
                $writer->addRow(Row::fromValues($row, $titleStyle));
            } else {
                $writer->addRow(Row::fromValues([$row[0], $row[1]]));
            }
        }
        $writer->addRow(Row::fromValues([])); // spacer
    }

    private function yn($bool): string
    {
        if ($bool === null) return '';
        return $bool ? 'Yes' : 'No';
    }
}
