<?php
namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Clinic;
use App\Models\Company;
use App\Models\Log;
use App\Models\MedicalFundMember;
use App\Models\Patient;
use App\Models\Triage;
use App\Models\User;
use Illuminate\Http\Request;
// use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log as LaravelLog; // your Logs model
use Illuminate\Validation\ValidationException;
use Inertia\Inertia;
use Carbon\Carbon;


class TriageController extends Controller
{
    public function index(Request $request)
    {
        $query = Triage::query()->with(['patient', 'clinic', 'nurse', 'consultor', 'paymentCompany']);

        // Filters
        if ($request->filled('clinic_id')) {
            $query->where('clinic_id', $request->clinic_id);
        }
        if ($request->filled('consultor_id')) {
            $query->where('consultor_id', $request->consultor_id);
        }
        if ($request->filled('triage_level')) {
            $query->where('triage_level', $request->triage_level);
        }
        if ($request->filled('bill_type')) {
            $query->where('bill_type', $request->bill_type);
        }
        if ($request->filled('bill_class')) {
            $query->where('bill_class', $request->bill_class);
        }
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('patient', function ($q) use ($search) {
                $q->where('first_name', 'like', "%$search%")
                    ->orWhere('surname', 'like', "%$search%");
            });
        }

        // Date Range Filter using Laravel's timestamps
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [
                now()->createFromFormat('Y-m-d', $request->start_date)->startOfDay(),
                now()->createFromFormat('Y-m-d', $request->end_date)->endOfDay(),
            ]);
        }

        return Inertia::render('Triage/Index', [
            'triages'        => $query->latest()->paginate(10)->withQueryString(),
            'filters'        => $request->only([
                'search', 'clinic_id', 'consultor_id', 'triage_level',
                'bill_type', 'bill_class', 'payment_method',
                'start_date', 'end_date', // Add date range filters to the filters array
            ]),
            'clinics'        => Clinic::all(),
            'consultors'     => User::where('role', '!=', \App\Enums\UserRole::User)->get(),
            'triageLevels'   => ['Emergency', 'Urgent', 'Routine', 'Non-Urgent'],
            'billTypes'      => [
                'Consultation Adult', 'Consultation Child', 'Follow-up', 'Emergency', 'Vaccination',
                'Lab Test', 'Radiology', 'Review Adult', 'Review Child', 'Antigen Test', 'PCR Test',
                'Rapid Test', 'Vaccine', 'Lead', 'Treatment', 'Other',
            ],
            'billClasses'    => ['Self', 'Family', 'Parent', 'Extended Family', 'Other'],
            'paymentMethods' => [
                'Payroll', 'Pensioner', 'Attachee', 'Customer', 'Cash', 'Insurance', 'Minerva',
                'PSMAS (Cash/5)', 'FLIMAS', 'Cimas', 'EMF', 'Alliance', 'Generation', 'FML',
                'Bonvie', 'Fidelity', 'First Mutual', 'Mars', 'Cellmed', 'Railmed', 'NMAS',
                'Mammoth', 'Emerald', 'Varichem',
            ],
        ]);
    }

    public function show($id)
    {
        $triage = Triage::with(['patient', 'clinic', 'consultor', 'paymentCompany'])->findOrFail($id);

        return Inertia::render('Triage/Show', [
            'triage' => $triage,
        ]);
    }

    public function create(Request $request)
    {
        $patientId       = $request->input('patient_id');
        $selected        = $patientId ? Patient::with(['parent', 'company'])->find($patientId) : null;

        // Determine account holder (parent if dependent, otherwise self)
        $accountHolder = null;
        if ($selected) {
            $accountHolder = $selected->parent ?: $selected;
            $accountHolder->loadMissing('company');
        }

        // Check Medical Fund membership for the account holder by employee_number
        $accountHolderFund = null;
        if ($accountHolder && $accountHolder->employee_number) {
            $fund = MedicalFundMember::with('company')
                ->where('employee_number', $accountHolder->employee_number)
                ->where('tag', 'Providence Fund')
                ->first();

            $accountHolderFund = $fund
                ? [
                    'on_fund'        => true,
                    'tag'            => $fund->tag,
                    'employee_number'=> $fund->employee_number,
                    'company_name'   => $fund->company?->name,
                  ]
                : ['on_fund' => false];
        } else {
            $accountHolderFund = ['on_fund' => false];
        }

        return Inertia::render('Triage/Create', [
            'patient_id'      => $patientId,
            'selectedPatient' => $selected ? [
                'id'                     => $selected->id,
                'company_id'             => $selected->company_id,
                'parent_patient_id'      => $selected->parent_patient_id,
                'parent_company_id'      => $selected->parent?->company_id,
                'parent_employee_number' => $selected->parent?->employee_number,
                // FIX: this previously (accidentally) returned employee_number
                'parent_employee_gender' => $selected->parent?->gender,
            ] : null,

            // NEW: pass account holder snapshot + fund status
            'accountHolder'   => $accountHolder ? [
                'id'               => $accountHolder->id,
                'full_name'        => $accountHolder->full_name,
                'employee_number'  => $accountHolder->employee_number,
                'company_id'       => $accountHolder->company_id,
                'company_name'     => $accountHolder->company?->name,
            ] : null,

            'accountHolderFund' => $accountHolderFund,

            'clinics'         => Clinic::orderBy('name')->get(),
            // 'nurses'          => User::select('id', 'name')->orderBy('name')->get(),
            'nurses'          => User::select('id', 'name')
                                ->whereIn('role', [UserRole::Nurse->value, UserRole::Doctor->value])
                                ->orderBy('name')
                                ->get(),
            'companies'       => Company::orderBy('name')->get(),
            'patients'        => Patient::select(
                                    'id','first_name','surname','company_id','parent_patient_id',
                                    'id_number','employee_number','gender','updated_at','created_at'
                                )
                                ->orderByDesc('updated_at')
                                ->orderByDesc('created_at')
                                ->get(),

            'triageLevels'    => ['Emergency', 'Urgent', 'Routine', 'Non-Urgent'],
            'billTypes'       => [
                'Consultation Adult','Consultation Child','Follow-up','Emergency','Vaccination',
                'Lab Test','Radiology','Review Adult','Review Child','Antigen Test','PCR Test',
                'Rapid Test','Vaccine','Lead','Treatment','Other',
            ],
            'billClasses'     => ['Self', 'Family', 'Parent', 'Extended Family', 'Other'],
            'paymentMethods'  => [
                'Payroll','Pensioner','Attachee','Customer','Cash','Insurance','Minerva',
                'PSMAS (Cash/5)','FLIMAS','Cimas','EMF','Alliance','Generation','FML',
                'Bonvie','Fidelity','First Mutual','Mars','Cellmed','Railmed','NMAS',
                'Mammoth','Emerald','Varichem','Providence Medical Fund','Trainee','Casual','Contractor',
            ],
        ]);
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'patient_id'      => 'required|exists:patients,id',
                'consultor_id'    => 'required|exists:users,id',
                'consultation_id' => 'nullable|exists:consultations,id',
                'triage_level'    => 'nullable|string',
                'bill_type'       => 'nullable|string',
                'bill_class'      => 'nullable|string',
                'payment_point'   => 'nullable|exists:companies,id',
                'payment_method'  => 'nullable|string',
                'chief_complaint' => 'nullable|string',
                'notes'           => 'nullable|string',
            ]);

            $user = Auth::user();

            if (! $user) {
                return back()->withErrors(['auth' => 'You must be logged in to perform this action.']);
            }

            $data['nurse_id']  = $user->id;
            $data['clinic_id'] = $user->clinic_id;

            $triage = Triage::create($data);

            // 🔹 Create a log entry
            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'created',
                'description'   => 'Created a new triage record for patient ID: ' . $data['patient_id'],
                'loggable_type' => Triage::class,
                'loggable_id'   => $triage->id,
                'old_values'    => null,
                'new_values'    => $triage->toArray(),
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            return redirect()->route('patient.search', $triage->id)
                ->with('success', 'Triage record created successfully.');
        } catch (ValidationException $e) {
            return back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            LaravelLog::error('Triage store error: ' . $e->getMessage(), [
                'request' => $request->all(),
                'user_id' => optional(Auth::user())->id,
            ]);

            return back()->with('error', 'An unexpected error occurred. Please try again later.');
        }
    }

    public function edit($id)
    {
        $triage = Triage::with(['patient.parent'])->findOrFail($id);

        return Inertia::render('Triage/Edit', [
            'triage'          => $triage,
            'selectedPatient' => $triage->patient ? [
                'id'                => $triage->patient->id,
                'company_id'        => $triage->patient->company_id,
                'parent_patient_id' => $triage->patient->parent_patient_id,
                'parent_company_id' => $triage->patient->parent?->company_id,
            ] : null,
            'clinics'         => Clinic::all(),
            'nurses'          => User::where('role', '!=', UserRole::User)->get(),
            'companies'       => Company::all(),
            'patients'        => Patient::all(),
            'triageLevels'    => ['Emergency', 'Urgent', 'Routine', 'Non-Urgent'],
            'billTypes'       => [
                'Consultation Adult', 'Consultation Child', 'Follow-up', 'Emergency', 'Vaccination',
                'Lab Test', 'Radiology', 'Review Adult', 'Review Child', 'Antigen Test', 'PCR Test',
                'Rapid Test', 'Vaccine', 'Lead', 'Treatment', 'Other',
            ],
            'billClasses'     => ['Self', 'Family', 'Parent', 'Extended Family', 'Other'],
            'paymentMethods'  => [
                'Payroll','Pensioner','Attachee','Customer','Cash','Insurance','Minerva',
                'PSMAS (Cash/5)','FLIMAS','Cimas','EMF','Alliance','Generation','FML',
                'Bonvie','Fidelity','First Mutual','Mars','Cellmed','Railmed','NMAS',
                'Mammoth','Emerald','Varichem','Providence Medical Fund','Trainee','Casual','Contractor',
            ],
        ]);
    }

    public function update(Request $request, $id)
    {
        try {
            $data = $request->validate([
                'patient_id'      => 'required|exists:patients,id',
                'consultor_id'    => 'required|exists:users,id',
                'consultation_id' => 'nullable|exists:consultations,id',
                'triage_level'    => 'nullable|string',
                'bill_type'       => 'nullable|string',
                'bill_class'      => 'nullable|string',
                'payment_point'   => 'nullable|exists:companies,id',
                'payment_method'  => 'nullable|string',
                'chief_complaint' => 'nullable|string',
                'notes'           => 'nullable|string',
            ]);

            $user = Auth::user();

            if (! $user) {
                return back()->withErrors(['auth' => 'You must be logged in to perform this action.']);
            }

            $triage = Triage::findOrFail($id);

            // Store old values before update
            $oldValues = $triage->getOriginal();

            // Update the triage record
            $triage->update($data);

            // Store new values after update
            $newValues = $triage->fresh()->toArray();

            // 🔹 Create a log entry
            Log::create([
                'user_id'       => Auth::id(),
                'user_name'     => Auth::user()->name ?? null,
                'action'        => 'updated',
                'description'   => 'Updated triage record ID: ' . $triage->id,
                'loggable_type' => Triage::class,
                'loggable_id'   => $triage->id,
                'old_values'    => $oldValues,
                'new_values'    => $newValues,
                'ip_address'    => $request->ip(),
                'user_agent'    => $request->header('User-Agent'),
            ]);

            return redirect()->route('triages.index')
                ->with('success', 'Triage record updated successfully.');
        } catch (ValidationException $e) {
            return back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            LaravelLog::error('Triage update error: ' . $e->getMessage(), [
                'request' => $request->all(),
                'user_id' => optional(Auth::user())->id,
            ]);

            return back()->with('error', 'An unexpected error occurred. Please try again later.');
        }
    }

    public function getByConsultor($consultorId)
    {
        // Return only OPEN (is_active) triages created today; ignore $consultorId
        $triages = Triage::with(['patient', 'clinic', 'nurse', 'consultor', 'paymentCompany'])
            ->where('is_active', true)
            ->whereDate('created_at', Carbon::today())
            ->latest()
            ->get();

        return response()->json([
            'success' => true,
            'data'    => $triages,
        ]);
    }

    public function bulkClose(Request $request)
    {
        $user = $request->user();

        // Authorize (superadmin only)
        if (! $user || ($user->role ?? null) !== 'superadmin') {
            return response()->json([
                'success' => false,
                'message' => 'Forbidden',
            ], 403);
        }

        $validated = $request->validate([
            'ids'   => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'distinct', 'exists:triages,id'],
        ]);

        // Identify active triages that will be closed
        $closingIds = Triage::query()
            ->whereIn('id', $validated['ids'])
            ->where('is_active', true)
            ->pluck('id');

        if ($closingIds->isEmpty()) {
            return response()->json([
                'success'    => true,
                'message'    => 'No active triages to close.',
                'closed_ids' => [],
                'affected'   => 0,
            ]);
        }

        // Perform the update
        $affected = Triage::query()
            ->whereIn('id', $closingIds)
            ->update([
                'is_active'  => false,
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => "{$affected} triage(s) closed.",
            'closed_ids' => $closingIds->values(),
            'affected'   => $affected,
        ]);
    }

}
