<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class UserCompanyAccessController extends Controller
{
    /**
     * Show the UI to edit a user's company access.
     */
    public function edit(User $user)
    {
        $companies = Company::orderBy('name')->get(['id','name']);

        // Use the relationship, not the pivot model directly
        $selectedCompanyIds = $user->accessibleCompanies()->pluck('companies.id')->all();

        return inertia('Users/CompanyAccess', [
            'user'                => $user->only(['id','name','email','role']),
            'companies'           => $companies,
            'selectedCompanyIds'  => $selectedCompanyIds,
        ]);
    }

    /**
     * Sync the set of accessible companies for a user.
     */
    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'company_ids'   => ['array'],
            'company_ids.*' => ['integer', Rule::exists('companies','id')],
        ]);

        $ids = collect($data['company_ids'] ?? [])
            ->map(fn ($id) => (int) $id)
            ->unique()
            ->values();

        $user->accessibleCompanies()->sync($ids->all());

        return back()->with('success', 'Company access updated successfully.');
    }

    /**
     * Grant a single company (AJAX-friendly).
     */
    public function store(Request $request, User $user)
    {
        $validated = $request->validate([
            'company_id' => ['required','integer', Rule::exists('companies','id')],
        ]);

        $user->accessibleCompanies()->syncWithoutDetaching([$validated['company_id']]);

        return back()->with('success', 'Access granted.');
    }

    /**
     * Revoke a single company (AJAX-friendly).
     */
    public function destroy(User $user, Company $company)
    {
        $user->accessibleCompanies()->detach($company->id);

        return back()->with('success', 'Access revoked.');
    }
}
