<?php
namespace App\Http\Controllers;

use App\Enums\UserRole;
use App\Models\Clinic;
use App\Models\Consultation;
use App\Models\Dispensation;
use App\Models\DoctorNote;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class UserController extends Controller
{
    public function index(Request $request)
    {
        // Fetch distinct roles once (for validation + dropdown)
        $validRoles = User::query()
            ->whereNotNull('role')
            ->distinct()
            ->pluck('role')
            ->toArray();

        // Validate inputs (allow the sentinel "__all__" safely)
        $validated = $request->validate([
            'search' => ['nullable', 'string', 'max:255'],
            'role'   => ['nullable', 'string', Rule::in(array_merge($validRoles, ['__all__']))],
        ]);

        $search = (string) ($validated['search'] ?? '');
        $role   = (string) ($validated['role'] ?? '');

        // Map sentinel to "no filter"
        if ($role === '__all__') {
            $role = '';
        }

        $users = User::with('clinic')
            ->when($search !== '', function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                });
            })
            ->when($role !== '', fn($q) => $q->where('role', $role))
            ->orderBy('name')
            ->paginate(10)
            ->withQueryString();

        // Roles for dropdown
        $roles = User::query()
            ->whereNotNull('role')
            ->select('role')
            ->distinct()
            ->orderBy('role')
            ->pluck('role')
            ->values();

        return Inertia::render('Users/Index', [
            'users'   => $users,
            'filters' => [
                'search' => $search,
                // Send the raw role back (null if not filtering, so placeholder shows)
                'role'   => $role !== '' ? $role : null,
            ],
            'roles'   => $roles,
        ]);
    }

    public function edit(User $user)
    {
        $clinics = Clinic::select('id', 'name')->get();
        $roles   = ['user', 'superadmin', 'receptionist', 'nurse', 'doctor']; // adjust as per your roles

        return inertia('Users/Edit', [
            'user'    => $user->only(['id', 'name', 'email', 'clinic_id', 'role']),
            'clinics' => $clinics,
            'roles'   => $roles,
        ]);
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name'           => 'required|string|max:255',
            'email'          => 'required|email|unique:users,email,' . $user->id,
            'clinic_id'      => 'nullable|exists:clinics,id',
            'role'           => 'required|string',
            'qualifications' => 'nullable|string|max:255',
            'signature'      => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Optional image
            'password'       => 'nullable|string|min:8|confirmed',                // Optional password
        ]);

        // Handle signature upload
        if ($request->hasFile('signature')) {
            $signaturePath          = $request->file('signature')->store('signatures', 'public');
            $validated['signature'] = $signaturePath;
        }

        // Conditionally update the password if it is provided
        if ($request->filled('password')) {
            $validated['password'] = Hash::make($validated['password']);
        }

        $user->update($validated);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    // public function changePasswordByEmail(Request $request)
    // {
    //     $validated = $request->validate([
    //         'email'    => 'required|email|exists:users,email',
    //         'password' => 'required|string|min:8|confirmed',
    //     ]);

    //     $user = User::where('email', $validated['email'])->first();

    //     if (! $user) {
    //         return response()->json(['errors' => ['email' => 'User with this email not found']], 404);
    //     }

    //     $user->update([
    //         'password' => Hash::make($validated['password']),
    //     ]);

    //     return response()->json(['message' => 'Password changed successfully.'], 200);
    // }

    public function changePasswordByEmail(Request $request)
    {
        $validated = $request->validate([
            'email'    => 'required|email|exists:users,email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user = User::where('email', $validated['email'])->firstOrFail();
        $user->update(['password' => Hash::make($validated['password'])]);

        return redirect()
            ->route('users.index')
            ->with('success', 'Password changed successfully.');
    }

    public function destroy(Request $request, User $user)
    {
        // Optional policy: $this->authorize('delete', $user);

        // 1) Don’t allow deleting yourself
        if (Auth::id() === $user->id) {
            return back()->withErrors(['user' => 'You cannot delete your own account.']);
        }

        // 2) Don’t allow deleting the last superadmin
        if ($user->role?->value === 'superadmin') {
            $otherSuperadmins = User::query()
                ->where('id', '!=', $user->id)
                ->where('role', UserRole::SuperAdmin) // adjust if your enum uses a different case name
                ->count();

            if ($otherSuperadmins === 0) {
                return back()->withErrors(['user' => 'You cannot delete the last superadmin.']);
            }
        }

        DB::transaction(function () use ($user) {
            // --- Delete stored signature file if present ---
            if (! empty($user->signature)) {
                try {
                    Storage::disk('public')->delete($user->signature);
                } catch (\Throwable $e) {
                    // ignore missing file/errors
                }
            }

            // --- Delete doctor's notes authored by this user (doctor_id) ---
            if (class_exists(DoctorNote::class)) {
                DoctorNote::where('doctor_id', $user->id)
                    ->chunkById(500, function ($notes) {
                        foreach ($notes as $note) {
                            $note->delete();
                        }
                    });
            }

            // --- Delete dispensations done by this user ---
            // IMPORTANT: correct column is 'dispensed_by' (not dispensed_by_id)
            if (class_exists(Dispensation::class)) {
                Dispensation::where('dispensed_by', $user->id)
                    ->chunkById(500, function ($rows) {
                        foreach ($rows as $row) {
                            $row->delete();
                        }
                    });
            }

            // --- Delete consultations where this user is the doctor ---
            if (class_exists(Consultation::class)) {
                Consultation::where('doctor_id', $user->id)
                    ->chunkById(200, function ($consultations) {
                        foreach ($consultations as $c) {
                            // Best-effort deletes for known relations
                            try {
                                if (method_exists($c, 'dispensations')) {
                                    $c->dispensations()->delete();
                                }
                            } catch (\Throwable $e) {}

                            try {
                                if (method_exists($c, 'doctorsnote')) {
                                    $c->doctorsnote()->delete();
                                }
                            } catch (\Throwable $e) {}

                            try {
                                if (method_exists($c, 'triages')) {
                                    $c->triages()->delete();
                                }
                            } catch (\Throwable $e) {}

                            try {
                                if (method_exists($c, 'logs')) {
                                    $c->logs()->delete();
                                }
                            } catch (\Throwable $e) {}

                            $c->delete();
                        }
                    });
            }

            // --- Delete logs authored by this user (if you have such a model) ---
            if (class_exists(\App\Models\Log::class)) {
                \App\Models\Log::where('user_id', $user->id)->delete();
            }

            // --- Finally: delete the user account itself ---
            $user->delete();
        });

        return redirect()
            ->route('users.index')
            ->with('success', 'User and related records deleted successfully.');
    }

}
