<?php
namespace App\Http\Controllers;

use App\Models\Module;
use App\Models\User;
use App\Models\UserModuleAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class UserModuleAccessController extends Controller
{
    

    /**
     * Show the edit screen to manage which modules a user can access.
     * GET /users/{user}/module-access
     */
    public function edit(User $user)
    {
        // Optional: $this->authorize('update', $user);

        // All available modules
        $modules = Module::query()
            ->orderBy('name')
            ->get(['id', 'slug', 'name']);

        // Current access rows for this user (pivot)
        $accessRows = UserModuleAccess::query()
            ->where('user_id', $user->id)
            ->get(['module_id', 'is_enabled']);

        // Build [module_id => is_enabled] map
        $enabledMap = [];
        foreach ($accessRows as $row) {
            $enabledMap[(int) $row->module_id] = (bool) $row->is_enabled;
        }

        // Superadmins effectively have access to all modules
        $isSuperadmin = $user->role?->value === 'superadmin';
        if ($isSuperadmin) {
            foreach ($modules as $m) {
                $enabledMap[(int) $m->id] = true;
            }
        }

        return Inertia::render('Users/ModuleAccessEdit', [
            'user'        => [
                'id'    => $user->id,
                'name'  => $user->name,
                'email' => $user->email,
                'role'  => $user->role?->value,
            ],
            'modules'     => $modules,
            'enabled_map' => $enabledMap,   // { [module_id]: true|false }
            'readonly'    => $isSuperadmin, // disable form on FE if superadmin
        ]);
    }

    /**
     * Update module access for a user.
     *
     * PUT /users/{user}/module-access
     *
     * Accepts either:
     *  1) { module_ids: number[] }  // enabled list
     *  2) { modules: [{ id:number, enabled:boolean }, ...] } // explicit states
     *
     * Also tolerates module_ids as a JSON string and coerces it to array.
     */
    public function update(Request $request, User $user)
    {
        // Optional: $this->authorize('update', $user);

        // Superadmin: do not allow modifying (always all-access)
        if ($user->role?->value === 'superadmin') {
            return back()->with('warning', 'Superadmin already has access to all modules.');
        }

        // Coerce JSON string → array if needed for backward compatibility
        if (is_string($request->input('module_ids'))) {
            $decoded = json_decode($request->input('module_ids'), true);
            if (is_array($decoded)) {
                $request->merge(['module_ids' => $decoded]);
            }
        }

        $validated = $request->validate([
            'module_ids'        => ['sometimes', 'array'],
            'module_ids.*'      => ['integer', Rule::exists('modules', 'id')],

            'modules'           => ['sometimes', 'array'],
            'modules.*.id'      => ['required_with:modules', 'integer', Rule::exists('modules', 'id')],
            'modules.*.enabled' => ['required_with:modules', 'boolean'],
        ]);

        // Pull all module ids once
        $allModuleIds = Module::query()
            ->pluck('id')
            ->map(fn($id) => (int) $id)
            ->all();

        // Build desired state map: [module_id => bool enabled]
        $desired = [];

        if (array_key_exists('modules', $validated)) {
            foreach ($validated['modules'] as $row) {
                $desired[(int) $row['id']] = (bool) $row['enabled'];
            }
            // default any not present to false
            foreach ($allModuleIds as $mid) {
                if (! array_key_exists($mid, $desired)) {
                    $desired[$mid] = false;
                }
            }
        } else {
            // Using module_ids list as the enabled set
            $enabledIds = collect($validated['module_ids'] ?? [])
                ->map(fn($v) => (int) $v)
                ->all();

            foreach ($allModuleIds as $mid) {
                $desired[$mid] = in_array($mid, $enabledIds, true);
            }
        }

        // Current state
        $currentRows = UserModuleAccess::query()
            ->where('user_id', $user->id)
            ->get(['id', 'module_id', 'is_enabled']);

        $currentMap = [];
        foreach ($currentRows as $row) {
            $currentMap[(int) $row->module_id] = (bool) $row->is_enabled;
        }

        DB::transaction(function () use ($user, $desired, $currentMap) {
            foreach ($desired as $moduleId => $enabled) {
                $was = $currentMap[$moduleId] ?? null;

                if ($enabled === true) {
                    if ($was === true) {
                        continue; // no change
                    }
                    // create/enable
                    UserModuleAccess::query()->updateOrCreate(
                        ['user_id' => $user->id, 'module_id' => $moduleId],
                        ['is_enabled' => true]
                    );
                } else {
                    // desired false
                    if ($was === null) {
                        continue; // nothing to remove
                    }
                    // delete row to keep table small (could also set is_enabled=false if you prefer)
                    UserModuleAccess::query()
                        ->where('user_id', $user->id)
                        ->where('module_id', $moduleId)
                        ->delete();
                }
            }
        });

        return redirect()
            ->route('users.module-access.edit', $user->id)
            ->with('success', 'Module access updated.');
    }
}
