<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EnsureModuleChosen
{
    protected function moduleDashboardRoute(string $slug): string
    {
        return match ($slug) {
            'dental'     => 'dental.dashboard',
            'chronics'   => 'chronics.dashboard',
            'secondary'  => 'secondary.dashboard', // ✅ Secondary Health
            'medicals'   => 'medicals.dashboard',  // ✅ Medicals
            'primary'    => 'dashboard',
            default      => 'dashboard',
        };
    }

    /** Routes that should NOT trigger kiosk/module logic (always allowed). */
    protected function isWhitelistedRoute(Request $request): bool
    {
        $name = optional($request->route())->getName();
        if (!$name) return false;

        $whitelist = [
            // kiosk
            'kiosk.index', 'kiosk.choose', 'kiosk.reset',
            // module access admin
            'users.module-access.edit', 'users.module-access.update',
        ];

        return in_array($name, $whitelist, true);
    }

    public function handle(Request $request, Closure $next)
    {
        if (!Auth::check()) {
            return $next($request);
        }

        if ($this->isWhitelistedRoute($request)) {
            return $next($request);
        }

        $user       = $request->user();
        $routeName  = optional($request->route())->getName();
        $allowed    = $user->enabledModuleSlugs(); // e.g. ['primary','dental','secondary','medicals']
        $inKiosk    = str_starts_with((string) $routeName, 'kiosk.');

        // 0 modules -> default to primary
        if (count($allowed) === 0) {
            $request->session()->put('active_module', 'primary');

            $target = $this->moduleDashboardRoute('primary');
            if ($routeName !== $target) {
                return redirect()->route($target);
            }

            return $next($request);
        }

        // Exactly 1 module -> set it in session, avoid loops
        if (count($allowed) === 1) {
            $only = $allowed[0];

            if ($request->session()->get('active_module') !== $only) {
                $request->session()->put('active_module', $only);
            }

            $target = $this->moduleDashboardRoute($only);

            if ($inKiosk) {
                return redirect()->route($target);
            }

            if ($routeName === $target) {
                return $next($request);
            }

            return $next($request);
        }

        // Multiple modules
        if (!$request->session()->has('active_module')) {
            // No selection yet → send to kiosk to choose
            return redirect()->route('kiosk.index');
        }

        $active = (string) $request->session()->get('active_module');

        // If saved module is no longer allowed, reset to kiosk
        if (!in_array($active, $allowed, true)) {
            $request->session()->forget('active_module');
            return redirect()->route('kiosk.index')
                ->with('warning', 'You no longer have access to that module. Please choose one.');
        }

        return $next($request);
    }
}
