<?php

namespace App\Http\Middleware;

use App\Models\Patient;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class EnsurePatientReadyForTriage
{
    public function handle(Request $request, Closure $next): Response
    {
        $patientId = $request->input('patient_id');

        // If no patient chosen yet, let the request proceed (triage create can show selectors)
        if (!$patientId) {
            return $next($request);
        }

        $patient = Patient::with('parent')->find($patientId);

        // If invalid patient id, let controller handle 404/validation
        if (!$patient) {
            return $next($request);
        }

        // Helpers
        $isPhoneComplete = function (?string $phone): bool {
            $digits = preg_replace('/\D+/', '', (string) $phone);
            return strlen($digits) >= 9;
        };

        // Treat "" and "263" as missing phone (legacy), also incomplete phone
        $phoneValue = trim((string) $patient->phone);
        $needsPhone = ($phoneValue === '' || $phoneValue === '263' || !$isPhoneComplete($phoneValue));

        // Account holder = parent (if dependent) or the patient
        $accountHolder = $patient->parent ?: $patient;
        $needsCompany  = empty($accountHolder->company_id);

        // Emergency Contact checks
        $needsEmergencyName   = empty(trim((string) $patient->emergency_contact_name));
        $needsEmergencyRel    = empty(trim((string) $patient->emergency_contact_relation));
        $needsEmergencyPhone  = !$isPhoneComplete($patient->emergency_contact_phone ?? '');
        $needsEmergency       = ($needsEmergencyName || $needsEmergencyRel || $needsEmergencyPhone);

        // Date of Birth check (must be present)
        $needsDob = empty($patient->date_of_birth);

        if ($needsPhone || $needsCompany || $needsEmergency || $needsDob) {
            // Which record needs editing?
            // - If patient’s own phone/emergency/dob missing -> edit the selected patient
            // - Else if company missing -> edit the account holder
            $patientToEditId = ($needsPhone || $needsEmergency || $needsDob) ? $patient->id : $accountHolder->id;

            // Return URL back to triage create with the same patient pre-selected
            $returnUrl = route('triages.create', ['patient_id' => $patient->id]);

            // Build human-friendly missing list
            $missing = [];
            if ($needsPhone)        { $missing[] = 'phone number'; }
            if ($needsCompany)      { $missing[] = 'company (account holder)'; }
            if ($needsDob)          { $missing[] = 'date of birth'; }
            if ($needsEmergencyName){ $missing[] = 'emergency contact name'; }
            if ($needsEmergencyRel) { $missing[] = 'emergency contact relationship'; }
            if ($needsEmergencyPhone){ $missing[] = 'emergency contact phone'; }

            $reasonText = 'missing ' . implode(' and ', $missing);

            return redirect()
                ->route('patients.edit', ['patient' => $patientToEditId, 'return' => $returnUrl])
                ->with('warning', "Please complete the patient's profile ($reasonText) before creating a triage.")
                ->with('missing_fields', $missing);
        }

        return $next($request);
    }
}
