<?php
namespace App\Http\Middleware;

use App\Models\User; // 👈 needed for name lookup of the real user
use Illuminate\Foundation\Inspiring;
use Illuminate\Http\Request;
use Inertia\Middleware;
use Tighten\Ziggy\Ziggy;

class HandleInertiaRequests extends Middleware
{
    protected $rootView = 'app';

    public function version(Request $request): ?string
    {
        return parent::version($request);
    }

    public function share(Request $request): array
    {
        [$message, $author] = str(Inspiring::quotes()->random())->explode('-');

        // laravel-impersonate session key (default is 'impersonated_by')
        $sessionKey     = config('impersonate.session_key', 'impersonated_by');
        $impersonatorId = session($sessionKey);
        $impersonator   = $impersonatorId ? User::find($impersonatorId) : null;

        return [
             ...parent::share($request),

            'name'          => config('app.name'),
            'quote'         => ['message' => trim($message), 'author' => trim($author)],

            // 🔹 Share the currently authenticated (effective) user
            'auth'          => [
                // You can send the whole $request->user() as you did,
                // or a trimmed version like below:
                'user' => $request->user()
                ? [
                    'id'        => $request->user()->id,
                    'name'      => $request->user()->name,
                    'email'     => $request->user()->email,
                    'role'      => $request->user()->role?->value, // if using enum
                    'clinic_id' => $request->user()->clinic_id,
                ]
                : null,
            ],
            'flash' => [
                'success' => fn () => $request->session()->get('success'),
                'error'   => fn () => $request->session()->get('error'),
                'info'    => fn () => $request->session()->get('info'),
                // include our bulk summary payload
                'bulk_delete_result' => fn () => $request->session()->get('bulk_delete_result'),
            ],

            // 🔶 Impersonation state for your banner
            'impersonation' => [
                'active'            => app('impersonate')->isImpersonating(),
                'impersonator_id'   => $impersonatorId,      // original admin's id
                'impersonator_name' => $impersonator?->name, // original admin's name
                                                             // optional convenience flags:
                'can_impersonate'   => (bool) ($request->user()?->canImpersonate()),
            ],

            'ziggy'         => fn(): array        => [
                 ...(new Ziggy)->toArray(),
                'location' => $request->url(),
            ],

            'sidebarOpen'   => ! $request->hasCookie('sidebar_state') || $request->cookie('sidebar_state') === 'true',
        ];
    }
}
