<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class Clinic extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'address',
        'city',
        'afhoz_number',
        'contact_person',
        'manager',
        'country',
        'email_address',
    ];

    /**
     * Users that belong to this clinic.
     */
    public function users(): HasMany
    {
        return $this->hasMany(User::class);
    }

    public function medicationStocks(): HasMany
    {
        return $this->hasMany(ClinicMedicationStock::class);
    }

    // NOTE: Keep only if your patients table actually has clinic_id.
    // Otherwise don't use this relation for counts.
    public function patients(): HasMany
    {
        return $this->hasMany(Patient::class);
    }

    public function triages(): HasMany
    {
        return $this->hasMany(Triage::class);
    }

    public function consultations(): HasMany
    {
        return $this->hasMany(Consultation::class);
    }

    /**
     * Dispensations made at this clinic via consultations.
     * clinics.id -> consultations.clinic_id -> dispensations.consultation_id
     */
    public function dispensations(): HasManyThrough
    {
        return $this->hasManyThrough(
            Dispensation::class, // Final related model
            Consultation::class, // Intermediate model
            'clinic_id',         // Foreign key on consultations table
            'consultation_id',   // Foreign key on dispensations table
            'id',                // Local key on clinics table
            'id'                 // Local key on consultations table
        );
    }

    public function medicationRequests(): HasMany
    {
        return $this->hasMany(MedicationRequest::class);
    }

    public function usersWithAccess(): BelongsToMany
    {
        return $this->belongsToMany(
            \App\Models\User::class,
            'user_clinic_accesses'
        )->withTimestamps();
    }
}
