<?php
namespace App\Models;

use App\Models\Clinic;
use App\Models\Company;
use App\Models\Patient;
use App\Models\User;

// Add missing imports for related models you reference
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\DB;

class Consultation extends Model
{
    protected $fillable = [
        'patient_id',
        'doctor_id',
        'clinic_id',
        'injury_on_duty',
        'consultation_date',

        // Vitals
        'systolic',
        'diastolic',
        'temperature',
        'heart_rate',
        'respiratory_rate',
        'oxygen_saturation',
        'weight',
        'height',
        'bmi',

                                   // Glucose (simple inline recording)
        'blood_sugar',             // stored canonically as mmol/L
        'blood_sugar_unit',        // 'mmol/L' or 'mg/dL' (we’ll persist 'mmol/L' by convention)
        'blood_sugar_context',     // fasting/random/postprandial/ogtt/other
        'blood_sugar_measured_at', // datetime

        // Other tests
        'urine_test_result',
        'hiv_status',
        'pregnancy_test_result',

        // Notes
        'presenting_complaint',
        'clinical_notes',
        'diagnosis',
        'treatment_plan',
        'prescription',

        // NEW
        'restricted_duty',

        // NEW
        'referred_hospital',

        // Follow-up / leave
        'sick_leave',
        'reason_for_leave',
        'sick_leave_notes',
        'number_of_days',
        'follow_up_date',
        // (You had 'follow_up_required' in fillable previously, include only if it exists in DB)
        'follow_up_required',

        // Referrals
        'referral_notes',
        'refer_for_radiology',
        'refer_for_lab',
        'refer_to_specialist',
        'specialist_name',
        'refer_to_casualty',
        'refer_out_patient',

        'is_active',
        'status',
    ];

    protected $casts = [
        'consultation_date'       => 'datetime',
        'blood_sugar'             => 'decimal:2',
        'blood_sugar_measured_at' => 'datetime',

        // Nice to explicitly cast booleans if you're using MySQL TINYINT(1)
        'injury_on_duty'          => 'boolean',
        'sick_leave'              => 'boolean',
        'follow_up_required'      => 'boolean',
        'refer_for_radiology'     => 'boolean',
        'refer_for_lab'           => 'boolean',
        'refer_to_specialist'     => 'boolean',
        'refer_to_casualty'       => 'boolean',
        'refer_out_patient'       => 'boolean',
        'is_active'               => 'boolean',
        'status'                  => 'boolean',

        // Decimals
        'temperature'             => 'decimal:1',
        'heart_rate'              => 'decimal:2',
        'respiratory_rate'        => 'decimal:2',
        'oxygen_saturation'       => 'decimal:2',
        'weight'                  => 'decimal:2',
        'height'                  => 'decimal:2',
        'bmi'                     => 'decimal:2',
        'blood_sugar'             => 'decimal:2',
    ];

    /**
     * Accessor: blood sugar in mg/dL (derived from canonical mmol/L).
     */
    public function getBloodSugarMgdlAttribute(): ?float
    {
        if ($this->blood_sugar === null) {
            return null;
        }
        return round((float) $this->blood_sugar * 18, 2);
    }

    /**
     * Mutator helper: set blood sugar with unit handling.
     * Use like: $consultation->setBloodSugarWithUnit($value, 'mg/dL');
     */
    public function setBloodSugarWithUnit(float $value, string $unit = 'mmol/L'): void
    {
        if ($unit === 'mg/dL') {
            $this->blood_sugar      = round($value / 18, 2); // store as mmol/L
            $this->blood_sugar_unit = 'mmol/L';
        } else {
            $this->blood_sugar      = round($value, 2);
            $this->blood_sugar_unit = 'mmol/L';
        }
    }

    // -------- Aggregates / dashboards --------

    public static function getSickLeaveIssuedPerMonth($year, $clinicId = null)
    {
        $months = [
            'January', 'February', 'March', 'April', 'May', 'June',
            'July', 'August', 'September', 'October', 'November', 'December',
        ];

        $results = [];

        foreach ($months as $idx => $month) {
            $start_date = Carbon::createFromDate($year, $idx + 1, 1)->startOfMonth();
            $end_date   = (clone $start_date)->endOfMonth();

            $query = DB::table('consultations')
                ->where('sick_leave', 1)
                ->whereBetween('consultation_date', [$start_date, $end_date]);

            if ($clinicId) {
                $query->where('clinic_id', $clinicId);
            }

            $results[] = [
                'month' => $month,
                'count' => (int) $query->count(),
            ];
        }

        return $results;
    }

    public static function getInjuryOnDutyCasesPerCompany($year, $companyId = null, $clinicId = null)
    {
        $months = [
            'January', 'February', 'March', 'April', 'May', 'June',
            'July', 'August', 'September', 'October', 'November', 'December',
        ];

        if (! $companyId) {
            $firstCompany = Company::first();
            if ($firstCompany) {
                $companyId = $firstCompany->id;
            } else {
                return [];
            }
        }

        $company = Company::find($companyId);
        if (! $company) {
            return [];
        }

        $companyData = [
            'company_id'   => $company->id,
            'company_name' => $company->name,
            'months'       => [],
        ];

        foreach ($months as $monthIndex => $month) {
            $start_date = Carbon::createFromDate($year, $monthIndex + 1, 1)->startOfMonth();
            $end_date   = (clone $start_date)->endOfMonth();

            $query = DB::table('consultations')
                ->join('patients', 'consultations.patient_id', '=', 'patients.id')
                ->where('consultations.injury_on_duty', 1)
                ->where('patients.company_id', $company->id)
                ->whereBetween('consultations.consultation_date', [$start_date, $end_date]);

            if ($clinicId) {
                $query->where('consultations.clinic_id', $clinicId);
            }

            $companyData['months'][] = [
                'month' => $month,
                'count' => (int) $query->count(),
            ];
        }

        return [$companyData];
    }

    public static function getReferralsPerMonth($year, $clinicId = null)
    {
        $months = [
            'January', 'February', 'March', 'April', 'May', 'June',
            'July', 'August', 'September', 'October', 'November', 'December',
        ];

        $referralTypes = [
            'refer_for_radiology',
            'refer_for_lab',
            'refer_to_specialist',
            'refer_to_casualty',
            'refer_out_patient',
        ];

        $results = [];

        foreach ($months as $idx => $month) {
            $start_date = Carbon::createFromDate($year, $idx + 1, 1)->startOfMonth();
            $end_date   = (clone $start_date)->endOfMonth();

            $monthData = [
                'month'  => $month,
                'counts' => [],
            ];

            foreach ($referralTypes as $type) {
                $query = DB::table('consultations')
                    ->where($type, 1)
                    ->whereBetween('consultation_date', [$start_date, $end_date]);

                if ($clinicId) {
                    $query->where('clinic_id', $clinicId);
                }

                $monthData['counts'][] = [
                    'type'  => $type,
                    'count' => (int) $query->count(),
                ];
            }

            $results[] = $monthData;
        }

        return $results;
    }

    // -------- Relationships --------

    // in App\Models\Consultation
    public function imagingReferrals()
    {return $this->hasMany(\App\Models\ImagingReferral::class);}
    public function labReferrals()
    {return $this->hasMany(\App\Models\LabReferral::class);}

    public function patient(): BelongsTo
    {
        return $this->belongsTo(Patient::class);
    }

    public function doctor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'doctor_id');
    }

    public function clinic(): BelongsTo
    {
        return $this->belongsTo(Clinic::class);
    }

    public function medicationUsages()
    {
        return $this->hasMany(MedicationUsage::class);
    }

    public function dispensations()
    {
        return $this->hasMany(Dispensation::class);
    }

    public function triage()
    {
        return $this->hasOne(Triage::class);
    }

    public function doctorsnote()
    {
        return $this->hasOne(DoctorNote::class);
    }

    /**
     * NOTE: The triage() relation in your snippet had a comment "In Patient.php model".
     * If you need a triage relation, define it on the correct model (likely Patient or Consultation),
     * but it's removed here to avoid confusion.
     */

    // Limit consultations to user's company access (incl. dependents via patient->parent)
    public function scopeAccessibleByUser($query, \App\Models\User $user)
    {
        if ($user->role?->value === 'superadmin') {
            return $query;
        }
        $ids = $user->accessibleCompanyIds();
        if (empty($ids)) {
            return $query->whereRaw('1=0');
        }

        return $query->whereHas('patient', function ($q) use ($ids) {
            $q->whereIn('company_id', $ids)
                ->orWhereHas('parent', function ($q2) use ($ids) {
                    $q2->whereIn('company_id', $ids);
                });
        });
    }

}
