<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class DoctorNote extends Model
{
    use SoftDeletes;

    protected $table = 'doctor_notes';

    protected $fillable = [
        'consultation_id',
        'doctor_id',
        'patient_id',
        'title',
        'body',
        'note_type',
        'private',
        'meta',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'private' => 'boolean',
        'meta'    => 'array',
    ];

    /** Utility: apply withTrashed() only if the related model uses SoftDeletes */
    protected function maybeWithTrashed(BelongsTo $relation, string $relatedClass): BelongsTo
    {
        try {
            if (in_array(SoftDeletes::class, class_uses_recursive($relatedClass), true)) {
                // BelongsTo::withTrashed exists; will only affect queries if related uses SoftDeletes
                $relation->withTrashed();
            }
        } catch (\Throwable $e) {
            // ignore – stay safe if reflection fails
        }
        return $relation;
    }

    /** Patient (include soft-deleted when applicable) */
    public function patient(): BelongsTo
    {
        $rel = $this->belongsTo(Patient::class, 'patient_id');
        $rel = $this->maybeWithTrashed($rel, Patient::class);

        return $rel->select(['id', 'first_name', 'surname', 'employee_number']);
    }

    /** Consultation (include soft-deleted when applicable) */
    public function consultation(): BelongsTo
    {
        $rel = $this->belongsTo(Consultation::class, 'consultation_id');
        $rel = $this->maybeWithTrashed($rel, Consultation::class);

        return $rel->select(['id', 'consultation_date']);
    }

    /** Doctor (slim columns) */
    public function doctor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'doctor_id')
            ->select(['id', 'name', 'email']);
    }

    // Optional helper for short preview
    public function excerpt(): Attribute
    {
        return Attribute::get(fn() => Str::limit(strip_tags($this->body), 150));
    }

    // Robust getter if meta might sometimes be stored as string
    protected function meta(): Attribute
    {
        return Attribute::make(
            get: fn($value) => is_string($value) ? (json_decode($value, true) ?? []) : ($value ?? [])
        );
    }
}
