<?php
namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class Patient extends Model
{
    // Mass assignable attributes
    protected $fillable = [
        'first_name',
        'middle_name',
        'surname',
        'employee_number',

        'home_address',
        'work_area',
        'suburb',

        'company_id',
        'parent_patient_id',
        'date_of_birth',
        'email',
        'gender',
        'id_number',
        'medical_aid_number',
        'medical_aid_provider',
        'relationship',
        'phone',
        
        'emp_start_date',
        'emp_end_date',

        'emergency_contact_name',
        'emergency_contact_relation',
        'emergency_contact_phone',

        'allergies',
        'is_smoker',
        'occupation',
        'marital_status',
        'is_chronic_patient',
    ];

    // Casts
    protected $casts = [
        'date_of_birth'      => 'date:Y-m-d',
        'is_smoker'          => 'boolean',
        'is_chronic_patient' => 'boolean',
    ];

    /*
    |--------------------------------------------------------------------------
    | Mutators (normalize & Title Case names)
    |--------------------------------------------------------------------------
    */
    protected function firstName(): Attribute
    {
        return Attribute::make(
            set: fn($value) => $this->normalizeName($value)
        );
    }

    protected function middleName(): Attribute
    {
        return Attribute::make(
            set: fn($value) => $this->normalizeName($value)
        );
    }

    protected function surname(): Attribute
    {
        return Attribute::make(
            set: fn($value) => $this->normalizeName($value)
        );
    }

    public function chronicDispensations()
    {
        return $this->hasMany(\App\Models\ChronicDispensation::class);
    }

    protected function normalizeName(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }
        // Trim & collapse whitespace, then Title Case
        $value = preg_replace('/\s+/u', ' ', trim($value));
        return Str::title($value);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors
    |--------------------------------------------------------------------------
    */
    public function getFullNameAttribute(): string
    {
        // Join non-empty parts cleanly (handles null/empty middle names)
        $parts = array_filter([$this->first_name, $this->middle_name, $this->surname], fn($v) => filled($v));
        return trim(implode(' ', $parts));
    }

    /*
    |--------------------------------------------------------------------------
    | Query helpers / scopes
    |--------------------------------------------------------------------------
    */
    public function scopeRecentlyUpdated($query)
    {
        return $query->orderByDesc('updated_at')->orderByDesc('created_at');
    }

    /*
    |--------------------------------------------------------------------------
    | Aggregates / Stats
    |--------------------------------------------------------------------------
    */
    public static function getGenderDistribution()
    {
        return DB::table('patients')
            ->select('gender', DB::raw('COUNT(*) as count'))
            ->groupBy('gender')
            ->get()
            ->toArray();
    }

    public static function getPatientsByAgeRange()
    {
        $currentDate = Carbon::now();
        $ageRanges   = [
            ['range' => '0-10', 'min_age' => 0, 'max_age' => 10],
            ['range' => '11-20', 'min_age' => 11, 'max_age' => 20],
            ['range' => '21-30', 'min_age' => 21, 'max_age' => 30],
            ['range' => '31-40', 'min_age' => 31, 'max_age' => 40],
            ['range' => '41-50', 'min_age' => 41, 'max_age' => 50],
            ['range' => '51-60', 'min_age' => 51, 'max_age' => 60],
            ['range' => '61+', 'min_age' => 61, 'max_age' => null],
        ];

        $results = [];

        foreach ($ageRanges as $range) {
            $minAge = $range['min_age'];
            $maxAge = $range['max_age'];

            $start_date = $currentDate->copy()->subYears($minAge)->startOfDay();
            $end_date   = $maxAge ? $currentDate->copy()->subYears($maxAge)->endOfDay() : null;

            $count = DB::table('patients')
                ->whereDate('date_of_birth', '<=', $start_date)
                ->when($end_date, fn($query) => $query->whereDate('date_of_birth', '>', $end_date))
                ->count();

            $results[] = [
                'range' => $range['range'],
                'count' => $count,
            ];
        }

        return $results;
    }

    public static function getEmployeeVsDependentsCount()
    {
        $employees  = DB::table('patients')->whereNull('parent_patient_id')->count();
        $dependents = DB::table('patients')->whereNotNull('parent_patient_id')->count();

        return [
            ['category' => 'Employees', 'count' => $employees],
            ['category' => 'Dependents', 'count' => $dependents],
        ];
    }

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */
    /**
     * The company the patient belongs to (nullable for dependents).
     */

    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    public function clinic(): BelongsTo
    {
        return $this->belongsTo(Clinic::class);
    }
    /**
     * The parent patient if this is a dependent.
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(Patient::class, 'parent_patient_id');
    }

    /**
     * Dependents for this patient.
     */
    public function dependents(): HasMany
    {
        return $this->hasMany(Patient::class, 'parent_patient_id');
    }

    public function consultations(): HasMany
    {
        return $this->hasMany(Consultation::class);
    }

    public function latestConsultation()
    {
        // hasOne->latestOfMany() returns a HasOne, but many codebases type as BelongsTo|HasOne.
        return $this->hasOne(Consultation::class)->latestOfMany();
    }

    // Limit patients to a set of company IDs (honoring dependents via parent->company_id)
    public function scopeWhereCompanyInOrParentCompanyIn($query, array $companyIds)
    {
        return $query->where(function ($q) use ($companyIds) {
            $q->whereIn('company_id', $companyIds)
                ->orWhereHas('parent', function ($q2) use ($companyIds) {
                    $q2->whereIn('company_id', $companyIds);
                });
        });
    }

// Convenience: limit to currently logged-in user's accessible companies
    public function scopeAccessibleByUser($query, \App\Models\User $user)
    {
        if ($user->role?->value === 'superadmin') {
            return $query; // no restriction
        }
        $ids = $user->accessibleCompanyIds();
        if (empty($ids)) {
            // Force empty result if no companies assigned
            return $query->whereRaw('1=0');
        }
        return $this->scopeWhereCompanyInOrParentCompanyIn($query, $ids);
    }

}
