<?php

namespace App\Models;

use App\Enums\UserRole;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Lab404\Impersonate\Models\Impersonate;

class User extends Authenticatable
{
    use HasFactory, Notifiable, Impersonate;

    protected $fillable = [
        'name',
        'email',
        'password',
        'clinic_id',
        'role',
        'qualifications',
        'signature',
    ];

    protected $hidden = ['password','remember_token'];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password'          => 'hashed',
            'role'              => UserRole::class,
        ];
    }

    // ───────────────────────────────── Relationships ─────────────────────────────────

    public function clinic(): BelongsTo
    {
        return $this->belongsTo(Clinic::class);
    }

    public function clinicAccesses(): HasMany
    {
        return $this->hasMany(UserClinicAccess::class);
    }

    public function accessibleClinics(): BelongsToMany
    {
        return $this->belongsToMany(Clinic::class, 'user_clinic_accesses')->withTimestamps();
    }

    public function companyAccesses(): HasMany
    {
        return $this->hasMany(UserCompanyAccess::class);
    }

    public function accessibleCompanies(): BelongsToMany
    {
        return $this->belongsToMany(Company::class, 'user_company_accesses')->withTimestamps();
    }

    /** Pivot to modules with the extra `is_enabled` column. */
    public function modules(): BelongsToMany
    {
        return $this->belongsToMany(Module::class, 'user_module_accesses')
            ->withPivot('is_enabled')
            ->withTimestamps();
    }

    // ───────────────────────────────── Impersonation ─────────────────────────────────

    public function canImpersonate(): bool
    {
        return $this->role?->value === 'superadmin';
    }

    public function canBeImpersonated(): bool
    {
        return $this->role?->value !== 'superadmin';
    }

    // ───────────────────────────────── Helpers / Scopes ─────────────────────────────────

    public function accessibleClinicIds(): array
    {
        $ids = $this->accessibleClinics()->pluck('clinics.id')->all();
        if (!empty($this->clinic_id) && !in_array((int)$this->clinic_id, $ids, true)) {
            $ids[] = (int)$this->clinic_id;
        }
        return array_values(array_unique($ids));
    }

    public function hasClinicAccess(int|string $clinicId): bool
    {
        $clinicId = (int) $clinicId;
        if (!empty($this->clinic_id) && (int)$this->clinic_id === $clinicId) {
            return true;
        }
        return $this->accessibleClinics()->where('clinics.id', $clinicId)->exists();
    }

    public function scopeWhereHasAccessToClinic(Builder $query, int|string $clinicId): Builder
    {
        $clinicId = (int) $clinicId;
        return $query->where(function (Builder $q) use ($clinicId) {
            $q->where('clinic_id', $clinicId)
              ->orWhereHas('accessibleClinics', fn (Builder $q2) => $q2->where('clinics.id', $clinicId));
        });
    }

    public function accessibleCompanyIds(): array
    {
        if ($this->role?->value === 'superadmin') {
            return Company::query()->pluck('id')->all();
        }
        return $this->accessibleCompanies()->pluck('companies.id')->all();
    }

    public function hasCompanyAccess(int|string $companyId): bool
    {
        if ($this->role?->value === 'superadmin') return true;
        return in_array((int)$companyId, $this->accessibleCompanyIds(), true);
    }

    /** Enabled module slugs for this user (superadmin = all). */
    public function enabledModuleSlugs(): array
    {
        if ($this->role?->value === 'superadmin') {
            // If you prefer to read from DB instead, replace with: Module::pluck('slug')->all();
            return ['primary','dental','chronics','secondary','medicals'];
        }

        return $this->modules()
            ->wherePivot('is_enabled', true)
            ->pluck('modules.slug')
            ->all();
    }

    public function hasModuleAccess(string $slug): bool
    {
        if ($this->role?->value === 'superadmin') return true;
        return in_array($slug, $this->enabledModuleSlugs(), true);
    }
}
