<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\MedicationBatch;
use App\Models\Medication;
use App\Models\Supplier;
use Carbon\Carbon;

class MedicationBatchSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run()
    {
        // Get all medications
        $medications = Medication::all();

        // Get all suppliers
        $suppliers = Supplier::all();

        // Ensure there are medications and suppliers
        if ($medications->isEmpty() || $suppliers->isEmpty()) {
            $this->command->info('No medications or suppliers found. Please seed these tables first.');
            return;
        }

        // Define a central store ID (assuming central_store_id 1 exists)
        $centralStoreId = 1;

        // Create batches for each medication
        foreach ($medications as $medication) {
            // Create 3 batches for each medication
            for ($i = 1; $i <= 3; $i++) {
                $batchNumber = $medication->name . '-Batch-' . $i;
                $expiryDate = Carbon::now()->addMonths($i * 6); // Different expiry dates

                // Check if the batch number already exists
                while (MedicationBatch::where('batch_number', $batchNumber)->exists()) {
                    $batchNumber .= '-' . rand(1000, 9999); // Append a random number to make it unique
                }

                MedicationBatch::create([
                    'medication_id' => $medication->id,
                    'central_store_id' => $centralStoreId,
                    'supplier_id' => $suppliers->random()->id, // Random supplier
                    'batch_number' => $batchNumber,
                    'quantity' => rand(10, 100), // Random quantity
                    'expiry_date' => $expiryDate->format('Y-m-d'),
                    'manufacture_date' => Carbon::now()->subMonths($i * 2)->format('Y-m-d'), // Different manufacture dates
                    'received_date' => Carbon::now()->subMonths($i)->format('Y-m-d'), // Different received dates
                ]);
            }
        }

        $this->command->info('Medication batches seeded successfully.');
    }
}
