<?php

namespace Database\Seeders;

use App\Enums\UserRole;
use App\Models\Patient;
use App\Models\Triage;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Carbon;

class TriageSeeder extends Seeder
{
    public function run()
    {
        $patients = Patient::inRandomOrder()->get();
        $nurses = User::where('role', UserRole::Nurse->value)->get();
        $consultors = User::where('role', '!=', UserRole::User->value)->get();

        $triageLevels = ['Emergency', 'Urgent', 'Routine', 'Non-Urgent'];
        $billTypes = [
            'Consultation Adult', 'Consultation Child', 'Follow-up', 'Emergency', 'Vaccination',
            'Lab Test', 'Radiology', 'Review Adult', 'Review Child', 'Antigen Test', 'PCR Test',
            'Rapid Test', 'Vaccine', 'Lead', 'Treatment', 'Other',
        ];
        $billClasses = ['Self', 'Family', 'Parent', 'Extended Family', 'Other'];
        $paymentMethods = ['Payroll', 'Pensioner', 'Cimas', 'Bonvie'];

        foreach ($patients as $patient) {
            $triageCount = rand(1, 3); // You can adjust this to generate more/less entries per patient

            for ($i = 0; $i < $triageCount; $i++) {
                $clinicId = $patient->company?->clinic_id ?? $nurses->first()?->clinic_id;
                $matchingNurses = $nurses->where('clinic_id', $clinicId);

                if ($matchingNurses->isEmpty()) {
                    continue;
                }

                $nurse = $matchingNurses->random();
                $consultor = $consultors->random();

                // Spread created_at from start of the year to now (Zimbabwe timezone)
                $startOfYear = Carbon::create(null, 1, 1, 0, 0, 0, 'Africa/Harare');
                $now = Carbon::now('Africa/Harare');
                $createdAt = $startOfYear->copy()->addSeconds(rand(0, $now->diffInSeconds($startOfYear)));

                Triage::create([
                    'patient_id'      => $patient->id,
                    'clinic_id'       => $nurse->clinic_id,
                    'nurse_id'        => $nurse->id,
                    'consultor_id'    => $consultor->id,
                    'consultation_id' => null,
                    'triage_level'    => $triageLevels[array_rand($triageLevels)],
                    'bill_type'       => $billTypes[array_rand($billTypes)],
                    'bill_class'      => $billClasses[array_rand($billClasses)],
                    'payment_point'   => $patient->company_id,
                    'payment_method'  => $paymentMethods[array_rand($paymentMethods)],
                    'chief_complaint' => $this->generateChiefComplaint(),
                    'notes'           => $this->generateNotes(),
                    'created_at'      => $createdAt,
                    'updated_at'      => $createdAt,
                ]);
            }

            $this->command->info("Created triages for patient ID: {$patient->id}");
        }
    }

    private function generateChiefComplaint(): string
    {
        $complaints = [
            'Headache', 'Fever', 'Cough', 'Chest pain', 'Abdominal pain',
            'Shortness of breath', 'Dizziness', 'Fatigue', 'Joint pain',
            'Back pain', 'Sore throat', 'Nausea', 'Vomiting', 'Diarrhea',
            'Rash', 'Allergic reaction', 'Injury', 'Anxiety', 'Depression',
        ];

        return $complaints[array_rand($complaints)] . ' for ' . rand(1, 14) . ' days';
    }

    private function generateNotes(): string
    {
        $notes = [
            'Patient reports mild symptoms',
            'Symptoms worsening over time',
            'No known allergies',
            'History of similar complaints',
            'Vital signs stable',
            'Referred for further testing',
            'Prescribed medication',
            'Recommended follow-up in 1 week',
            'Patient advised to rest and hydrate',
            'No immediate concerns noted',
        ];

        return $notes[array_rand($notes)];
    }
}
