import React, { useEffect, useMemo, useState } from "react";
import { Card, CardHeader, CardTitle, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from "@/components/ui/table";
import { Button } from "@/components/ui/button";
import { Loader2, CheckCircle, XCircle, Search as SearchIcon, X as XIcon, Calendar } from "lucide-react";
import { Checkbox } from "@/components/ui/checkbox";
import { Input } from "@/components/ui/input";
import { type SharedData } from '@/types';
import { usePage, router } from '@inertiajs/react';
import { API } from "@/config";
import Swal from "sweetalert2";
import moment from "moment";
import axios from "axios";

// recharts (for the blue-ish chart)
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  Tooltip,
  CartesianGrid,
} from "recharts";

interface Patient { id: number; first_name: string; surname: string; }
interface Clinic { id: number; name: string; }
interface Company { id: number; name: string; }
interface Consultor { id: number; name: string; role: string; }

interface Triage {
  id: number;
  patient: Patient;
  clinic: Clinic;
  triage_level: string;
  bill_type: string;
  payment_method: string;
  created_at: string;
  payment_company: Company | null;
  consultor: Consultor | null;
  is_active: boolean;
}

// Stats types
type DayPoint = { date: string; label: string; count: number };
type ClinicCount = { clinic_id: number; clinic_name: string; count: number };
type DiagnosisCount = { name: string; count: number };
type PaymentMethodCount = { payment_method: string; count: number };

type SummaryStats = {
  day: string;
  nurseName: string;
  totalToday: number;
  last7: DayPoint[];
  referrals: {
    radiology: number;
    lab: number;
    specialist: number;
    casualty: number;
    out_patient: number;
    sick_leave: number;
    total_referrals: number;
  };
  iodCount: number;
  perClinic: ClinicCount[];
  diagnoses: DiagnosisCount[];
  paymentMethods: PaymentMethodCount[];
  avgVitals: { temperature: number; systolic: number; diastolic: number };
  allowedClinics: { id: number; name: string }[];
};

export default function ConsultorTriages() {
  const page = usePage<SharedData>();
  const { auth } = page.props;

  const [loading, setLoading] = useState(true);
  const [triages, setTriages] = useState<Triage[]>([]);
  const [closing, setClosing] = useState(false);
  const [selectedIds, setSelectedIds] = useState<Set<number>>(new Set());
  const [query, setQuery] = useState("");

  // NEW: stats state
  const [stats, setStats] = useState<SummaryStats | null>(null);
  const [statsLoading, setStatsLoading] = useState(false);
  const [day, setDay] = useState<string>(new Date().toISOString().slice(0, 10));

  const isSuperadmin = (auth?.user?.role ?? '') === 'superadmin';
  const CONSULTATION_DRAFT_KEY = 'consultation_form_draft_v1';

  const ROLE_STYLES: Record<string, string> = {
    doctor: 'bg-emerald-100 text-emerald-800 dark:bg-emerald-900/40 dark:text-emerald-200',
    nurse: 'bg-sky-100 text-sky-800 dark:bg-sky-900/40 dark:text-sky-200',
    superadmin: 'bg-fuchsia-100 text-fuchsia-800 dark:bg-fuchsia-900/40 dark:text-fuchsia-200',
    receptionist: 'bg-orange-100 text-orange-800 dark:bg-orange-900/40 dark:text-orange-200',
    lab_technician: 'bg-indigo-100 text-indigo-800 dark:bg-indigo-900/40 dark:text-indigo-200',
    radiologist: 'bg-purple-100 text-purple-800 dark:bg-purple-900/40 dark:text-purple-200',
    pharmacist: 'bg-teal-100 text-teal-800 dark:bg-teal-900/40 dark:text-teal-200',
    nurse_aid: 'bg-cyan-100 text-cyan-800 dark:bg-cyan-900/40 dark:text-cyan-200',
    specialist: 'bg-rose-100 text-rose-800 dark:bg-rose-900/40 dark:text-rose-200',
    casualty: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/40 dark:text-yellow-200',
    admin: 'bg-zinc-100 text-zinc-800 dark:bg-zinc-900/40 dark:text-zinc-200',
    user: 'bg-gray-100 text-gray-800 dark:bg-gray-900/40 dark:text-gray-200',
    patient: 'bg-gray-100 text-gray-800 dark:bg-gray-900/40 dark:text-gray-200',
  };
  const getRoleClass = (role?: string) => ROLE_STYLES[role ?? 'user'] ?? ROLE_STYLES.user;
  const initials = (name?: string) => (name ?? '-').trim().split(/\s+/).map((n) => n[0]?.toUpperCase() ?? '').slice(0, 2).join('') || '-';
  const pretty = (role?: string) => (role ? role.replace(/_/g, ' ') : 'N/A');

  async function fetchTriages() {
    setLoading(true);
    try {
      const response = await fetch(`${API}/triages/consultor/${auth.user.id}`);
      const data = await response.json();
      if (data.success) setTriages(data.data);
      else Swal.fire("Error", "Failed to fetch triages.", "error");
    } catch (error) {
      console.error(error);
      Swal.fire("Error", "Failed to fetch triages.", "error");
    } finally {
      setLoading(false);
    }
  }

  // NEW: fetch stats JSON
  async function fetchStats() {
    setStatsLoading(true);
    try {
      const { data } = await axios.get(route('consultor.reports.stats'), {
        params: { date: day },
      });
      setStats(data as SummaryStats);
    } catch (e) {
      setStats(null);
    } finally {
      setStatsLoading(false);
    }
  }

  useEffect(() => { fetchTriages(); }, [auth.user.id]);
  useEffect(() => { fetchStats(); }, [day]);

  const filteredTriages = useMemo(() => {
    const q = query.trim().toLowerCase();
    if (!q) return triages;
    return triages.filter((t) => {
      const patientName = `${t.patient?.first_name ?? ''} ${t.patient?.surname ?? ''}`.toLowerCase();
      const companyName = (t.payment_company?.name ?? '').toLowerCase();
      const consultorName = (t.consultor?.name ?? '').toLowerCase();
      const triageLevel = (t.triage_level ?? '').toLowerCase();
      const billType = (t.bill_type ?? '').toLowerCase();
      const paymentMethod = (t.payment_method ?? '').toLowerCase();
      const dateStr = moment(t.created_at).format('YYYY-MM-DD dddd D MMMM YYYY').toLowerCase();
      return [patientName, companyName, consultorName, triageLevel, billType, paymentMethod, dateStr].some((f) => f.includes(q));
    });
  }, [triages, query]);

  async function handleStartConsultation(triage: Triage) {
    const assignedToId = triage.consultor?.id;
    const assignedToName = triage.consultor?.name ?? '—';
    const isReassignment = !!assignedToId && assignedToId !== auth.user.id;

    const baseTitle = isReassignment ? "Take Over Consultation?" : "Start Consultation?";
    const baseText = isReassignment
      ? `You are starting a consultation that was previously assigned to ${assignedToName}, and this will be recorded as your consultation.`
      : "You are about to start a consultation for this patient.";

    const hasDraft = Boolean(localStorage.getItem(CONSULTATION_DRAFT_KEY));
    const text = hasDraft
      ? `${baseText}\n\nA local consultation draft exists on this device. Starting now will clear that draft.`
      : baseText;

    const confirmButtonText = hasDraft
      ? (isReassignment ? "Yes, take over & clear draft" : "Yes, start & clear draft")
      : (isReassignment ? "Yes, take over" : "Yes, start");

    const res = await Swal.fire({
      title: baseTitle,
      text,
      icon: "question",
      showCancelButton: true,
      confirmButtonText,
      cancelButtonText: "Cancel",
      reverseButtons: true,
      confirmButtonColor: hasDraft ? '#dc2626' : undefined,
    });
    if (!res.isConfirmed) return;

    if (hasDraft) {
      try { localStorage.removeItem(CONSULTATION_DRAFT_KEY); } catch { }
      await Swal.fire({ icon: 'success', title: 'Draft cleared', timer: 900, showConfirmButton: false });
    }

    router.visit(route('consultations.create', { patient_id: triage.patient.id }));
  }

  // selection helpers
  const activeIds = useMemo(() => triages.filter(t => t.is_active).map(t => t.id), [triages]);
  const selectedActiveIds = useMemo(() => Array.from(selectedIds).filter(id => activeIds.includes(id)), [selectedIds, activeIds]);
  const allActiveSelected = selectedActiveIds.length > 0 && selectedActiveIds.length === activeIds.length;
  const someActiveSelected = selectedActiveIds.length > 0 && selectedActiveIds.length < activeIds.length;

  const toggleRow = (id: number, enabled: boolean) => {
    setSelectedIds(prev => {
      const next = new Set(prev);
      if (enabled) next.add(id); else next.delete(id);
      return next;
    });
  };

  const toggleAll = (checked: boolean | "indeterminate") => {
    setSelectedIds(prev => {
      if (checked) {
        return new Set([...prev, ...activeIds]);
      } else {
        const next = new Set(prev);
        activeIds.forEach(id => next.delete(id));
        return next;
      }
    });
  };

  const closeSelected = async () => {
    const ids = selectedActiveIds;
    if (ids.length === 0) return;

    const { isConfirmed } = await Swal.fire({
      title: "Close selected triage files?",
      text: `You are about to close ${ids.length} active triage file(s).`,
      icon: "warning",
      showCancelButton: true,
      confirmButtonText: "Yes, close",
      cancelButtonText: "Cancel",
      reverseButtons: true,
    });
    if (!isConfirmed) return;

    try {
      setClosing(true);
      const res = await fetch(`${API}/triages/close`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({ ids }),
      });
      const json = await res.json();
      if (!res.ok || !json.success) throw new Error(json?.message || 'Failed to close triages');

      const closedIds: number[] = (json.closed_ids ?? []) as number[];
      if (closedIds.length > 0) {
        const closedSet = new Set(closedIds);
        setTriages(prev => prev.map(t => (closedSet.has(t.id) ? { ...t, is_active: false } : t)));
        setSelectedIds(prev => {
          const next = new Set(prev);
          closedIds.forEach(id => next.delete(id));
          return next;
        });
      }
      Swal.fire("Done", json.message || "Selected triages closed.", "success");
    } catch (e: any) {
      console.error(e);
      Swal.fire("Error", e?.message || "Failed to close triages.", "error");
    } finally {
      setClosing(false);
    }
  };

  return (
    <Card className="p-4">
      {/* ===== NEW: STATS PANEL ===== */}
      <div className="mb-6">
        <div className="mb-3 flex flex-wrap items-start justify-between gap-3">
          <div>
            <div className="text-xl font-semibold">Your Activity</div>
            <div className="text-xs text-muted-foreground">
              Snapshot for <b>{moment(day).format('ddd, D MMM YYYY')}</b>. The chart shows the last 7 days ending on this date.
            </div>
          </div>

          <div className="flex flex-col items-end gap-2">
            <label htmlFor="stats-date" className="text-sm font-medium">Report Date</label>
            <div className="relative">
              <Calendar className="pointer-events-none absolute left-3 top-1/2 h-5 w-5 -translate-y-1/2 text-muted-foreground" />
              <input
                id="stats-date"
                type="date"
                value={day}
                onChange={(e) => setDay(e.target.value)}
                className="h-11 w-[220px] rounded-md border px-10 text-base"
              />
            </div>
          </div>
        </div>

        {/* Cards Row */}
        <div className="grid gap-4 sm:grid-cols-12">
          <Card className="sm:col-span-3">
            <CardHeader><CardTitle>Consultations Today</CardTitle></CardHeader>
            <CardContent>
              <div className="text-4xl font-bold">{statsLoading ? '—' : (stats?.totalToday ?? 0)}</div>
              <div className="mt-2 text-xs text-muted-foreground">Across your allowed clinics.</div>
            </CardContent>
          </Card>

          <Card className="sm:col-span-5">
            <CardHeader><CardTitle>Referrals & Sick Leave (All Time)</CardTitle></CardHeader>
            <CardContent>
              {statsLoading || !stats ? (
                <div className="text-sm text-muted-foreground">Loading…</div>
              ) : (
                <>
                  <div className="grid grid-cols-2 gap-3 text-sm">
                    <div className="flex items-center justify-between"><span>Radiology</span><Badge variant="secondary">{stats.referrals.radiology}</Badge></div>
                    <div className="flex items-center justify-between"><span>Lab</span><Badge variant="secondary">{stats.referrals.lab}</Badge></div>
                    <div className="flex items-center justify-between"><span>Specialist</span><Badge variant="secondary">{stats.referrals.specialist}</Badge></div>
                    <div className="flex items-center justify-between"><span>Casualty</span><Badge variant="secondary">{stats.referrals.casualty}</Badge></div>
                    <div className="flex items-center justify-between"><span>Out-patient</span><Badge variant="secondary">{stats.referrals.out_patient}</Badge></div>
                    <div className="flex items-center justify-between"><span>Sick Leave</span><Badge className="bg-amber-600 text-white hover:bg-amber-600">{stats.referrals.sick_leave}</Badge></div>
                  </div>
                  <div className="mt-4 flex items-center justify-between rounded-md border bg-muted/40 p-3 text-sm">
                    <span className="font-medium">Total Referrals</span>
                    <Badge className="bg-blue-600 hover:bg-blue-600">{stats.referrals.total_referrals}</Badge>
                  </div>
                </>
              )}
            </CardContent>
          </Card>

          <Card className="sm:col-span-4">
            <CardHeader><CardTitle>Injury On Duty (All Time)</CardTitle></CardHeader>
            <CardContent>
              <div className="text-4xl font-bold">{statsLoading ? '—' : (stats?.iodCount ?? 0)}</div>
              <div className="mt-2 text-xs text-muted-foreground">Recorded by you.</div>
            </CardContent>
          </Card>
        </div>

        {/* Chart + Per-clinic */}
        <div className="mt-4 grid gap-4 sm:grid-cols-12">
          <Card className="sm:col-span-8">
            <CardHeader><CardTitle>Consultations — Last 7 Days</CardTitle></CardHeader>
            <CardContent style={{ height: 300 }}>
              {statsLoading || !stats ? (
                <div className="text-sm text-muted-foreground">Loading…</div>
              ) : (
                <ResponsiveContainer width="100%" height="100%">
                  <BarChart data={stats.last7}>
                    <defs>
                      <linearGradient id="blueFade" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="0%" stopColor="#60a5fa" stopOpacity={0.95} />
                        <stop offset="100%" stopColor="#3b82f6" stopOpacity={0.85} />
                      </linearGradient>
                    </defs>
                    <CartesianGrid stroke="rgba(59,130,246,0.15)" strokeDasharray="3 3" />
                    <XAxis dataKey="label" tick={{ fill: '#1e40af' }} axisLine={{ stroke: 'rgba(59,130,246,0.35)' }} tickLine={{ stroke: 'rgba(59,130,246,0.35)' }} />
                    <YAxis allowDecimals={false} tick={{ fill: '#1e40af' }} axisLine={{ stroke: 'rgba(59,130,246,0.35)' }} tickLine={{ stroke: 'rgba(59,130,246,0.35)' }} />
                    <Tooltip contentStyle={{ borderColor: 'rgba(59,130,246,0.35)' }} />
                    <Bar dataKey="count" fill="url(#blueFade)" stroke="#1d4ed8" strokeWidth={1} radius={[6, 6, 0, 0]} />
                  </BarChart>
                </ResponsiveContainer>
              )}
            </CardContent>
          </Card>

          <Card className="sm:col-span-4">
            <CardHeader><CardTitle>Consulted Today by Clinic</CardTitle></CardHeader>
            <CardContent>
              {statsLoading || !stats ? (
                <div className="text-sm text-muted-foreground">Loading…</div>
              ) : (
                <div className="space-y-2">
                  {stats.perClinic.length === 0 ? (
                    <div className="text-sm text-muted-foreground">No consultations recorded today.</div>
                  ) : (
                    stats.perClinic.map((c) => (
                      <div key={c.clinic_id} className="flex items-center justify-between">
                        <Badge className="border border-blue-200 bg-blue-50 text-blue-700 hover:bg-blue-50">
                          {c.clinic_name}
                        </Badge>
                        <span className="font-semibold">{c.count}</span>
                      </div>
                    ))
                  )}
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
      {/* ===== END STATS PANEL ===== */}

      <CardHeader className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
        <CardTitle>Patient Pending Files</CardTitle>

        <div className="flex w-full flex-col gap-2 sm:w-auto sm:flex-row sm:items-center">
          {/* Search box */}
          <div className="relative w-full sm:w-80">
            <SearchIcon className="pointer-events-none absolute left-2 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
            <Input
              value={query}
              onChange={(e) => setQuery(e.target.value)}
              placeholder="Search patient, company, consultor..."
              className="pl-8"
              aria-label="Search triages"
            />
            {query && (
              <button
                type="button"
                onClick={() => setQuery("")}
                className="absolute right-2 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                aria-label="Clear search"
                style={{ cursor: "pointer" }}
              >
                <XIcon className="h-4 w-4" />
              </button>
            )}
          </div>

          <div className="flex items-center gap-2">
            <Badge variant="outline" className="whitespace-nowrap">
              Showing {filteredTriages.length} / {triages.length}
            </Badge>

            {isSuperadmin && (
              <Button
                variant="destructive"
                size="sm"
                onClick={closeSelected}
                disabled={closing || selectedActiveIds.length === 0}
                style={{ cursor: "pointer" }}
              >
                {closing ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" /> Closing…
                  </>
                ) : (
                  <>Close Selected ({selectedActiveIds.length})</>
                )}
              </Button>
            )}

            <Button
              variant="default"
              size="sm"
              onClick={fetchTriages}
              disabled={loading}
              style={{ cursor: "pointer" }}
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" /> Loading...
                </>
              ) : (
                "Get Pending Files"
              )}
            </Button>
          </div>
        </div>
      </CardHeader>

      <CardContent>
        <div className="mb-4">
          <h3 className="text-base font-semibold text-gray-900 dark:text-gray-100">Your Tasks</h3>
          <div className="mt-2 rounded-md border border-yellow-300 bg-yellow-50 p-3 text-sm text-yellow-900 dark:border-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-100">
            <strong>NB:</strong> If you start a consultation on a file assigned to another consultor, it will be recorded as <em>your</em> consultation and reflected in the audit log.
          </div>
        </div>

        {loading ? (
          <div className="flex items-center justify-center p-10">
            <Loader2 className="h-6 w-6 animate-spin" />
          </div>
        ) : triages.length === 0 ? (
          <p className="text-center text-muted-foreground">No pending files found for this consultor.</p>
        ) : filteredTriages.length === 0 ? (
          <p className="text-center text-muted-foreground">No matches for “{query}”.</p>
        ) : (
          <Table>
            <TableHeader>
              <TableRow>
                {isSuperadmin && (
                  <TableHead className="w-[48px]">
                    <div className="flex items-center">
                      <Checkbox
                        checked={allActiveSelected ? true : (someActiveSelected ? "indeterminate" : false)}
                        onCheckedChange={(v) => toggleAll(v)}
                        aria-label="Select all active"
                      />
                    </div>
                  </TableHead>
                )}
                <TableHead>Patient</TableHead>
                <TableHead>Company</TableHead>
                <TableHead>Triage Level</TableHead>
                <TableHead>Bill Type</TableHead>
                <TableHead>Payment Method</TableHead>
                <TableHead>Assigned Consultor</TableHead>
                <TableHead>Date</TableHead>
                <TableHead>State</TableHead>
                <TableHead>Action</TableHead>
              </TableRow>
            </TableHeader>

            <TableBody>
              {filteredTriages.map((triage) => {
                const isRowSelected = selectedIds.has(triage.id);
                const canSelect = isSuperadmin && triage.is_active;

                return (
                  <TableRow key={triage.id} className={isRowSelected ? "bg-muted/40" : ""}>
                    {isSuperadmin && (
                      <TableCell>
                        <Checkbox
                          checked={isRowSelected}
                          onCheckedChange={(v) => toggleRow(triage.id, Boolean(v))}
                          disabled={!canSelect}
                          aria-label={`Select triage ${triage.id}`}
                        />
                      </TableCell>
                    )}

                    <TableCell>
                      {triage.patient
                        ? `${triage.patient.first_name} ${triage.patient.surname}`
                        : "Unknown"}
                    </TableCell>

                    <TableCell>
                      {triage.payment_company?.name ? (
                        <Badge variant="secondary">{triage.payment_company.name}</Badge>
                      ) : (
                        <span className="text-muted-foreground">N/A</span>
                      )}
                    </TableCell>

                    <TableCell>{triage.triage_level || "-"}</TableCell>
                    <TableCell>{triage.bill_type || "-"}</TableCell>
                    <TableCell>{triage.payment_method || "-"}</TableCell>

                    <TableCell>
                      <div className="flex items-center gap-3 max-w-[280px]">
                        <div
                          className="flex h-8 w-8 shrink-0 items-center justify-center rounded-full bg-gray-200 text-xs font-semibold text-gray-700
                                     dark:bg-gray-800 dark:text-gray-200"
                          title={triage.consultor?.name || '-'}
                          aria-label={triage.consultor?.name || 'No name'}
                        >
                          {initials(triage.consultor?.name)}
                        </div>

                        <div className="min-w-0">
                          <div className="truncate font-medium text-gray-900 dark:text-gray-100" title={triage.consultor?.name || '-'}>
                            {triage.consultor?.name || '-'}
                          </div>

                          <Badge
                            variant="secondary"
                            className={`mt-1 capitalize ${getRoleClass(triage.consultor?.role)}`}
                            title={pretty(triage.consultor?.role)}
                          >
                            {pretty(triage.consultor?.role)}
                          </Badge>
                        </div>
                      </div>
                    </TableCell>

                    <TableCell>{moment(triage.created_at).format("dddd D MMMM YYYY")}</TableCell>

                    <TableCell>
                      {triage.is_active ? (
                        <Badge className="bg-green-600">
                          <CheckCircle className="mr-1 h-3.5 w-3.5" />
                          Active
                        </Badge>
                      ) : (
                        <Badge variant="outline" className="border-gray-300 text-gray-700">
                          <XCircle className="mr-1 h-3.5 w-3.5" />
                          Inactive
                        </Badge>
                      )}
                    </TableCell>

                    <TableCell>
                      {triage.patient ? (
                        <Button
                          variant="default"
                          size="sm"
                          onClick={() => handleStartConsultation(triage)}
                          style={{ cursor: "pointer" }}
                        >
                          Start Consultation
                        </Button>
                      ) : (
                        <span className="text-muted-foreground">N/A</span>
                      )}
                    </TableCell>
                  </TableRow>
                );
              })}
            </TableBody>
          </Table>
        )}

        <div className="mt-4 flex justify-end">
          <Button variant="secondary" onClick={() => history.back()}>
            Back
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}
