import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface PatientFormProps {
    data: any;
    errors: any;
    onFieldChange: (field: string, value: any) => void;
    isDependent?: boolean;
}

interface AddressInfoSectionProps {
    data: {
        home_address: string;
        work_area: string;
        suburb: string;
    };
    errors: {
        home_address?: string;
        work_area?: string;
        suburb?: string;
    };
    onFieldChange: (field: string, value: string) => void;
}

export function PersonalInfoSection({ data, errors, onFieldChange }: PatientFormProps) {
    return (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 md:grid-cols-3">
            <div className="space-y-2">
                <Label htmlFor="first_name">First Name*</Label>
                <Input id="first_name" value={data.first_name} onChange={(e) => onFieldChange('first_name', e.target.value)} />
                {errors.first_name && <p className="text-sm text-red-500">{errors.first_name}</p>}
            </div>

            <div className="space-y-2">
                <Label htmlFor="middle_name">Middle Name</Label>
                <Input id="middle_name" value={data.middle_name || ''} onChange={(e) => onFieldChange('middle_name', e.target.value)} />
            </div>

            <div className="space-y-2">
                <Label htmlFor="surname">Surname*</Label>
                <Input id="surname" value={data.surname} onChange={(e) => onFieldChange('surname', e.target.value)} />
                {errors.surname && <p className="text-sm text-red-500">{errors.surname}</p>}
            </div>

            <div className="space-y-2">
                <Label htmlFor="id_number">ID Number/Passport*</Label>
                <Input id="id_number" value={data.id_number || ''} onChange={(e) => onFieldChange('id_number', e.target.value)} />
                {errors.id_number && <p className="text-sm text-red-500">{errors.id_number}</p>}
            </div>

            <div className="space-y-2">
                <Label htmlFor="date_of_birth">Date of Birth</Label>
                <Input
                    id="date_of_birth"
                    type="date"
                    value={data.date_of_birth || ''}
                    onChange={(e) => onFieldChange('date_of_birth', e.target.value)}
                />
            </div>

            <div className="space-y-2">
                <Label htmlFor="gender">Gender</Label>
                <Select value={data.gender || ''} onValueChange={(value) => onFieldChange('gender', value)}>
                    <SelectTrigger>
                        <SelectValue placeholder="Select gender" />
                    </SelectTrigger>
                    <SelectContent>
                        <SelectItem value="Male">Male</SelectItem>
                        <SelectItem value="Female">Female</SelectItem>
                        <SelectItem value="Other">Other</SelectItem>
                    </SelectContent>
                </Select>
            </div>
        </div>
    );
}

export function ContactInfoSection({ data, errors, onFieldChange }: PatientFormProps) {
    return (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
            <div className="space-y-2">
                <Label htmlFor="email">Email</Label>
                <Input id="email" type="email" value={data.email || ''} onChange={(e) => onFieldChange('email', e.target.value)} />
            </div>

            <div className="space-y-2">
                <Label htmlFor="phone">Phone*</Label>
                <Input id="phone" value={data.phone || ''} onChange={(e) => onFieldChange('phone', e.target.value)} />
                {errors.phone && <p className="text-sm text-red-500">{errors.phone}</p>}
            </div>

            <div className="space-y-2">
                <Label htmlFor="marital_status">Marital Status</Label>
                <Select value={data.marital_status || ''} onValueChange={(value) => onFieldChange('marital_status', value)}>
                    <SelectTrigger>
                        <SelectValue placeholder="Select status" />
                    </SelectTrigger>
                    <SelectContent>
                        <SelectItem value="Single">Single</SelectItem>
                        <SelectItem value="Married">Married</SelectItem>
                        <SelectItem value="Divorced">Divorced</SelectItem>
                        <SelectItem value="Widowed">Widowed</SelectItem>
                    </SelectContent>
                </Select>
            </div>

            <div className="space-y-2">
                <Label htmlFor="occupation">Occupation</Label>
                <Input id="occupation" value={data.occupation || ''} onChange={(e) => onFieldChange('occupation', e.target.value)} />
            </div>
        </div>
    );
}

export function MedicalInfoSection({ data, onFieldChange }: PatientFormProps) {
    return (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
            <div className="space-y-2">
                <Label htmlFor="medical_aid_number">Medical Aid Number</Label>
                <Input
                    id="medical_aid_number"
                    value={data.medical_aid_number || ''}
                    onChange={(e) => onFieldChange('medical_aid_number', e.target.value)}
                />
            </div>

            <div className="space-y-2">
                <Label htmlFor="medical_aid_provider">Medical Aid Provider</Label>
                <select
                    id="medical_aid_provider"
                    value={data.medical_aid_provider || ''}
                    onChange={(e) => onFieldChange('medical_aid_provider', e.target.value)}
                    className="border-border bg-background text-foreground w-full rounded border px-3 py-2 text-sm dark:border-neutral-700 dark:bg-neutral-900 dark:text-white"
                >
                    <option value="">Select Provider</option>
                    <option value="PSMAS">PSMAS</option>
                    <option value="CIMAS">CIMAS</option>
                    <option value="First Mutual Health">First Mutual Health</option>
                    <option value="ZIMRE Health">ZIMRE Health</option>
                    <option value="Generation Health">Generation Health</option>
                    <option value="Masca">Masca</option>
                    <option value="Health International">Health International</option>
                    <option value="Premier Service Medical Aid">Premier Service Medical Aid</option>
                    <option value="Altfin Medical Aid">Altfin Medical Aid</option>
                    <option value="Cellmed Health">Cellmed Health</option>
                    <option value="CORHealth">CORHealth</option>
                    <option value="Sanctuary Health">Sanctuary Health</option>
                    <option value="Falcon Health">Falcon Health</option>
                    <option value="Econet Health">Econet Health</option>
                    <option value="Bonvie">Bonvie</option>
                    <option value="EMF">EMF</option>


                    <option value="Alliance Health">Alliance Health</option>
                    <option value="AGRIMED">AGRIMED (Agricultural Medical Aid Society)</option>
                    <option value="Corp24 Medical Aid">Corp24 Medical Aid</option>
                    <option value="FA-MAS">FA-MAS</option>
                    <option value="FLIMAS">FLIMAS (Fidelity Life Medical Aid Society)</option>
                    <option value="Maisha Health Fund">Maisha Health Fund</option>
                    <option value="Nemas">Nemas (National Employment Medical Aid Society)</option>
                    <option value="NSSA">NSSA (National Social Security Authority Health)</option>
                    <option value="Northern Medical Aid Society (NMAS)">Northern Medical Aid Society (NMAS)</option>
                    <option value="Parksmed Health Fund">Parksmed Health Fund</option>
                    <option value="Pro-Health Medical Aid Society">Pro-Health Medical Aid Society</option>
                    <option value="Quest Vitality Medical">Quest Vitality Medical</option>
                    <option value="Salutem International Medical Fund">Salutem International Medical Fund</option>
                    <option value="SaveLife Medical Aid Fund">SaveLife Medical Aid Fund</option>
                    <option value="Ultra-Med Health Care">Ultra-Med Health Care</option>

                    <option value="Other">Other</option>
                </select>
            </div>

            <div className="space-y-2">
                <Label htmlFor="allergies">Allergies</Label>
                <textarea
                    id="allergies"
                    value={data.allergies || ''}
                    onChange={(e) => onFieldChange('allergies', e.target.value)}
                    className="w-full rounded-md border border-gray-300 px-3 py-2 text-sm shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none"
                />
            </div>

            <div className="flex items-center space-x-2 pt-6">
                <input
                    id="is_smoker"
                    type="checkbox"
                    checked={data.is_smoker || false}
                    onChange={(e) => onFieldChange('is_smoker', e.target.checked)}
                    className="h-5 w-5 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                />
                <Label htmlFor="is_smoker">Smoker</Label>
            </div>
        </div>
    );
}

export function AddressInfoSection({ data, errors, onFieldChange }: AddressInfoSectionProps) {
    return (
        <div>
            <div className="space-y-2">
                <Label htmlFor="home_address">Home Address</Label>
                <Input
                    id="home_address"
                    type="text"
                    value={data.home_address}
                    onChange={(e) => onFieldChange('home_address', e.target.value)}
                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-black dark:text-white"
                />
                {errors.home_address && <p className="text-sm text-red-500">{errors.home_address}</p>}
            </div>

            <div className="space-y-2">
                <Label htmlFor="work_area">Work Area</Label>
                <Input
                    id="work_area"
                    type="text"
                    value={data.work_area}
                    onChange={(e) => onFieldChange('work_area', e.target.value)}
                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-black dark:text-white"
                />
                {errors.work_area && <p className="text-sm text-red-500">{errors.work_area}</p>}
            </div>

            <div className="space-y-2">
                <Label htmlFor="suburb">Suburb</Label>
                <Input
                    id="suburb"
                    type="text"
                    value={data.suburb}
                    onChange={(e) => onFieldChange('suburb', e.target.value)}
                    className="w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-900 shadow-sm focus:border-blue-500 focus:ring-1 focus:ring-blue-500 focus:outline-none dark:border-gray-700 dark:bg-black dark:text-white"
                />
                {errors.suburb && <p className="text-sm text-red-500">{errors.suburb}</p>}
            </div>
        </div>
    );
}

export function EmergencyContactSection({ data, onFieldChange }: PatientFormProps) {
    return (
        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
            <div className="space-y-2">
                <Label htmlFor="emergency_contact_name">Emergency Contact Name (Next of Kin)</Label>
                <Input
                    id="emergency_contact_name"
                    value={data.emergency_contact_name || ''}
                    onChange={(e) => onFieldChange('emergency_contact_name', e.target.value)}
                />
            </div>

            <div className="space-y-2">
                <Label htmlFor="emergency_contact_relation">Relationship</Label>
                <Input
                    id="emergency_contact_relation"
                    value={data.emergency_contact_relation || ''}
                    onChange={(e) => onFieldChange('emergency_contact_relation', e.target.value)}
                />
            </div>

            <div className="space-y-2">
                <Label htmlFor="emergency_contact_phone">Emergency Phone</Label>
                <Input
                    id="emergency_contact_phone"
                    value={data.emergency_contact_phone || ''}
                    onChange={(e) => onFieldChange('emergency_contact_phone', e.target.value)}
                />
            </div>
        </div>
    );
}
