import { Link } from "@inertiajs/react"
import { TrendingUp, TrendingDown, Users, UserSquare2, Sigma } from "lucide-react"

import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import {
  Card,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card"
import { API } from "@/config"

type Stats = {
  employees?: number | null
  dependents?: number | null
  total?: number | null
}

type Props = {
  stats?: Stats
  prev?: Partial<Stats>
}

const fmt = (n?: number | null) =>
  typeof n === "number" ? new Intl.NumberFormat().format(n) : "—"

const pct = (curr?: number | null, prev?: number | null) => {
  if (typeof curr !== "number" || typeof prev !== "number" || prev === 0) return null
  return Math.round(((curr - prev) / prev) * 1000) / 10
}

function DeltaBadge({ value }: { value: number | null }) {
  if (value == null) return null
  const up = value >= 0
  const Icon = up ? TrendingUp : TrendingDown
  return (
    <Badge variant="outline" className="gap-1">
      <Icon className="size-4" />
      {up ? "+" : ""}
      {value}%
    </Badge>
  )
}

/** Small helper so the whole card is a link */
function ClickableCard({
  href,
  ariaLabel,
  children,
}: {
  href: string
  ariaLabel?: string
  children: React.ReactNode
}) {
  return (
    <Link href={href} aria-label={ariaLabel} className="block group">
      <Card
        className="
          cursor-pointer transition
          hover:shadow-md hover:border-primary/40
          focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-primary
        "
      >
        {children}
      </Card>
    </Link>
  )
}

export function SectionCards({ stats, prev }: Props) {
  const dEmployees = pct(stats?.employees, prev?.employees)
  const dDependents = pct(stats?.dependents, prev?.dependents)
  const dTotal = pct(stats?.total, prev?.total)

  return (
    <div
      className="
        col-span-full w-full min-w-0
        grid grid-cols-1 gap-4
        sm:grid-cols-2 lg:grid-cols-4
      "
    >
      {/* Employees (clickable -> /patients) */}
      <ClickableCard href={`${API}/patients`} ariaLabel="Open Patients">
        <CardHeader className="space-y-1">
          <div className="flex items-center justify-between">
            <CardDescription className="flex items-center gap-2">
              <Users className="size-4 text-muted-foreground" />
              <span className="underline decoration-dotted underline-offset-4 group-hover:text-primary">
                Account Holders (Database)
              </span>
            </CardDescription>
            <DeltaBadge value={dEmployees} />
          </div>
          <CardTitle className="text-2xl font-semibold tabular-nums @[250px]/card:text-3xl">
            {fmt(stats?.employees)}
          </CardTitle>
        </CardHeader>
        <CardFooter className="flex-col items-start gap-1.5 text-sm">
          <div className="font-medium">Active employee patients on record</div>
          <div className="text-muted-foreground">Company staff registered across clinics</div>
        </CardFooter>
      </ClickableCard>

      {/* Dependents (clickable -> /dependents) */}
      <ClickableCard href={`${API}/dependents`} ariaLabel="Open Dependents">
        <CardHeader className="space-y-1">
          <div className="flex items-center justify-between">
            <CardDescription className="flex items-center gap-2">
              <UserSquare2 className="size-4 text-muted-foreground" />
              <span className="underline decoration-dotted underline-offset-4 group-hover:text-primary">
                Dependents
              </span>
            </CardDescription>
            <DeltaBadge value={dDependents} />
          </div>
          <CardTitle className="text-2xl font-semibold tabular-nums @[250px]/card:text-3xl">
            {fmt(stats?.dependents)}
          </CardTitle>
        </CardHeader>
        <CardFooter className="flex-col items-start gap-1.5 text-sm">
          <div className="font-medium">Linked family members & beneficiaries</div>
          <div className="text-muted-foreground">Associated to employee profiles</div>
        </CardFooter>
      </ClickableCard>

      {/* Total Patients (not linked) */}
      <Card className="@container/card">
        <CardHeader className="space-y-1">
          <div className="flex items-center justify-between">
            <CardDescription className="flex items-center gap-2">
              <Sigma className="size-4 text-muted-foreground" />
              Total of Account Holders & Dependents
            </CardDescription>
            <DeltaBadge value={dTotal} />
          </div>
          <CardTitle className="text-2xl font-semibold tabular-nums @[250px]/card:text-3xl">
            {fmt(stats?.total)}
          </CardTitle>
        </CardHeader>
        <CardFooter className="flex-col items-start gap-1.5 text-sm">
          <div className="font-medium">Employees + dependents currently managed</div>
          <div className="text-muted-foreground">Unique individuals across all clinics</div>
        </CardFooter>
      </Card>

      {/* Reports Portal (unchanged) */}
      <Card className="@container/card">
        <CardHeader className="space-y-1">
          <CardDescription>Reports Portal</CardDescription>
          <CardTitle className="text-2xl font-semibold tabular-nums @[250px]/card:text-3xl">
            Generate Reports
          </CardTitle>
        </CardHeader>
        <CardFooter className="flex items-center justify-between gap-2">
          <div className="text-sm text-muted-foreground">
            Create Excel exports for consultations, patients, and inventory.
          </div>
          <Button asChild size="sm">
            <Link href={`${API}/reports/portal`}>Open</Link>
          </Button>
        </CardFooter>
      </Card>
    </div>
  )
}
