import * as React from 'react';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';
import { useReactToPrint } from 'react-to-print';
import { Button } from '@/components/ui/button';
import { Printer } from 'lucide-react';

const logoUrl = 'https://providence-human-capital.github.io/images/PHC_Logo_Health.png';
const SignatureUrl = 'http://127.0.0.1:8001';

export type MinimalPatient = {
  id: number | string;
  first_name?: string | null;
  surname?: string | null;
  gender?: string | null;
  id_number?: string | null;
};

export type MinimalClinic = {
  name?: string | null;
  address?: string | null;
  contact_person?: string | null;
  email_address?: string | null;
};

export type MinimalDoctor = {
  name?: string | null;
  signature?: string | null;
};

export type MinimalConsultation = {
  id: number | string;
  consultation_date?: string | null;
  created_at?: string | null; // source of truth for attendance timestamp
  patient?: MinimalPatient | null;
  clinic?: MinimalClinic | null;
  doctor?: MinimalDoctor | null;
};

interface Props {
  consultation?: MinimalConsultation | null;
  patient?: MinimalPatient | null;

  label?: string;
  buttonVariant?: 'default' | 'secondary' | 'outline' | 'destructive' | 'ghost' | 'link';
  className?: string;
}

export default function PrintAttendanceSlip({
  consultation,
  patient,
  label = 'Print Attendance Slip',
  buttonVariant = 'outline',
  className,
}: Props) {
  const contentRef = React.useRef<HTMLDivElement>(null);
  const handlePrint = useReactToPrint({ contentRef });

  const pat: MinimalPatient | undefined = patient ?? consultation?.patient ?? undefined;
  const doc: MinimalDoctor | undefined = consultation?.doctor ?? undefined;
  const clinic: MinimalClinic | undefined = consultation?.clinic ?? undefined;

  const patientName = [pat?.first_name, pat?.surname].filter(Boolean).join(' ');

  // Attendance timestamp: use created_at (fallback to "now" if missing)
  const attendedAt = consultation?.created_at ? moment(consultation.created_at) : moment();
  const attendedAtFull = attendedAt.format('MMMM Do YYYY, h:mm A');

  return (
    <>
      {/* Hidden printable content */}
      <div style={{ display: 'none' }}>
        <div ref={contentRef} className="print-page p-6 font-sans text-sm text-gray-800">
          <style>
            {`
              @page {
                size: A4;
                margin: 15mm 10mm; /* top/bottom 15mm, left/right 10mm */
              }
              @media print {
                body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                .section-break { page-break-after: always; }
                .avoid-break { page-break-inside: avoid; }

                /* Grid to pin footer to bottom of printable area */
                .print-page {
                  display: grid;
                  grid-template-rows: auto 1fr auto; /* header / main / footer */
                  height: calc(297mm - 30mm); /* A4 height minus vertical margins */
                }
              }
              @media screen {
                .print-page {
                  display: grid;
                  grid-template-rows: auto 1fr auto;
                  min-height: 100vh;
                }
              }
            `}
          </style>

          {/* HEADER */}
          <header>
            <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
              <div className="flex items-center">
                <img src={logoUrl} alt="PHC Logo" style={{ height: '4rem' }} />
              </div>
              <div className="text-center text-sm text-gray-800">
                <p className="text-lg font-bold">Clinic Attendance</p>
                <p>{attendedAtFull}</p>
              </div>
              <div className="flex items-center">
                <QRCodeSVG
                  value={`Consultation:${consultation?.id ?? ''}; Patient:${pat?.id ?? ''}`}
                  size={60}
                />
              </div>
            </div>
          </header>

          {/* MAIN */}
          <main className="pt-4">
            {/* Clinic */}
            <div className="mt-0 flex justify-between" style={{ fontSize: '11px' }}>
              <div className="flex-1"></div>
              <div className="flex-1 text-right">
                <h2 className="text-lg font-bold">Clinic Information</h2>
                <p>{clinic?.name || '-'}</p>
                <p>Clinic Address: {clinic?.address || '-'}</p>
                <p>Contact Person: {clinic?.contact_person || '-'}</p>
                <p>Email Address: {clinic?.email_address || '-'}</p>
              </div>
            </div>

            {/* Patient */}
            <div className="mt-4" style={{ fontSize: '11px' }}>
              <h2 className="text-lg font-bold">Patient Information</h2>
              <p>Patient Name: {patientName || '-'}</p>
              <p>Patient Gender: {pat?.gender || '-'}</p>
              <p>
                ID Number:{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {pat?.id_number || '-'}
                </span>
              </p>
            </div>

            {/* Body (short, corrected English) */}
            <div className="mt-4">
              <p>
                This is to certify that{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {patientName || '-'}
                </span>{' '}
                (National ID:{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {pat?.id_number || '-'}
                </span>
                ) attended our health facility,{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {clinic?.name || '-'}
                </span>
                , on{' '}
                <span style={{ fontWeight: 'bold', color: '#000080' }}>
                  {attendedAtFull}
                </span>
                .
              </p>
            </div>

            {/* Doctor */}
            <div className="mt-16">
              <h2 className="text-lg font-bold">Consultor&apos;s Information</h2>
              <p>Doctor&apos;s / RGN&apos;s Name: {doc?.name || '-'}</p>
              <p>Doctor&apos;s Signature</p>
              <img
                src={
                  doc?.signature
                    ? `${SignatureUrl}/storage/${doc.signature}`
                    : '/placeholder-signature.png'
                }
                alt="Doctor Signature"
                style={{ height: '4rem' }}
              />
            </div>
          </main>

          {/* FOOTER (sticks to bottom of page) */}
          <footer className="pt-2">
            <div className="flex justify-between">
              <div style={{ fontSize: '11px' }}>
                <p>Date of Print: {moment().format('MMMM Do YYYY, h:mm A')}</p>
              </div>
              <div className="h-24 w-34 border-2 border-dashed border-gray-300" />
            </div>
          </footer>
        </div>
      </div>

      {/* Trigger button */}
      <Button
        variant={buttonVariant}
        onClick={handlePrint}
        className={className}
        style={{ cursor: 'pointer' }}
      >
        <Printer className="mr-2 h-4 w-4" />
        {label}
      </Button>
    </>
  );
}
