import { Breadcrumbs } from '@/components/breadcrumbs';
import { Icon } from '@/components/icon';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Button } from '@/components/ui/button';
import { DropdownMenu, DropdownMenuContent, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { NavigationMenu, NavigationMenuItem, NavigationMenuList, navigationMenuTriggerStyle } from '@/components/ui/navigation-menu';
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetTrigger } from '@/components/ui/sheet';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { UserMenuContent } from '@/components/user-menu-content';
import { API } from '@/config';
import { useInitials } from '@/hooks/use-initials';
import { cn } from '@/lib/utils';
import { type BreadcrumbItem, type NavItem, type SharedData } from '@/types';
import { Link, usePage } from '@inertiajs/react';
import {
    Activity,
    Building,
    ChartArea,
    ChartSpline,
    ClipboardPlus,
    Cross,
    Hospital,
    LayoutGrid,
    Menu,
    Pill,
    PillBottle,
    Search,
    Tablets,
    User,
    User2,
    Warehouse,
} from 'lucide-react';
import AppLogo from './app-logo';
import AppLogoIcon from './app-logo-icon';

const activeItemStyles = 'text-neutral-900 dark:bg-neutral-800 dark:text-neutral-100';

interface AppHeaderProps {
    breadcrumbs?: BreadcrumbItem[];
}

export function AppHeader({ breadcrumbs = [] }: AppHeaderProps) {
    const page = usePage<SharedData>();
    const { auth } = page.props;
    const userRole = auth.user?.role; // Make sure this matches your role field name
    const getInitials = useInitials();

    // Define all possible navigation items
    const allNavItems = {
        // Common items visible to all roles
        common: [
            {
                title: 'Dashboard',
                href: `${API}/dashboard`,
                icon: LayoutGrid,
            },
        ],

        // Role-specific items
        doctor: [
            {
                title: 'Consultations',
                href: `${API}/consultations`,
                icon: ClipboardPlus,
            },
            {
                title: 'Prescriptions',
                href: `${API}/prescriptions`,
                icon: Tablets,
            },
        ],

        pharmacist: [
            {
                title: 'Medication',
                href: `${API}/medications`,
                icon: Pill,
            },
            {
                title: 'Suppliers',
                href: `${API}/suppliers`,
                icon: Activity,
            },
            {
                title: 'Stores',
                href: `${API}/central-stores`,
                icon: Warehouse,
            },
            {
                title: 'Clinics',
                href: `${API}/clinics`,
                icon: Hospital,
            },
        ],

        superadmin: [
            {
                title: 'Employees',
                href: `${API}/patients`,
                icon: User2,
            },
            {
                title: 'Dependants',
                href: `${API}/dependents`,
                icon: User2,
            },
            {
                title: 'Companies',
                href: `${API}/companies`,
                icon: Building,
            },
            {
                title: 'Users',
                href: `${API}/reports`,
                icon: User,
            },
        ],
    };

    // Get main navigation items based on role
    const getMainNavItems = (): NavItem[] => {
        const baseItems = [...allNavItems.common];

        // Add role-specific items
        if (userRole && allNavItems[userRole as keyof typeof allNavItems]) {
            baseItems.push(...allNavItems[userRole as keyof typeof allNavItems]);
        }

        return baseItems;
    };

    // Clinic medication stock item (visible to clinic staff)
    const clinicMedicationStockUrl = auth.user?.clinic_id ? `${API}/clinics/${auth.user.clinic_id}/medication-stocks` : null;

    // Reports items (common but can be restricted if needed)
    const getRightNavItems = (): NavItem[] => {
        const baseItems = [
            {
                title: 'Reports',
                href: `${API}/reports`,
                icon: ChartArea,
            },
            {
                title: 'Triage Reports',
                href: `${API}/triages`,
                icon: ChartSpline,
            },
            {
                title: 'Drug Use Reports',
                href: `${API}/dispensations`,
                icon: ChartSpline,
            },
        ];

        // Add clinic medication stock if available
        if (clinicMedicationStockUrl) {
            baseItems.unshift({
                title: 'Clinic Medication Stocks',
                href: clinicMedicationStockUrl,
                icon: PillBottle,
            });
        }

        // Admin-specific items
        if (userRole === 'superadmin') {
            baseItems.push(
                {
                    title: 'Diagnosis Presets',
                    href: `${API}/diagnosis-presets`,
                    icon: Cross,
                },
                {
                    title: 'Companies',
                    href: `${API}/companies`,
                    icon: Building,
                },
                {
                    title: 'Users',
                    href: `${API}/users`,
                    icon: User,
                },
            );
        }

        return baseItems;
    };

    const mainNavItems = getMainNavItems();
    const rightNavItems = getRightNavItems();

    return (
        <>
            <div className="border-sidebar-border/80 border-b">
                <div className="mx-auto flex h-16 items-center px-4 md:max-w-7xl">
                    {/* Mobile Menu */}
                    <div className="lg:hidden">
                        <Sheet>
                            <SheetTrigger asChild>
                                <Button variant="ghost" size="icon" className="mr-2 h-[34px] w-[34px]">
                                    <Menu className="h-5 w-5" />
                                </Button>
                            </SheetTrigger>
                            <SheetContent side="left" className="bg-sidebar flex h-full w-64 flex-col items-stretch justify-between">
                                <SheetTitle className="sr-only">Navigation Menu</SheetTitle>
                                <SheetHeader className="flex justify-start text-left">
                                    <AppLogoIcon className="h-6 w-6 fill-current text-black dark:text-white" />
                                </SheetHeader>
                                <div className="flex h-full flex-1 flex-col space-y-4 p-4">
                                    <div className="flex h-full flex-col justify-between text-sm">
                                        <div className="flex flex-col space-y-4">
                                            {mainNavItems.map((item) => (
                                                <Link key={item.title} href={item.href} className="flex items-center space-x-2 font-medium">
                                                    {item.icon && <Icon iconNode={item.icon} className="h-5 w-5" />}
                                                    <span>{item.title}</span>
                                                </Link>
                                            ))}
                                        </div>

                                        <div className="flex flex-col space-y-4">
                                            {rightNavItems.map((item) => (
                                                <Link key={item.title} href={item.href} className="flex items-center space-x-2 font-medium">
                                                    {item.icon && <Icon iconNode={item.icon} className="h-5 w-5" />}
                                                    <span>{item.title}</span>
                                                </Link>
                                            ))}
                                        </div>
                                    </div>
                                </div>
                            </SheetContent>
                        </Sheet>
                    </div>

                    <Link href={`${API}"/dashboard"`} prefetch className="flex items-center space-x-2">
                        <AppLogo />
                    </Link>

                    {/* Desktop Navigation */}
                    <div className="ml-6 hidden h-full items-center space-x-6 lg:flex">
                        <NavigationMenu className="flex h-full items-stretch">
                            <NavigationMenuList className="flex h-full items-stretch space-x-2">
                                {mainNavItems.map((item, index) => (
                                    <NavigationMenuItem key={index} className="relative flex h-full items-center">
                                        <Link
                                            href={item.href}
                                            className={cn(
                                                navigationMenuTriggerStyle(),
                                                page.url === item.href && activeItemStyles,
                                                'h-9 cursor-pointer px-3',
                                            )}
                                        >
                                            {item.icon && <Icon iconNode={item.icon} className="mr-2 h-4 w-4" />}
                                            {item.title}
                                        </Link>
                                        {page.url === item.href && (
                                            <div className="absolute bottom-0 left-0 h-0.5 w-full translate-y-px bg-black dark:bg-white"></div>
                                        )}
                                    </NavigationMenuItem>
                                ))}
                            </NavigationMenuList>
                        </NavigationMenu>
                    </div>

                    <div className="ml-auto flex items-center space-x-2">
                        <div className="relative flex items-center space-x-1">
                            <Button variant="ghost" size="icon" className="group h-9 w-9 cursor-pointer">
                                <Search className="!size-5 opacity-80 group-hover:opacity-100" />
                            </Button>
                            <div className="hidden lg:flex">
                                {rightNavItems.map((item) => (
                                    <TooltipProvider key={item.title} delayDuration={0}>
                                        <Tooltip>
                                            <TooltipTrigger>
                                                <Link
                                                    href={item.href}
                                                    className="group text-accent-foreground ring-offset-background hover:bg-accent hover:text-accent-foreground focus-visible:ring-ring ml-1 inline-flex h-9 w-9 items-center justify-center rounded-md bg-transparent p-0 text-sm font-medium transition-colors focus-visible:ring-2 focus-visible:ring-offset-2 focus-visible:outline-none disabled:pointer-events-none disabled:opacity-50"
                                                >
                                                    <span className="sr-only">{item.title}</span>
                                                    {item.icon && <Icon iconNode={item.icon} className="size-5 opacity-80 group-hover:opacity-100" />}
                                                </Link>
                                            </TooltipTrigger>
                                            <TooltipContent>
                                                <p>{item.title}</p>
                                            </TooltipContent>
                                        </Tooltip>
                                    </TooltipProvider>
                                ))}
                            </div>
                        </div>
                        <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                                <Button variant="ghost" className="size-10 rounded-full p-1">
                                    <Avatar className="size-8 overflow-hidden rounded-full">
                                        <AvatarImage src={auth.user.avatar} alt={auth.user.name} />
                                        <AvatarFallback className="rounded-lg bg-neutral-200 text-black dark:bg-neutral-700 dark:text-white">
                                            {getInitials(auth.user.name)}
                                        </AvatarFallback>
                                    </Avatar>
                                </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent className="w-56" align="end">
                                <UserMenuContent user={auth.user} />
                            </DropdownMenuContent>
                        </DropdownMenu>
                    </div>
                </div>
            </div>
            {breadcrumbs.length > 1 && (
                <div className="border-sidebar-border/70 flex w-full border-b">
                    <div className="mx-auto flex h-12 w-full items-center justify-start px-4 text-neutral-500 md:max-w-7xl">
                        <Breadcrumbs breadcrumbs={breadcrumbs} />
                    </div>
                </div>
            )}
        </>
    );
}
