import { NavFooter } from '@/components/nav-footer';
import { NavMain } from '@/components/nav-main';
import { NavUser } from '@/components/nav-user';
import {
  Sidebar,
  SidebarContent,
  SidebarFooter,
  SidebarHeader,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
} from '@/components/ui/sidebar';
import { API } from '@/config';
import { type NavItem, type SharedData } from '@/types';
import { Link, usePage, router } from '@inertiajs/react';
import {
  Activity,
  Building,
  ChartArea,
  ChartSpline,
  ClipboardList,
  ClipboardPlus,
  Cross,
  Hospital,
  LayoutGrid,
  Pill,
  PillBottle,
  Plus,
  Tablets,
  User,
  User2,
  Warehouse,
  Search,
  ClipboardEditIcon,
  Shield,
  NotebookPen,
  FileSpreadsheetIcon,
  ChartBar,
  ChartBarDecreasing,
  DockIcon,
  ArrowLeftRight, // ← Switch module icon
} from 'lucide-react';
import AppLogo from './app-logo';
import { useEffect, useState } from 'react';

// Define the possible user roles
type UserRole =
  | 'superadmin'
  | 'receptionist'
  | 'nurse'
  | 'doctor'
  | 'lab_technician'
  | 'radiologist'
  | 'pharmacist'
  | 'nurse_aid'
  | 'admin'
  | 'specialist'
  | 'casualty'
  | 'user'
  | 'patient';

type ImpersonationProp = {
  active: boolean;
  impersonator_id?: number;
  impersonator_name?: string;
};

// Modules we support via kiosk
type ActiveModule = 'primary' | 'dental' | 'chronics' | null;

// ✅ Look for any of these keys to find a consultation draft
const CONSULTATION_DRAFT_KEYS = [
  'consultation.form.draft.v1',
  'consultation_form_draft_v1',
  'consultation.form.v1',
  'consultationDraft',
] as const;

// Flip to true while debugging to see logs
const DEBUG_DRAFT = false;

function readDraftFromLocalStorage() {
  if (typeof window === 'undefined') return { has: false, href: `${API}/consultations/create` };

  for (const key of CONSULTATION_DRAFT_KEYS) {
    const raw = localStorage.getItem(key);
    if (!raw || !raw === null || !raw?.trim()) continue;

    try {
      const draft = JSON.parse(raw);
      const meaningful =
        !!draft?.patient_id ||
        !!draft?.presenting_complaint ||
        !!draft?.clinical_notes ||
        !!draft?.diagnosis ||
        !!draft?.treatment_plan ||
        !!draft?.referral_notes;

      if (!meaningful) continue;

      let href = `${API}/consultations/create`;
      if (draft?.patient_id) {
        href += `?patient_id=${encodeURIComponent(String(draft.patient_id))}`;
      }

      if (DEBUG_DRAFT) {
        // eslint-disable-next-line no-console
        console.log('[Consultation Draft] key:', key, 'draft:', draft, 'href:', href);
      }

      return { has: true, href };
    } catch {
      // ignore invalid JSON and try next key
    }
  }

  if (DEBUG_DRAFT) {
    // eslint-disable-next-line no-console
    console.log('[Consultation Draft] none found in keys:', CONSULTATION_DRAFT_KEYS);
  }
  return { has: false, href: `${API}/consultations/create` };
}

export function AppSidebar() {
  const page = usePage<SharedData & { impersonation?: ImpersonationProp; active_module?: ActiveModule }>();
  const { auth, impersonation, active_module } = page.props;

  // Treat undefined as primary, to preserve legacy behavior if not provided
  const currentModule: ActiveModule = (active_module ?? 'primary') as ActiveModule;
  const isPrimary = currentModule === 'primary';
  const isDental = currentModule === 'dental';
  const isChronics = currentModule === 'chronics';

  const userRole: UserRole | undefined = auth.user?.role as UserRole | undefined;
  const isImpersonating = Boolean(impersonation?.active);

  // 🔹 Detect a consultation draft in localStorage (PRIMARY ONLY, nurse)
  const [{ has: hasConsultationDraft, href: continueHref }, setDraftState] = useState(() =>
    readDraftFromLocalStorage()
  );

  useEffect(() => {
    setDraftState(readDraftFromLocalStorage());
    const onStorage = (e: StorageEvent) => {
      if (!e.key || !CONSULTATION_DRAFT_KEYS.includes(e.key as any)) return;
      setDraftState(readDraftFromLocalStorage());
    };
    window.addEventListener('storage', onStorage);
    return () => window.removeEventListener('storage', onStorage);
  }, []);

  // Map dashboard link to module-specific dashboard
  const dashboardHref =
    isDental ? `${API}/dental` : isChronics ? `${API}/chronics` : `${API}/dashboard`;

  // Define all possible navigation items (PRIMARY module feature set)
  const allNavItems = {
    superadmin: [
      {
        title: 'Find A Holder / Dependent',
        href: `${API}/patient/search`,
        icon: Search,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Add Account Holder',
        href: `${API}/patients/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Bulk Upload Holders',
        href: `${API}/patients/bulk-import`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Add Dependent',
        href: `${API}/dependents/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Providence Fund Members', href: `${API}/fund-members`, icon: User2 },
      { title: 'Dispense Corrections', href: `${API}/corrections`, icon: ClipboardList },
      { title: 'Companies', href: `${API}/companies`, icon: Building },
      { title: 'Users', href: `${API}/users`, icon: User },
      { title: 'Lab Reports', href: `${API}/referrals/lab`, icon: ChartBarDecreasing },
      { title: 'Imaging Reports', href: `${API}/referrals/imaging`, icon: ChartBarDecreasing },
    ],
    doctor: [
      { title: 'Consultations', href: `${API}/consultations`, icon: ClipboardPlus },
      { title: 'Prescriptions', href: `${API}/prescriptions`, icon: Tablets },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Previous Examinations', href: `${API}/doctor-notes`, icon: NotebookPen },
    ],
    pharmacist: [
      { title: 'Suppliers', href: `${API}/suppliers`, icon: Activity },
      { title: 'Stores', href: `${API}/central-stores`, icon: Warehouse },
      { title: 'Clinics', href: `${API}/clinics`, icon: Hospital },
      { title: 'Medication', href: `${API}/medications`, icon: Pill },
      { title: 'Medication Requests', href: `${API}/medication-requests`, icon: ClipboardList },
    ],
    nurse: [
      {
        title: 'Add Account Holder',
        href: `${API}/patients/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Add Dependent',
        href: `${API}/dependents/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      { title: 'Consultations', href: `${API}/consultations`, icon: ClipboardPlus },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Providence Fund Members', href: `${API}/fund-members`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Nurse Reports', href: `${API}/reports/nurse`, icon: ChartBar },
      { title: 'My Cabinet', href: `${API}/nurse-medication-stocks/${auth.user?.id}`, icon: PillBottle },
      { title: 'Lab Reports', href: `${API}/referrals/lab`, icon: ChartBarDecreasing },
      { title: 'Imaging Reports', href: `${API}/referrals/imaging`, icon: ChartBarDecreasing },
    ],
    receptionist: [
      {
        title: 'Find A Holder / Dependent',
        href: `${API}/patient/search`,
        icon: Search,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Add Traige',
        href: `${API}/triages/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Add Account Holder',
        href: `${API}/patients/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      {
        title: 'Add Dependent',
        href: `${API}/dependents/create`,
        icon: Plus,
        className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600',
      },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Providence Fund Members', href: `${API}/fund-members`, icon: User2 },
      { title: 'Companies', href: `${API}/companies`, icon: Building },
      { title: 'Consultations', href: `${API}/consultations`, icon: ClipboardPlus },
      { title: 'Reports', href: `${API}/reports`, icon: ChartArea },
      { title: 'Excel Reports', href: `${API}/reports/portal`, icon: FileSpreadsheetIcon },
      { title: 'Lab Reports', href: `${API}/referrals/lab`, icon: ChartBarDecreasing },
      { title: 'Imaging Reports', href: `${API}/referrals/imaging`, icon: ChartBarDecreasing },
    ],
  };

  // Clinic-specific items if user has a clinic_id (PRIMARY ONLY)
  const clinicMedicationItems: NavItem[] =
    isPrimary && auth.user?.clinic_id
      ? [
        {
          title: 'Clinic Medication Stocks',
          href: `${API}/clinics/${auth.user.clinic_id}/medication-stocks`,
          icon: PillBottle,
        },
        {
          title: 'Clinic Medication Requests',
          href: `${API}/clinics/${auth.user.clinic_id}/medication-requests`,
          icon: ClipboardList,
        },
      ]
      : [];

  const reportsItems: NavItem[] = isPrimary
    ? [
      { title: 'Reports', href: `${API}/reports`, icon: ChartArea },
      { title: 'Triage Reports', href: `${API}/triages`, icon: ChartSpline },
      { title: 'Manage Duplicates', href: `${API}/duplicates`, icon: DockIcon },
      { title: 'Drug Use Reports', href: `${API}/dispensations`, icon: ChartSpline },
      { title: 'Diagnosis Presets', href: `${API}/diagnosis-presets`, icon: Cross },
      { title: 'Logs', href: `${API}/logs`, icon: ClipboardEditIcon },
      { title: 'Gender Fix', href: `${API}/system-corrections/employees-without-gender`, icon: ClipboardEditIcon },
    ]
    : [];

  // Build main items based on role AND module.
  const getMainNavItems = (): NavItem[] => {
    let baseItems: NavItem[] = [
      { title: 'Dashboard', href: dashboardHref, icon: LayoutGrid },
    ];

    // In dental/chronics, ONLY Database is allowed in addition to Dashboard
    // if (isDental || isChronics) {
    //   baseItems.push({ title: 'Database', href: `${API}/patients`, icon: User2 });
    //   return baseItems;
    // }

    if (isDental || isChronics) {
      if (isDental) {
        baseItems.push({ title: 'Database', href: `${API}/patients`, icon: User2 });
      } else if (isChronics) {
        // ⬅️ point "Database" to the chronic patients index
        //baseItems.push({ title: 'Database', href: `${API}/chronics/patients`, icon: User2 });
        // (Optionally rename)
        baseItems.push({ title: 'Chronic Database', href: `${API}/chronics/patients`, icon: User2 });
      }
      return baseItems;
    }

    // PRIMARY module: role-based items
    const itemExists = (items: NavItem[], newItem: NavItem) =>
      items.some((item) => item.title === newItem.title && item.href === newItem.href);

    if (userRole === 'superadmin') {
      const roles = ['superadmin', 'doctor', 'pharmacist', 'nurse', 'receptionist'] as const;
      for (const role of roles) {
        (allNavItems as any)[role].forEach((item: NavItem) => {
          if (!itemExists(baseItems, item)) baseItems.push(item);
        });
      }
    } else if (userRole && (allNavItems as any)[userRole]) {
      (allNavItems as any)[userRole].forEach((item: NavItem) => {
        if (!itemExists(baseItems, item)) baseItems.push(item);
      });
    }

    if (userRole === 'superadmin') {
      baseItems = baseItems.filter(
        (item, index, self) => index === self.findIndex((t) => t.title === item.title && t.href === item.href)
      );
    }

    return baseItems;
  };

  const getFooterNavItems = (): NavItem[] => {
    if (!isPrimary) return []; // footer features are primary-only

    const items: NavItem[] = [];
    if (['superadmin', 'nurse', 'doctor'].includes(userRole ?? '') && clinicMedicationItems.length > 0) {
      items.push(...clinicMedicationItems);
    }

    if (userRole === 'receptionist') {
      items.push(...reportsItems.filter((item) => item.title === 'Triage Reports'));
    } else if (['superadmin', 'nurse', 'doctor'].includes(userRole ?? '')) {
      items.push(...reportsItems);
    }

    if (userRole === 'admin') {
      items.push(
        { title: 'Companies', href: `${API}/companies`, icon: Building },
        { title: 'Users', href: `${API}/users`, icon: User }
      );
    }

    return items;
  };

  // Stop impersonation (laravel-impersonate default route)
  const stopImpersonating = () => {
    router.visit(route('impersonate.leave'));
  };

  return (
    <Sidebar collapsible="icon" variant="inset">
      <SidebarHeader>
        <SidebarMenu>
          <SidebarMenuItem>
            <SidebarMenuButton size="lg" asChild>
              <Link href={dashboardHref} prefetch>
                <AppLogo />
              </Link>
            </SidebarMenuButton>
          </SidebarMenuItem>

          {/* 🔹 Only for nurses — show if a consultation draft exists AND in Primary module */}
          {isPrimary && userRole === 'nurse' && hasConsultationDraft && (
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={continueHref}
                  prefetch
                  className="w-full rounded-lg bg-gradient-to-r from-emerald-500 to-teal-600 px-3 py-2 text-white shadow-sm hover:from-emerald-600 hover:to-teal-700 focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-emerald-500"
                  title="Continue Consultation"
                >
                  <span className="inline-flex items-center gap-2">
                    <ClipboardPlus className="h-4 w-4" />
                    Continue Consultation
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          )}
        </SidebarMenu>
      </SidebarHeader>

      <SidebarContent>
        <NavMain items={getMainNavItems()} />
      </SidebarContent>

      <SidebarFooter>
        {/* 🔁 Switch Module button — visible in ALL modules */}
        <div className="mb-2">
          <SidebarMenu>
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={`${API}/kiosk`}
                  prefetch
                  className="w-full rounded-lg border px-3 py-2 text-sm font-medium hover:bg-accent hover:text-accent-foreground"
                  title="Switch Module"
                >
                  <span className="inline-flex items-center gap-2">
                    <ArrowLeftRight className="h-4 w-4" />
                    Switch Module
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          </SidebarMenu>
        </div>

        {/* 🔶 Impersonation banner */}
        {isImpersonating && (
          <div className="mb-2 rounded-lg border border-amber-300 bg-amber-50 px-3 py-2">
            <div className="flex items-center justify-between gap-2">
              <div className="flex items-center gap-2 text-amber-900">
                <Shield className="h-4 w-4 text-amber-600" />
                <span className="text-xs">
                  Impersonating <strong>{auth?.user?.name ?? 'user'}</strong>
                </span>
              </div>
              <button
                onClick={stopImpersonating}
                className="rounded bg-amber-600 px-2 py-1 text-xs font-medium text-white hover:bg-amber-700"
                title="Stop impersonating"
                style={{ cursor: 'pointer' }}
              >
                Stop
              </button>
            </div>
          </div>
        )}

        <NavFooter items={getFooterNavItems()} className="mt-auto" />
        <NavUser />
      </SidebarFooter>
    </Sidebar>
  );
}
