import { NavMain } from '@/components/nav-main';
import { NavUser } from '@/components/nav-user';
import { NavFooter } from '@/components/nav-footer';
import {
  Sidebar,
  SidebarContent,
  SidebarFooter,
  SidebarHeader,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
} from '@/components/ui/sidebar';
import { API } from '@/config';
import { type NavItem, type SharedData } from '@/types';
import { Link, router, usePage } from '@inertiajs/react';
import {
  LayoutGrid,
  ClipboardCheck,
  Shield,
  ArrowLeftRight,
  Users,
  Building,
  ChartArea,
  ChartSpline,
  ClipboardEditIcon,
  FileSpreadsheetIcon,
  User,
  ChevronDown,
  Settings2,
  UserCog,
  FileCog,
  Boxes,
  Wrench,
  CigaretteOff,
  Database,
  HeartCrack,
  CalendarHeart,
  Factory,
  DoorClosed,
  // NEW: quick-action icons
  PlusCircle,
  Upload,
  CalendarPlus,
  UserPlus,
} from 'lucide-react';
import AppLogo from '../app-logo';
import { useEffect, useMemo, useState } from 'react';

// Align roles with your primary app
type UserRole =
  | 'superadmin'
  | 'receptionist'
  | 'nurse'
  | 'doctor'
  | 'lab_technician'
  | 'radiologist'
  | 'pharmacist'
  | 'nurse_aid'
  | 'admin'
  | 'specialist'
  | 'casualty'
  | 'user'
  | 'patient';

type ImpersonationProp = {
  active: boolean;
  impersonator_id?: number;
  impersonator_name?: string;
};

export default function MedicalsSidebar() {
  const { auth, impersonation } =
    usePage<SharedData & { impersonation?: ImpersonationProp }>().props;
  const userRole = auth.user?.role as UserRole | undefined;

  const dashboardHref = `${API}/medicals`;

  // Main nav
  const items: NavItem[] = [
    { title: 'Dashboard', href: dashboardHref, icon: LayoutGrid },
    { title: 'Companies', href: `${API}/medicals/companies`, icon: Building },
    { title: 'Walk In Patients', href: `${API}/medicals/attendees`, icon: Users },
    { title: 'Patients Database', href: `${API}/medicals/patients`, icon: Users },
    { title: 'Daily Swab Register', href: `${API}/medicals/swabs`, icon: ClipboardCheck },
  ];

  // Footer catalog for medicals
  const reportsItemsMedicals: NavItem[] = [
    { title: 'Visual Reports', href: `${API}/medicals/reports`, icon: ChartArea },
    { title: 'Generated Reports', href: `${API}/medicals/reports/generated`, icon: ChartArea },
    { title: 'Certificates Reports', href: `${API}/medicals/reports/certificates`, icon: ChartSpline },
    { title: 'Excel Reports', href: `${API}/medicals/reports/portal`, icon: FileSpreadsheetIcon },
    { title: 'Corrections', href: `${API}/medicals/reports/corrections`, icon: ClipboardEditIcon },
    { title: 'Logs', href: `${API}/medicals/logs`, icon: ClipboardEditIcon },
  ];

  // Optional admin extras for medicals module
  const adminItems: NavItem[] = [
    { title: 'Companies', href: `${API}/medicals/companies`, icon: Building },
    { title: 'Users', href: `${API}/medicals/users`, icon: User }, // keep if route exists
  ];

  // Role-aware footer items
  const getFooterNavItems = (): NavItem[] => {
    const list: NavItem[] = [];

    if (userRole === 'receptionist') {
      list.push(
        reportsItemsMedicals.find((i) => i.title === 'Excel Reports') ?? {
          title: 'Excel Reports',
          href: `${API}/medicals/reports/portal`,
          icon: FileSpreadsheetIcon,
        },
      );
      return list;
    }

    if (['superadmin', 'nurse', 'doctor'].includes(userRole ?? '')) {
      list.push(...reportsItemsMedicals);
    }

    if (
      ['pharmacist', 'lab_technician', 'radiologist', 'nurse_aid', 'specialist', 'casualty'].includes(
        userRole ?? '',
      )
    ) {
      list.push(
        reportsItemsMedicals.find((i) => i.title === 'Visual Reports') ?? {
          title: 'Visual Reports',
          href: `${API}/medicals/reports`,
          icon: ChartArea,
        },
      );
    }

    if (userRole === 'admin') {
      list.push(...adminItems);
    }

    if (!list.length) {
      list.push(
        reportsItemsMedicals.find((i) => i.title === 'Visual Reports') ?? {
          title: 'Visual Reports',
          href: `${API}/medicals/reports`,
          icon: ChartArea,
        },
      );
    }

    // de-dupe
    return list.filter(
      (it, idx, self) => idx === self.findIndex((t) => t.title === it.title && t.href === it.href),
    );
  };

  // ──────────────────────────────────────────────────────────
  // Medicals Database (expand/collapse)
  // ──────────────────────────────────────────────────────────
  const DB_COLLAPSE_KEY = 'medicals.sidebar.db_open.v1';
  const [dbOpen, setDbOpen] = useState<boolean>(() => {
    if (typeof window === 'undefined') return false;
    try {
      const saved = localStorage.getItem(DB_COLLAPSE_KEY);
      if (saved === 'true') return true;
      if (saved === 'false') return false;
    } catch {}
    return false;
  });

  useEffect(() => {
    try {
      localStorage.setItem(DB_COLLAPSE_KEY, String(dbOpen));
    } catch {}
  }, [dbOpen]);

  // Database links (curate as you wish)
  const dbOps: NavItem[] = [
    { title: 'Pneumo Patients', href: `${API}/medicals/pneumo`, icon: HeartCrack },
    { title: 'Food-Hanndlers Patients', href: `${API}/medicals/cofh`, icon: CalendarHeart },
    { title: 'Industry Medicals Patients', href: `${API}/medicals/industry`, icon: Factory },
    { title: 'Exit Employement Patients', href: `${API}/medicals/exit-employment`, icon: DoorClosed },
    { title: 'Exit Pneumo Patients', href: `${API}/medicals/exit-employment`, icon: HeartCrack },
  ];

  // ──────────────────────────────────────────────────────────
  // Admin Operations (expand/collapse)
  // ──────────────────────────────────────────────────────────
  const isAdminish = useMemo(
    () => ['superadmin', 'admin'].includes(userRole ?? ''),
    [userRole],
  );

  const ADMIN_COLLAPSE_KEY = 'medicals.sidebar.admin_open.v1';
  const [adminOpen, setAdminOpen] = useState<boolean>(() => {
    if (typeof window === 'undefined') return false;
    try {
      const saved = localStorage.getItem(ADMIN_COLLAPSE_KEY);
      if (saved === 'true') return true;
      if (saved === 'false') return false;
    } catch {}
    return false;
  });

  useEffect(() => {
    try {
      localStorage.setItem(ADMIN_COLLAPSE_KEY, String(adminOpen));
    } catch {}
  }, [adminOpen]);

  // Admin operation links (adjust to your routes)
  const adminOps: NavItem[] = [
    { title: 'Users', href: `${API}/users`, icon: UserCog },
    { title: 'Illnesses Presets', href: `${API}/medicals/illnesses`, icon: Boxes },
    { title: 'Skin Conditions Presets', href: `${API}/medicals/skin-conditions`, icon: Settings2 },
    { title: 'Auscultates Presets', href: `${API}/medicals/auscultates`, icon: Wrench },
    { title: 'Smoking Presets', href: `${API}/medicals/tobacco-uses`, icon: CigaretteOff },
    { title: 'Export Center', href: `${API}/medicals/exports`, icon: FileCog },
  ];

  const stopImpersonating = () => router.visit(route('impersonate.leave'));

  return (
    <Sidebar collapsible="icon" variant="inset">
      <SidebarHeader>
        <SidebarMenu>
          <SidebarMenuItem>
            <SidebarMenuButton size="lg" asChild>
              <Link href={dashboardHref} prefetch>
                <AppLogo />
              </Link>
            </SidebarMenuButton>
          </SidebarMenuItem>
        </SidebarMenu>
      </SidebarHeader>

      <SidebarContent>
        {/* ───────────────────────────────────────────────
            QUICK ACTIONS (above Dashboard nav)
        ─────────────────────────────────────────────── */}
        <div className="mb-2">
          <SidebarMenu>
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={`${API}/medicals/bookings/create`}
                  prefetch
                  className="w-full rounded-lg px-3 py-2 text-sm font-medium text-white"
                  style={{ backgroundColor: '#2563eb' }} // blue-600
                >
                  <span className="inline-flex items-center gap-2">
                    <PlusCircle className="h-4 w-4" />
                    Add Single Booking
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>

            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={`${API}/medicals/patients/bulk-upload`}
                  prefetch
                  className="w-full rounded-lg px-3 py-2 text-sm font-medium text-white"
                  style={{ backgroundColor: '#059669' }} // emerald-600
                >
                  <span className="inline-flex items-center gap-2">
                    <Upload className="h-4 w-4" />
                    Bulk Patient Upload
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>

            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={`${API}/medicals/patients/bookings/create`}
                  prefetch
                  className="w-full rounded-lg px-3 py-2 text-sm font-medium text-white"
                  style={{ backgroundColor: '#7c3aed' }} // violet-600
                >
                  <span className="inline-flex items-center gap-2">
                    <CalendarPlus className="h-4 w-4" />
                    Add Patient Booking
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>

            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={`${API}/medicals/attendees/create`}
                  prefetch
                  className="w-full rounded-lg px-3 py-2 text-sm font-medium text-white"
                  style={{ backgroundColor: '#0284c7' }} // sky-600
                >
                  <span className="inline-flex items-center gap-2">
                    <UserPlus className="h-4 w-4" />
                    Add New Patient
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          </SidebarMenu>
        </div>

        {/* Main navigation */}
        <NavMain items={items} />

        {/* Medicals Database (expand/collapse) */}
        <div className="mt-2">
          <SidebarMenu>
            <SidebarMenuItem>
              <button
                type="button"
                onClick={() => setDbOpen((v) => !v)}
                className="flex w-full items-center justify-between rounded-md px-3 py-2 text-left text-sm font-medium hover:bg-accent"
                title="Medicals Database"
              >
                <span className="inline-flex items-center gap-2">
                  <Database className="h-4 w-4" />
                  Medicals Database
                </span>
                <ChevronDown
                  className={`h-4 w-4 transition-transform ${dbOpen ? 'rotate-180' : 'rotate-0'}`}
                />
              </button>
            </SidebarMenuItem>

            {dbOpen && (
              <div className="ml-2 space-y-1 border-l pl-3">
                {dbOps.map((op) => (
                  <SidebarMenuItem key={op.href}>
                    <SidebarMenuButton asChild>
                      <Link href={op.href} prefetch className="flex items-center gap-2">
                        <op.icon className="h-4 w-4" />
                        <span className="text-sm">{op.title}</span>
                      </Link>
                    </SidebarMenuButton>
                  </SidebarMenuItem>
                ))}
              </div>
            )}
          </SidebarMenu>
        </div>

        {/* Admin Operations (expand/collapse) */}
        {isAdminish && (
          <div className="mt-2">
            <SidebarMenu>
              <SidebarMenuItem>
                <button
                  type="button"
                  onClick={() => setAdminOpen((v) => !v)}
                  className="flex w-full items-center justify-between rounded-md px-3 py-2 text-left text-sm font-medium hover:bg-accent"
                  title="Admin Operations"
                >
                  <span className="inline-flex items-center gap-2">
                    <Settings2 className="h-4 w-4" />
                    Admin Operations
                  </span>
                  <ChevronDown
                    className={`h-4 w-4 transition-transform ${
                      adminOpen ? 'rotate-180' : 'rotate-0'
                    }`}
                  />
                </button>
              </SidebarMenuItem>

              {adminOpen && (
                <div className="ml-2 space-y-1 border-l pl-3">
                  {adminOps.map((op) => (
                    <SidebarMenuItem key={op.href}>
                      <SidebarMenuButton asChild>
                        <Link href={op.href} prefetch className="flex items-center gap-2">
                          <op.icon className="h-4 w-4" />
                          <span className="text-sm">{op.title}</span>
                        </Link>
                      </SidebarMenuButton>
                    </SidebarMenuItem>
                  ))}
                </div>
              )}
            </SidebarMenu>
          </div>
        )}
      </SidebarContent>

      <SidebarFooter>
        <div className="mb-2">
          <SidebarMenu>
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={`${API}/kiosk`}
                  prefetch
                  className="w-full rounded-lg border px-3 py-2 text-sm font-medium hover:bg-accent"
                >
                  <span className="inline-flex items-center gap-2">
                    <ArrowLeftRight className="h-4 w-4" /> Switch Module
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          </SidebarMenu>
        </div>

        {impersonation?.active && (
          <div className="mb-2 rounded-lg border border-amber-300 bg-amber-50 px-3 py-2">
            <div className="flex items-center justify-between gap-2">
              <div className="flex items-center gap-2 text-amber-900">
                <Shield className="h-4 w-4 text-amber-600" />
                <span className="text-xs">
                  Impersonating <strong>{auth?.user?.name ?? 'user'}</strong>
                </span>
              </div>
              <button
                onClick={stopImpersonating}
                className="rounded bg-amber-600 px-2 py-1 text-xs font-medium text-white hover:bg-amber-700"
              >
                Stop
              </button>
            </div>
          </div>
        )}

        {/* Footer nav (role-aware) */}
        <NavFooter items={getFooterNavItems()} className="mt-auto" />
        <NavUser />
      </SidebarFooter>
    </Sidebar>
  );
}
