import { NavFooter } from '@/components/nav-footer';
import { NavMain } from '@/components/nav-main';
import { NavUser } from '@/components/nav-user';
import {
  Sidebar, SidebarContent, SidebarFooter, SidebarHeader,
  SidebarMenu, SidebarMenuButton, SidebarMenuItem,
} from '@/components/ui/sidebar';
import { API } from '@/config';
import { type NavItem, type SharedData } from '@/types';
import { Link, router, usePage } from '@inertiajs/react';
import {
  Activity, Building, ChartArea, ChartBarDecreasing, ChartSpline, ClipboardEditIcon,
  ClipboardList, ClipboardPlus, Cross, DockIcon, FileSpreadsheetIcon, LayoutGrid,
  NotebookPen, Pill, PillBottle, Search, Tablets, User, User2, Warehouse, Shield,
  ArrowLeftRight,
} from 'lucide-react';
import AppLogo from '../app-logo';
import { useEffect, useState } from 'react';

type UserRole =
  | 'superadmin' | 'receptionist' | 'nurse' | 'doctor' | 'lab_technician'
  | 'radiologist' | 'pharmacist' | 'nurse_aid' | 'admin' | 'specialist'
  | 'casualty' | 'user' | 'patient';

type ImpersonationProp = {
  active: boolean;
  impersonator_id?: number;
  impersonator_name?: string;
};

// ===== Consultation draft util (PRIMARY ONLY) =====
const KEYS = [
  'consultation.form.draft.v1',
  'consultation_form_draft_v1',
  'consultation.form.v1',
  'consultationDraft',
] as const;

function readDraftFromLocalStorage() {
  if (typeof window === 'undefined') return { has: false, href: `${API}/consultations/create` };
  for (const key of KEYS) {
    const raw = localStorage.getItem(key);
    if (!raw || !raw?.trim()) continue;
    try {
      const draft = JSON.parse(raw);
      const meaningful = !!(draft?.patient_id || draft?.presenting_complaint || draft?.clinical_notes || draft?.diagnosis || draft?.treatment_plan || draft?.referral_notes);
      if (!meaningful) continue;
      let href = `${API}/consultations/create`;
      if (draft?.patient_id) href += `?patient_id=${encodeURIComponent(String(draft.patient_id))}`;
      return { has: true, href };
    } catch {}
  }
  return { has: false, href: `${API}/consultations/create` };
}

export default function PrimarySidebar() {
  const page = usePage<SharedData & { impersonation?: ImpersonationProp }>();
  const { auth, impersonation } = page.props;
  const userRole = auth.user?.role as UserRole | undefined;
  const isImpersonating = Boolean(impersonation?.active);

  const dashboardHref = `${API}/dashboard`;

  // primary-only extras
  const clinicMedicationItems: NavItem[] =
    auth.user?.clinic_id
      ? [
          { title: 'Clinic Medication Stocks', href: `${API}/clinics/${auth.user.clinic_id}/medication-stocks`, icon: PillBottle },
          { title: 'Clinic Medication Requests', href: `${API}/clinics/${auth.user.clinic_id}/medication-requests`, icon: ClipboardList },
        ]
      : [];

  const reportsItems: NavItem[] = [
    { title: 'Reports', href: `${API}/reports`, icon: ChartArea },
    { title: 'Triage Reports', href: `${API}/triages`, icon: ChartSpline },
    { title: 'Manage Duplicates', href: `${API}/duplicates`, icon: DockIcon },
    { title: 'Drug Use Reports', href: `${API}/dispensations`, icon: ChartSpline },
    { title: 'Diagnosis Presets', href: `${API}/diagnosis-presets`, icon: Cross },
    { title: 'Logs', href: `${API}/logs`, icon: ClipboardEditIcon },
  ];

  const allNavItems = {
    superadmin: [
      { title: 'Find A Holder / Dependent', href: `${API}/patient/search`, icon: Search, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Add Account Holder', href: `${API}/patients/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Bulk Upload Holders', href: `${API}/patients/bulk-import`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Add Dependent', href: `${API}/dependents/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Providence Fund Members', href: `${API}/fund-members`, icon: User2 },
      { title: 'Dispense Corrections', href: `${API}/corrections`, icon: ClipboardList },
      { title: 'Companies', href: `${API}/companies`, icon: Building },
      { title: 'Users', href: `${API}/users`, icon: User },
      { title: 'Lab Reports', href: `${API}/referrals/lab`, icon: ChartBarDecreasing },
      { title: 'Imaging Reports', href: `${API}/referrals/imaging`, icon: ChartBarDecreasing },
    ],
    doctor: [
      { title: 'Consultations', href: `${API}/consultations`, icon: ClipboardPlus },
      { title: 'Prescriptions', href: `${API}/prescriptions`, icon: Tablets },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Previous Examinations', href: `${API}/doctor-notes`, icon: NotebookPen },
    ],
    pharmacist: [
      { title: 'Suppliers', href: `${API}/suppliers`, icon: Activity },
      { title: 'Stores', href: `${API}/central-stores`, icon: Warehouse },
      { title: 'Clinics', href: `${API}/clinics`, icon: Building },
      { title: 'Medication', href: `${API}/medications`, icon: Pill },
      { title: 'Medication Requests', href: `${API}/medication-requests`, icon: ClipboardList },
    ],
    nurse: [
      { title: 'Add Account Holder', href: `${API}/patients/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Add Dependent', href: `${API}/dependents/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Consultations', href: `${API}/consultations`, icon: ClipboardPlus },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Providence Fund Members', href: `${API}/fund-members`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Nurse Reports', href: `${API}/reports/nurse`, icon: ChartArea },
      { title: 'My Cabinet', href: `${API}/nurse-medication-stocks/${auth.user?.id}`, icon: PillBottle },
      { title: 'Lab Reports', href: `${API}/referrals/lab`, icon: ChartBarDecreasing },
      { title: 'Imaging Reports', href: `${API}/referrals/imaging`, icon: ChartBarDecreasing },
    ],
    receptionist: [
      { title: 'Find A Holder / Dependent', href: `${API}/patient/search`, icon: Search, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Add Traige', href: `${API}/triages/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Add Account Holder', href: `${API}/patients/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Add Dependent', href: `${API}/dependents/create`, icon: ClipboardPlus, className: 'bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600' },
      { title: 'Database', href: `${API}/patients`, icon: User2 },
      { title: 'Dependants', href: `${API}/dependents`, icon: User2 },
      { title: 'Providence Fund Members', href: `${API}/fund-members`, icon: User2 },
      { title: 'Companies', href: `${API}/companies`, icon: Building },
      { title: 'Consultations', href: `${API}/consultations`, icon: ClipboardPlus },
      { title: 'Reports', href: `${API}/reports`, icon: ChartArea },
      { title: 'Excel Reports', href: `${API}/reports/portal`, icon: FileSpreadsheetIcon },
      { title: 'Lab Reports', href: `${API}/referrals/lab`, icon: ChartBarDecreasing },
      { title: 'Imaging Reports', href: `${API}/referrals/imaging`, icon: ChartBarDecreasing },
    ],
  };

  // build main items for primary
  const getMainNavItems = (): NavItem[] => {
    let base: NavItem[] = [{ title: 'Dashboard', href: dashboardHref, icon: LayoutGrid }];

    const addUnique = (arr: NavItem[], item: NavItem) => {
      if (!arr.some(i => i.title === item.title && i.href === item.href)) arr.push(item);
    };

    if (userRole === 'superadmin') {
      (['superadmin', 'doctor', 'pharmacist', 'nurse', 'receptionist'] as const).forEach(role => {
        (allNavItems as any)[role].forEach((i: NavItem) => addUnique(base, i));
      });
    } else if (userRole && (allNavItems as any)[userRole]) {
      (allNavItems as any)[userRole].forEach((i: NavItem) => addUnique(base, i));
    }

    // dedupe for superadmin
    if (userRole === 'superadmin') {
      base = base.filter((it, idx, self) => idx === self.findIndex(t => t.title === it.title && t.href === it.href));
    }

    return base;
  };

  // footer (primary only)
  const getFooterNavItems = (): NavItem[] => {
    const items: NavItem[] = [];
    if (['superadmin', 'nurse', 'doctor'].includes(userRole ?? '') && clinicMedicationItems.length > 0) {
      items.push(...clinicMedicationItems);
    }
    if (userRole === 'receptionist') {
      items.push(...reportsItems.filter(i => i.title === 'Triage Reports'));
    } else if (['superadmin', 'nurse', 'doctor'].includes(userRole ?? '')) {
      items.push(...reportsItems);
    }
    if (userRole === 'admin') {
      items.push(
        { title: 'Companies', href: `${API}/companies`, icon: Building },
        { title: 'Users', href: `${API}/users`, icon: User },
      );
    }
    return items;
  };

  // consultation draft (primary nurse shortcut)
  const [{ has: hasDraft, href: continueHref }, setDraft] = useState(readDraftFromLocalStorage());
  useEffect(() => {
    setDraft(readDraftFromLocalStorage());
    const onStorage = (e: StorageEvent) => {
      if (!e.key || !(KEYS as readonly string[]).includes(e.key)) return;
      setDraft(readDraftFromLocalStorage());
    };
    window.addEventListener('storage', onStorage);
    return () => window.removeEventListener('storage', onStorage);
  }, []);

  const stopImpersonating = () => router.visit(route('impersonate.leave'));

  return (
    <Sidebar collapsible="icon" variant="inset">
      <SidebarHeader>
        <SidebarMenu>
          <SidebarMenuItem>
            <SidebarMenuButton size="lg" asChild>
              <Link href={dashboardHref} prefetch><AppLogo /></Link>
            </SidebarMenuButton>
          </SidebarMenuItem>

          {userRole === 'nurse' && hasDraft && (
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link
                  href={continueHref}
                  prefetch
                  className="w-full rounded-lg bg-gradient-to-r from-emerald-500 to-teal-600 px-3 py-2 text-white shadow-sm hover:from-emerald-600 hover:to-teal-700"
                  title="Continue Consultation"
                >
                  <span className="inline-flex items-center gap-2">
                    <ClipboardPlus className="h-4 w-4" /> Continue Consultation
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          )}
        </SidebarMenu>
      </SidebarHeader>

      <SidebarContent>
        <NavMain items={getMainNavItems()} />
      </SidebarContent>

      <SidebarFooter>
        <div className="mb-2">
          <SidebarMenu>
            <SidebarMenuItem>
              <SidebarMenuButton asChild>
                <Link href={`${API}/kiosk`} prefetch className="w-full rounded-lg border px-3 py-2 text-sm font-medium hover:bg-accent">
                  <span className="inline-flex items-center gap-2">
                    <ArrowLeftRight className="h-4 w-4" /> Switch Module
                  </span>
                </Link>
              </SidebarMenuButton>
            </SidebarMenuItem>
          </SidebarMenu>
        </div>

        {isImpersonating && (
          <div className="mb-2 rounded-lg border border-amber-300 bg-amber-50 px-3 py-2">
            <div className="flex items-center justify-between gap-2">
              <div className="flex items-center gap-2 text-amber-900">
                <Shield className="h-4 w-4 text-amber-600" />
                <span className="text-xs">Impersonating <strong>{auth?.user?.name ?? 'user'}</strong></span>
              </div>
              <button onClick={stopImpersonating} className="rounded bg-amber-600 px-2 py-1 text-xs font-medium text-white hover:bg-amber-700">
                Stop
              </button>
            </div>
          </div>
        )}

        <NavFooter items={getFooterNavItems()} className="mt-auto" />
        <NavUser />
      </SidebarFooter>
    </Sidebar>
  );
}
