// AgeRangeChart.tsx
import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';
import { API } from '@/config';
import axios from 'axios';
import { useEffect, useMemo, useState } from 'react';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Skeleton } from '@/components/ui/skeleton';

// recharts
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
} from 'recharts';

interface AgeRangeDistribution {
  range: string;
  count: number;
}

type TooltipProps = {
  active?: boolean;
  payload?: Array<{ value: number }>;
  label?: string;
};

export function AgeRangeChart() {
  const [data, setData] = useState<AgeRangeDistribution[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const res = await axios.get<AgeRangeDistribution[]>(
          `${API}/patients/age-range-distribution`
        );
        setData(res.data || []);
        setError(null);
      } catch (e) {
        console.error(e);
        setError('Failed to fetch age range distribution data');
      } finally {
        setLoading(false);
      }
    })();
  }, []);

  const categories = useMemo(() => data.map((d) => d.range), [data]);

  const CustomTooltip = ({ active, payload, label }: TooltipProps) => {
    if (!active || !payload?.length) return null;
    return (
      <div className="rounded-md border bg-popover px-3 py-2 text-sm shadow-sm">
        <div className="font-medium">{label}</div>
        <div className="text-muted-foreground">Patients: {payload[0].value}</div>
      </div>
    );
  };

  // Blue palette
  const BLUE_TICK = '#60A5FA';   // blue-400
  const BLUE_GRID = '#BFDBFE';   // blue-200
  const BLUE_STROKE = '#2563EB'; // blue-600

  return (
    <Card className="rounded-xl border shadow-sm">
      <CardHeader>
        <CardTitle className="text-lg">Age Range Distribution</CardTitle>
        <CardDescription>Patients grouped by age brackets</CardDescription>
      </CardHeader>

      <CardContent className="pt-4">
        {loading ? (
          <div className="h-[420px]">
            <Skeleton className="mb-3 h-4 w-40" />
            <Skeleton className="mb-2 h-[380px] w-full" />
          </div>
        ) : error ? (
          <Alert variant="destructive">
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : data.length === 0 ? (
          <PlaceholderPattern />
        ) : (
          <div className="h-[420px]">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={data} margin={{ top: 12, right: 20, left: 8, bottom: 28 }}>
                <defs>
                  <linearGradient id="ageRangeBlue" x1="0" y1="0" x2="0" y2="1">
                    <stop offset="0%" stopColor="#93C5FD" stopOpacity={0.95} />
                    <stop offset="60%" stopColor="#60A5FA" stopOpacity={0.9} />
                    <stop offset="100%" stopColor="#3B82F6" stopOpacity={0.85} />
                  </linearGradient>
                </defs>

                <CartesianGrid stroke={BLUE_GRID} strokeDasharray="3 3" opacity={0.6} />
                <XAxis
                  dataKey="range"
                  tick={{ fontSize: 12, fill: BLUE_TICK }}
                  interval={0}
                  angle={categories.length > 8 ? -20 : 0}
                  textAnchor={categories.length > 8 ? 'end' : 'middle'}
                  axisLine={false}
                  tickLine={false}
                />
                <YAxis
                  allowDecimals={false}
                  tick={{ fontSize: 12, fill: BLUE_TICK }}
                  axisLine={false}
                  tickLine={false}
                />
                <Tooltip content={<CustomTooltip />} />
                <Bar
                  dataKey="count"
                  name="Patients"
                  fill="url(#ageRangeBlue)"
                  stroke={BLUE_STROKE}
                  strokeWidth={1}
                  radius={[8, 8, 0, 0]}
                />
              </BarChart>
            </ResponsiveContainer>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
