import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';
import { API } from '@/config';
import axios from 'axios';
import { useEffect, useMemo, useState } from 'react';

// recharts
import {
  ResponsiveContainer,
  AreaChart,
  Area,
  XAxis,
  YAxis,
  Tooltip,
  Legend,
  ReferenceLine,
} from 'recharts';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Skeleton } from '@/components/ui/skeleton';
import { Alert, AlertDescription } from '@/components/ui/alert';

type ConsultationRow = {
  month: string | number;     // e.g. "2025-09" or 9
  employees: number;
  dependents: number;
  total?: number;
};
type Clinic = { id: number; name: string };
type Year   = { year: number };

const MONTHS = [
  { label: 'Jan', key: 'january', n: 1 },
  { label: 'Feb', key: 'february', n: 2 },
  { label: 'Mar', key: 'march', n: 3 },
  { label: 'Apr', key: 'april', n: 4 },
  { label: 'May', key: 'may', n: 5 },
  { label: 'Jun', key: 'june', n: 6 },
  { label: 'Jul', key: 'july', n: 7 },
  { label: 'Aug', key: 'august', n: 8 },
  { label: 'Sep', key: 'september', n: 9 },
  { label: 'Oct', key: 'october', n: 10 },
  { label: 'Nov', key: 'november', n: 11 },
  { label: 'Dec', key: 'december', n: 12 },
];

export function ConsultationChart() {
  const [data, setData] = useState<ConsultationRow[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);
  const [clinics, setClinics] = useState<Clinic[]>([]);
  const [years, setYears] = useState<Year[]>([]);
  const [selectedClinic, setSelectedClinic] = useState<number | null>(null);
  const [selectedYear, setSelectedYear] = useState<number | null>(new Date().getFullYear());

  // Load clinics & years
  useEffect(() => {
    (async () => {
      try {
        const [clinicsRes, yearsRes] = await Promise.all([
          axios.get<Clinic[]>(`${API}/reports/clinics`),
          axios.get<Year[]>(`${API}/reports/years`),
        ]);
        setClinics(clinicsRes.data ?? []);
        setYears(yearsRes.data ?? []);
      } catch (err) {
        console.error(err);
      }
    })();
  }, []);

  // Fetch series
  useEffect(() => {
    (async () => {
      if (selectedYear == null) return;
      try {
        setLoading(true);
        setError(null);
        const res = await axios.get<ConsultationRow[]>(`${API}/reports/consultations/monthly`, {
          params: {
            year: selectedYear,
            clinic_id: selectedClinic ?? undefined,
          },
        });
        setData(res.data ?? []);
      } catch (err) {
        console.error(err);
        setError('Failed to fetch consultation data');
      } finally {
        setLoading(false);
      }
    })();
  }, [selectedClinic, selectedYear]);

  // Robust month parser (supports "YYYY-MM", "YYYY-M", "YYYY-MM-DD", numeric, names)
  const monthIndexFromValue = (m: string | number): number | null => {
    if (typeof m === 'number') return m >= 1 && m <= 12 ? m - 1 : null;
    const s = String(m).trim();

    let match = s.match(/^(\d{4})[-/](\d{1,2})$/);
    if (match) {
      const n = Number(match[2]);
      return n >= 1 && n <= 12 ? n - 1 : null;
    }
    match = s.match(/^(\d{4})[-/](\d{1,2})[-/](\d{1,2})$/);
    if (match) {
      const n = Number(match[2]);
      return n >= 1 && n <= 12 ? n - 1 : null;
    }
    if (/^\d{1,2}$/.test(s)) {
      const n = Number(s);
      return n >= 1 && n <= 12 ? n - 1 : null;
    }
    const idx = MONTHS.findIndex(
      (mm) => mm.key === s.toLowerCase() || mm.label.toLowerCase() === s.toLowerCase()
    );
    return idx >= 0 ? idx : null;
  };

  // Build 12-month rows with employees + dependents stacked
  const chartData = useMemo(() => {
    const base = MONTHS.map((m) => ({ month: m.label, employees: 0, dependents: 0, total: 0 }));
    for (const row of data) {
      const idx = monthIndexFromValue(row.month);
      if (idx != null) {
        base[idx].employees  = Number(row.employees ?? 0);
        base[idx].dependents = Number(row.dependents ?? 0);
        base[idx].total      = Number(row.total ?? (base[idx].employees + base[idx].dependents));
      }
    }
    return base;
  }, [data]);

  const total = useMemo(
    () => chartData.reduce((s, r) => s + (r.total ?? r.employees + r.dependents), 0),
    [chartData]
  );

  // Theme-friendly blues
  const EMP_GRAD_ID = 'areaEmpBlue';
  const DEP_GRAD_ID = 'areaDepBlue';
  const tickColor   = '#60a5fa';    // blue-400
  const refColor    = '#93c5fd';    // blue-300

  return (
    <Card className="rounded-xl">
      <CardHeader className="pb-3">
        <div className="flex flex-col gap-1 sm:flex-row sm:items-end sm:justify-between">
          <div>
            <CardTitle className="text-lg">Monthly Attendance</CardTitle>
            <CardDescription>
              Total for {selectedYear ?? '—'}:{' '}
              <span className="font-medium text-foreground">{total}</span>
            </CardDescription>
          </div>

          <div className="flex flex-wrap gap-2">
            {/* Clinic selector */}
            <Select
              value={selectedClinic ? String(selectedClinic) : 'all'}
              onValueChange={(v) => setSelectedClinic(v === 'all' ? null : Number(v))}
            >
              <SelectTrigger className="w-[200px]">
                <SelectValue placeholder="Select clinic" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Clinics</SelectItem>
                {clinics.map((c) => (
                  <SelectItem key={c.id} value={String(c.id)}>
                    {c.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>

            {/* Year selector */}
            <Select
              value={selectedYear ? String(selectedYear) : ''}
              onValueChange={(v) => setSelectedYear(v ? Number(v) : null)}
            >
              <SelectTrigger className="w-[140px]">
                <SelectValue placeholder="Year" />
              </SelectTrigger>
              <SelectContent>
                {years.length > 0
                  ? years
                      .map((y) => y.year)
                      .sort((a, b) => b - a)
                      .map((y) => (
                        <SelectItem key={y} value={String(y)}>
                          {y}
                        </SelectItem>
                      ))
                  : [...Array(5)]
                      .map((_, i) => new Date().getFullYear() - i)
                      .map((y) => (
                        <SelectItem key={y} value={String(y)}>
                          {y}
                        </SelectItem>
                      ))}
              </SelectContent>
            </Select>
          </div>
        </div>
      </CardHeader>

      <CardContent>
        {loading ? (
          <div className="h-[360px]">
            <Skeleton className="h-full w-full rounded-lg" />
          </div>
        ) : error ? (
          <Alert variant="destructive">
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : chartData.some((d) => (d.employees + d.dependents) > 0) ? (
          <div className="h-[380px] w-full">
            <ResponsiveContainer width="100%" height="100%">
              <AreaChart data={chartData} margin={{ top: 8, right: 16, bottom: 8, left: 0 }}>
                {/* Blue gradients */}
                <defs>
                  <linearGradient id={EMP_GRAD_ID} x1="0" y1="0" x2="0" y2="1">
                    <stop offset="0%" stopColor="#3b82f6" stopOpacity={0.9} />
                    <stop offset="100%" stopColor="#3b82f6" stopOpacity={0.35} />
                  </linearGradient>
                  <linearGradient id={DEP_GRAD_ID} x1="0" y1="0" x2="0" y2="1">
                    <stop offset="0%" stopColor="#93c5fd" stopOpacity={0.9} />
                    <stop offset="100%" stopColor="#93c5fd" stopOpacity={0.35} />
                  </linearGradient>
                </defs>

                {/* No grid (kept consistent with your preference) */}
                <XAxis
                  dataKey="month"
                  tickLine={false}
                  axisLine={false}
                  tick={{ fill: tickColor }}
                />
                <YAxis
                  allowDecimals={false}
                  width={36}
                  tickLine={false}
                  axisLine={false}
                  tick={{ fill: tickColor }}
                />
                <Tooltip
                  cursor={{ fill: 'rgba(59,130,246,0.08)' }} // blue-500 @ 8%
                  contentStyle={{
                    background: 'hsl(var(--popover))',
                    border: '1px solid rgba(59,130,246,0.25)',
                    borderRadius: '0.5rem',
                  }}
                  labelStyle={{ color: 'hsl(var(--foreground))' }}
                  itemStyle={{ color: 'hsl(var(--foreground))' }}
                  formatter={(val: any, name: any) => [val, name]}
                />
                <Legend wrapperStyle={{ color: tickColor }} verticalAlign="top" height={24} />
                <ReferenceLine y={0} stroke={refColor} />

                {/* Stacked areas */}
                <Area
                  type="monotone"
                  dataKey="employees"
                  name="Employees"
                  stackId="consults"
                  stroke="#3b82f6"
                  fill={`url(#${EMP_GRAD_ID})`}
                  strokeWidth={2}
                  dot={false}
                />
                <Area
                  type="monotone"
                  dataKey="dependents"
                  name="Dependents"
                  stackId="consults"
                  stroke="#93c5fd"
                  fill={`url(#${DEP_GRAD_ID})`}
                  strokeWidth={2}
                  dot={false}
                />
              </AreaChart>
            </ResponsiveContainer>
          </div>
        ) : (
          <div className="h-[360px]">
            <PlaceholderPattern />
          </div>
        )}
      </CardContent>
    </Card>
  );
}
