// EmployeeDependentsChart.tsx
import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';
import { API } from '@/config';
import axios from 'axios';
import { useEffect, useMemo, useState } from 'react';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Skeleton } from '@/components/ui/skeleton';

// recharts
import {
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell,
  Tooltip,
  Legend,
} from 'recharts';

interface EmployeeVsDependents {
  category: string;
  count: number;
}

type TooltipProps = {
  active?: boolean;
  payload?: Array<{ value: number; name: string }>;
};

export function EmployeeDependentsChart() {
  const [data, setData] = useState<EmployeeVsDependents[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  // fetch once
  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const res = await axios.get<EmployeeVsDependents[]>(
          `${API}/patients/employee-vs-dependents`
        );
        setData(res.data || []);
        setError(null);
      } catch (e) {
        console.error(e);
        setError('Failed to fetch employee vs dependents data');
      } finally {
        setLoading(false);
      }
    })();
  }, []);

  const total = useMemo(
    () => data.reduce((sum, d) => sum + (d.count ?? 0), 0),
    [data]
  );

  // Light, friendly blues (works on light & dark)
  const COLORS = ['#60A5FA', '#93C5FD', '#BFDBFE', '#DBEAFE']; // blue-400, blue-300, blue-200, blue-100

  const CustomTooltip = ({ active, payload }: TooltipProps) => {
    if (!active || !payload?.length) return null;
    const item = payload[0];
    return (
      <div className="rounded-md border bg-popover px-3 py-2 text-sm shadow-sm">
        <div className="font-medium">{item.name}</div>
        <div className="text-muted-foreground">Count: {item.value}</div>
      </div>
    );
  };

  // Center label for donut
  const renderCenterLabel = () => {
    if (!total) return null;
    return (
      <>
        <text
          x="50%"
          y="46%"
          textAnchor="middle"
          className="fill-foreground"
          style={{ fontSize: 15, fontWeight: 700 }}
        >
          Total
        </text>
        <text
          x="50%"
          y="64%"
          textAnchor="middle"
          className="fill-muted-foreground"
          style={{ fontSize: 13 }}
        >
          {total}
        </text>
      </>
    );
  };

  return (
    <Card className="rounded-xl border shadow-sm">
      <CardHeader>
        <CardTitle className="text-lg">Employee vs Dependents</CardTitle>
        <CardDescription>Distribution of patients by category</CardDescription>
      </CardHeader>

      <CardContent className="pt-4">
        {loading ? (
          <div className="h-[420px]">
            <Skeleton className="mb-3 h-4 w-48" />
            <Skeleton className="h-[360px] w-full" />
          </div>
        ) : error ? (
          <Alert variant="destructive">
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : data.length === 0 ? (
          <PlaceholderPattern />
        ) : (
          <div className="h-[420px]">
            <ResponsiveContainer width="100%" height="100%">
              <PieChart>
                <Tooltip content={<CustomTooltip />} />
                <Legend verticalAlign="bottom" height={28} />
                <Pie
                  data={data}
                  dataKey="count"
                  nameKey="category"
                  cx="50%"
                  cy="50%"
                  innerRadius={78}   // was 70
                  outerRadius={135}  // was 110
                  stroke="hsl(var(--border))"
                  strokeWidth={1}
                  labelLine={false}
                >
                  {data.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                  {renderCenterLabel()}
                </Pie>
              </PieChart>
            </ResponsiveContainer>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
