// GenderDistributionChart.tsx
import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';
import { API } from '@/config';
import axios from 'axios';
import { useEffect, useMemo, useState } from 'react';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Skeleton } from '@/components/ui/skeleton';

// recharts
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  Cell,
  LabelList,
} from 'recharts';

interface GenderDistribution {
  gender: string;
  count: number;
}

type TooltipProps = {
  active?: boolean;
  payload?: Array<{ value: number; name: string; payload: { gender: string; count: number } }>;
};

export function GenderDistributionChart() {
  const [data, setData] = useState<GenderDistribution[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const res = await axios.get<GenderDistribution[]>(`${API}/patients/gender-distribution`);
        setData(res.data || []);
        setError(null);
      } catch (e) {
        console.error(e);
        setError('Failed to fetch gender distribution data');
      } finally {
        setLoading(false);
      }
    })();
  }, []);

  // Normalize and compute total
  const normalized = useMemo<GenderDistribution[]>(
    () =>
      (data || []).map((d) => ({
        gender: d.gender?.trim() ? d.gender : 'Unknown',
        count: Number(d.count ?? 0),
      })),
    [data]
  );

  const total = useMemo(() => normalized.reduce((sum, d) => sum + (d.count ?? 0), 0), [normalized]);

  // Light blue palette
  const COLORS = ['#60A5FA', '#93C5FD', '#BFDBFE', '#DBEAFE', '#3B82F6', '#1D4ED8'];

  const CustomTooltip = ({ active, payload }: TooltipProps) => {
    if (!active || !payload?.length) return null;
    const p = payload[0];
    const count = p.value ?? 0;
    const name = p.payload?.gender ?? p.name;
    const pct = total ? Math.round((Number(count) / total) * 100) : 0;

    return (
      <div className="rounded-md border bg-popover px-3 py-2 text-sm shadow-sm">
        <div className="font-medium">{name}</div>
        <div className="text-muted-foreground">Count: {count}</div>
        <div className="text-muted-foreground">{pct}%</div>
      </div>
    );
  };

  return (
    <Card className="rounded-xl border shadow-sm">
      <CardHeader>
        <CardTitle className="text-lg">Gender Distribution</CardTitle>
        <CardDescription>
          Share of patients by gender{total ? ` · Total: ${total}` : ''}
        </CardDescription>
      </CardHeader>

      <CardContent className="pt-4">
        {loading ? (
          <div className="h-[420px]">
            <Skeleton className="mb-3 h-4 w-48" />
            <Skeleton className="h-[360px] w-full" />
          </div>
        ) : error ? (
          <Alert variant="destructive">
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : normalized.length === 0 ? (
          <PlaceholderPattern />
        ) : (
          <div className="h-[420px]">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart
                data={normalized}
                layout="vertical"
                margin={{ top: 8, right: 24, bottom: 24, left: 24 }}
                barCategoryGap={16}
              >
                <CartesianGrid strokeDasharray="3 3" vertical={false} />
                <XAxis
                  type="number"
                  tickLine={false}
                  axisLine={{ stroke: 'hsl(var(--border))' }}
                />
                <YAxis
                  dataKey="gender"
                  type="category"
                  width={100}
                  tickLine={false}
                  axisLine={{ stroke: 'hsl(var(--border))' }}
                />
                <Tooltip content={<CustomTooltip />} />
                <Legend verticalAlign="bottom" height={28} />

                <Bar dataKey="count" name="Count" radius={[6, 6, 6, 6]}>
                  {normalized.map((_, idx) => (
                    <Cell key={`cell-${idx}`} fill={COLORS[idx % COLORS.length]} />
                  ))}
                  <LabelList
                    dataKey="count"
                    position="right"
                    className="fill-foreground"
                    formatter={(v: number) => (v ?? 0).toString()}
                  />
                </Bar>
              </BarChart>
            </ResponsiveContainer>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
