import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';
import { API } from '@/config';
import axios from 'axios';
import { useEffect, useMemo, useState } from 'react';
import {
  ResponsiveContainer,
  LineChart,
  Line,
  XAxis,
  YAxis,
  Tooltip,
  CartesianGrid,
  Legend,
} from 'recharts';
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';

interface InjuryOnDutyData {
  company_id: number;
  company_name: string;
  months: Array<{
    month: string;
    count: number;
  }>;
}

interface Company {
  id: number;
  name: string;
}

interface Clinic {
  id: number;
  name: string;
}

export default function InjuryOnDutyChart() {
  const [companies, setCompanies] = useState<Company[]>([]);
  const [clinics, setClinics] = useState<Clinic[]>([]);
  const [injuryData, setInjuryData] = useState<InjuryOnDutyData | null>(null);
  const [year, setYear] = useState<number>(new Date().getFullYear());
  const [companyId, setCompanyId] = useState<number | null>(null); // no default
  const [clinicId, setClinicId] = useState<number | null>(null);   // no default
  const [loading, setLoading] = useState<boolean>(false);
  const [loadingCompanies, setLoadingCompanies] = useState<boolean>(true);
  const [loadingClinics, setLoadingClinics] = useState<boolean>(true);

  // Fetch companies on mount (no auto-select)
  useEffect(() => {
    setLoadingCompanies(true);
    axios
      .get<Company[]>(`${API}/api/companies`)
      .then((res) => setCompanies(res.data ?? []))
      .finally(() => setLoadingCompanies(false));
  }, []);

  // Fetch clinics on mount (no auto-select)
  useEffect(() => {
    setLoadingClinics(true);
    axios
      .get<Clinic[]>(`${API}/reports/clinics`)
      .then((res) => setClinics(res.data ?? []))
      .finally(() => setLoadingClinics(false));
  }, []);

  // Fetch IOD data when filters change (requires both selections)
  useEffect(() => {
    if (companyId == null || clinicId == null) {
      setInjuryData(null);
      return;
    }
    setLoading(true);
    axios
      .get<InjuryOnDutyData[]>(`${API}/consultations/injury-on-duty-cases`, {
        params: { year, company_id: companyId, clinic_id: clinicId },
      })
      .then((res) => {
        // API returns an array with one object per company
        setInjuryData(res.data?.[0] ?? null);
      })
      .finally(() => setLoading(false));
  }, [year, companyId, clinicId]);

  // Build chart data
  const chartData = useMemo(() => {
    if (!injuryData) return [];
    return injuryData.months.map((m) => ({
      month: m.month,
      count: m.count,
    }));
  }, [injuryData]);

  // Colors (blue theme)
  const stroke = '#2563eb'; // blue-600
  const gridColor = 'rgba(148, 163, 184, 0.25)'; // slate-400/25
  const axisColor = '#475569'; // slate-600

  return (
    <Card className="col-span-12 md:col-span-6 rounded-xl border shadow-sm">
      <CardHeader className="pb-2">
        <CardTitle className="text-base">Injury on Duty Cases</CardTitle>
      </CardHeader>

      <CardContent>
        {/* Controls */}
        <div className="mb-4 flex flex-wrap items-center justify-end gap-3">
          <div className="flex items-center gap-2">
            <label className="text-sm text-muted-foreground">Year:</label>
            <select
              value={year}
              onChange={(e) => setYear(parseInt(e.target.value))}
              className="rounded-md border border-border bg-background p-1.5 text-sm"
            >
              {Array.from({ length: 10 }, (_, i) => new Date().getFullYear() - i).map((y) => (
                <option key={y} value={y}>
                  {y}
                </option>
              ))}
            </select>
          </div>

          <div className="flex items-center gap-2">
            <label className="text-sm text-muted-foreground">Company:</label>
            <select
              value={companyId ?? ''}
              onChange={(e) => setCompanyId(e.target.value ? parseInt(e.target.value, 10) : null)}
              className="min-w-[200px] rounded-md border border-border bg-background p-1.5 text-sm"
            >
              <option value="" disabled>
                {loadingCompanies ? 'Loading…' : 'Select a company'}
              </option>
              {companies.map((c) => (
                <option key={c.id} value={c.id}>
                  {c.name}
                </option>
              ))}
            </select>
          </div>

          <div className="flex items-center gap-2">
            <label className="text-sm text-muted-foreground">Clinic:</label>
            <select
              value={clinicId ?? ''}
              onChange={(e) => setClinicId(e.target.value ? parseInt(e.target.value, 10) : null)}
              className="min-w-[200px] rounded-md border border-border bg-background p-1.5 text-sm"
            >
              <option value="" disabled>
                {loadingClinics ? 'Loading…' : 'Select a clinic'}
              </option>
              {clinics.map((c) => (
                <option key={c.id} value={c.id}>
                  {c.name}
                </option>
              ))}
            </select>
          </div>
        </div>

        {/* Chart */}
        {loading ? (
          <div className="flex h-72 items-center justify-center">
            <div className="h-12 w-12 animate-spin rounded-full border-b-2 border-t-2 border-primary" />
          </div>
        ) : chartData.length > 0 ? (
          <div className="h-[350px] w-full">
            <ResponsiveContainer width="100%" height="100%">
              <LineChart data={chartData} margin={{ left: 8, right: 8, top: 10, bottom: 0 }}>
                <CartesianGrid stroke={gridColor} vertical={false} />
                <XAxis dataKey="month" stroke={axisColor} tickLine={false} axisLine={false} />
                <YAxis stroke={axisColor} tickLine={false} axisLine={false} allowDecimals={false} />
                <Tooltip
                  contentStyle={{
                    background: 'hsl(var(--popover))',
                    border: '1px solid hsl(var(--border))',
                    borderRadius: 8,
                    color: 'hsl(var(--popover-foreground))',
                  }}
                />
                <Legend />
                <Line
                  type="monotone"
                  dataKey="count"
                  name={injuryData?.company_name || 'Cases'}
                  stroke={stroke}
                  strokeWidth={2}
                  dot={{ r: 3 }}
                  activeDot={{ r: 5 }}
                />
              </LineChart>
            </ResponsiveContainer>
          </div>
        ) : (
          <PlaceholderPattern />
        )}
      </CardContent>
    </Card>
  );
}
