// ReferralChart.tsx
import { useEffect, useMemo, useState } from 'react';
import axios from 'axios';
import { API } from '@/config';
import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Skeleton } from '@/components/ui/skeleton';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

// recharts
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
} from 'recharts';

interface ReferralCount {
  type: string;
  count: number;
}
interface ReferralData {
  month: string | number; // API may send "January" or 1..12 or "2025-09"
  counts: ReferralCount[];
}
interface Clinic {
  id: number;
  name: string;
}

const MONTHS_SHORT = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
const MONTHS_FULL  = ['January','February','March','April','May','June','July','August','September','October','November','December'];

function monthToIndex(m: string | number): number {
  if (typeof m === 'number') return Math.min(Math.max(m - 1, 0), 11);
  const s = String(m);
  // handle "YYYY-MM" or "YYYY-M"
  const mm = s.match(/^\d{4}[-/](\d{1,2})/);
  if (mm) {
    const n = Number(mm[1]);
    return n >= 1 && n <= 12 ? n - 1 : 0;
  }
  const byFull = MONTHS_FULL.findIndex(x => x.toLowerCase() === s.toLowerCase());
  if (byFull !== -1) return byFull;
  const byShort = MONTHS_SHORT.findIndex(x => x.toLowerCase() === s.slice(0,3).toLowerCase());
  return byShort !== -1 ? byShort : 0;
}

// Safe object key for Recharts dataKey
function keyOfType(label: string): string {
  return label.toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/^_+|_+$/g, '');
}

// 🔵 Blue-only palette (light → dark)
const BLUES = [
  '#DBEAFE', // blue-100
  '#BFDBFE', // blue-200
  '#93C5FD', // blue-300
  '#60A5FA', // blue-400
  '#3B82F6', // blue-500
  '#2563EB', // blue-600
  '#1D4ED8', // blue-700
  '#1E40AF', // blue-800
];

export function ReferralChart() {
  const [data, setData] = useState<ReferralData[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);
  const [year, setYear] = useState<number>(new Date().getFullYear());
  const [clinics, setClinics] = useState<Clinic[]>([]);
  const [selectedClinic, setSelectedClinic] = useState<number | null>(null); // 👈 stays null by default

  // Load clinics once (NO default selection)
  useEffect(() => {
    (async () => {
      try {
        const res = await axios.get<Clinic[]>(`${API}/reports/clinics`);
        setClinics(res.data || []);
        // intentionally not selecting a default clinic
      } catch (e) {
        console.error(e);
      }
    })();
  }, []);

  // Load referrals when filters change
  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const res = await axios.get<ReferralData[]>(
          `${API}/consultations/referrals-per-month`,
          { params: { year, clinic_id: selectedClinic ?? undefined } } // only include clinic_id if chosen
        );
        setData(res.data || []);
        setError(null);
      } catch (e) {
        console.error(e);
        setError('Failed to fetch referral data');
      } finally {
        setLoading(false);
      }
    })();
  }, [year, selectedClinic]);

  // Collect unique types (stable order)
  const seriesMeta = useMemo(() => {
    const types = Array.from(
      new Set(
        data.flatMap((m) => (m.counts || []).map((c) => c.type))
      )
    );
    return types.map((type, i) => ({
      type,
      key: keyOfType(type) || `type_${i}`,
      color: BLUES[i % BLUES.length],
    }));
  }, [data]);

  // Normalize to 12 months with stacked values per type
  const chartData = useMemo(() => {
    const base = Array.from({ length: 12 }, (_, i) => {
      const row: Record<string, any> = { label: MONTHS_SHORT[i] };
      for (const s of seriesMeta) row[s.key] = 0;
      return row;
    });

    for (const monthRow of data) {
      const idx = monthToIndex(monthRow.month);
      for (const c of (monthRow.counts || [])) {
        const meta = seriesMeta.find((s) => s.type === c.type);
        if (meta) base[idx][meta.key] = c.count ?? 0;
      }
    }
    return base;
  }, [data, seriesMeta]);

  const hasAnyData = useMemo(
    () => chartData.some((row) => seriesMeta.some((s) => row[s.key] > 0)),
    [chartData, seriesMeta]
  );

  // Axis/tooltip styling with subtle blues
  const BLUE_TICK = '#60A5FA'; // blue-400
  const BLUE_GRID = '#BFDBFE'; // blue-200

  return (
    <Card className="rounded-xl border shadow-sm">
      <CardHeader className="pb-3">
        <CardTitle className="text-lg">Referrals Per Month</CardTitle>
        <CardDescription>Stacked counts by referral type, per month</CardDescription>
      </CardHeader>

      <CardContent className="pt-0">
        {/* Filters */}
        <div className="mb-4 flex flex-col items-start gap-3 sm:flex-row sm:items-end sm:justify-end">
          <div className="w-full min-w-[220px] sm:w-auto">
            <Label className="mb-1 block text-sm">Clinic</Label>
            <Select
              value={selectedClinic !== null ? String(selectedClinic) : 'all'}
              onValueChange={(val) => setSelectedClinic(val === 'all' ? null : Number(val))}
            >
              <SelectTrigger className="w-full sm:w-[220px]">
                <SelectValue placeholder="All Clinics" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Clinics</SelectItem>
                {clinics.map((c) => (
                  <SelectItem key={c.id} value={String(c.id)}>
                    {c.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="w-full min-w-[140px] sm:w-auto">
            <Label className="mb-1 block text-sm">Year</Label>
            <Select value={String(year)} onValueChange={(v) => setYear(Number(v))}>
              <SelectTrigger className="w-full sm:w-[140px]">
                <SelectValue placeholder="Year" />
              </SelectTrigger>
              <SelectContent>
                {Array.from({ length: 10 }, (_, i) => new Date().getFullYear() - i).map((y) => (
                  <SelectItem key={y} value={String(y)}>
                    {y}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* Chart */}
        {loading ? (
          <div className="h-[350px]">
            <Skeleton className="mb-3 h-4 w-48" />
            <Skeleton className="h-[300px] w-full" />
          </div>
        ) : error ? (
          <Alert variant="destructive">
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : hasAnyData ? (
          <div className="h-[350px]">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={chartData}>
                <CartesianGrid stroke={BLUE_GRID} strokeDasharray="3 3" strokeOpacity={0.25} />
                <XAxis
                  dataKey="label"
                  tick={{ fill: BLUE_TICK }}
                  tickLine={false}
                  axisLine={false}
                />
                <YAxis
                  allowDecimals={false}
                  tick={{ fill: BLUE_TICK }}
                  tickLine={false}
                  axisLine={false}
                />
                <Tooltip
                  contentStyle={{
                    background: 'hsl(var(--popover))',
                    border: '1px solid hsl(var(--border))',
                    borderRadius: 8,
                  }}
                  labelStyle={{ color: 'hsl(var(--foreground))' }}
                  itemStyle={{ color: 'hsl(var(--foreground))' }}
                />
                <Legend />
                {seriesMeta.map((s) => (
                  <Bar
                    key={s.key}
                    dataKey={s.key}
                    name={s.type}
                    stackId="referrals"
                    fill={s.color}
                    radius={[4, 4, 0, 0]}
                  />
                ))}
              </BarChart>
            </ResponsiveContainer>
          </div>
        ) : (
          <PlaceholderPattern />
        )}
      </CardContent>
    </Card>
  );
}
