// SickLeaveChart.tsx
import { useEffect, useMemo, useState } from 'react';
import axios from 'axios';
import { API } from '@/config';
import { PlaceholderPattern } from '@/components/ui/placeholder-pattern';

// shadcn/ui
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Skeleton } from '@/components/ui/skeleton';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

// recharts
import {
  ResponsiveContainer,
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
} from 'recharts';

interface SickLeaveData {
  month: string | number; // API may return "January" or 1..12
  count: number;
}
interface Clinic {
  id: number;
  name: string;
}

const MONTHS_SHORT = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
const MONTHS_FULL  = ['January','February','March','April','May','June','July','August','September','October','November','December'];

function monthToIndex(m: string | number): number {
  if (typeof m === 'number') return Math.min(Math.max(m - 1, 0), 11);
  const byFull = MONTHS_FULL.findIndex(x => x.toLowerCase() === m.toLowerCase());
  if (byFull !== -1) return byFull;
  const byShort = MONTHS_SHORT.findIndex(x => x.toLowerCase() === String(m).slice(0,3).toLowerCase());
  return byShort !== -1 ? byShort : 0;
}

export function SickLeaveChart() {
  const [data, setData] = useState<SickLeaveData[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);
  const [year, setYear] = useState<number>(new Date().getFullYear());
  const [clinics, setClinics] = useState<Clinic[]>([]);
  const [selectedClinic, setSelectedClinic] = useState<number | null>(null); // <- stays null by default

  // fetch clinics once (NO default selection)
  useEffect(() => {
    (async () => {
      try {
        const res = await axios.get<Clinic[]>(`${API}/reports/clinics`);
        setClinics(res.data || []);
        // intentionally NOT selecting any clinic here
      } catch (e) {
        console.error(e);
      }
    })();
  }, []);

  // fetch sick-leave series when filters change
  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const res = await axios.get<SickLeaveData[]>(
          `${API}/consultations/sick-leave-per-month`,
          {
            params: {
              // The backend will ignore `year` when clinic_id is null (all years aggregated),
              // but we still send it—it’s harmless.
              year,
              clinic_id: selectedClinic ?? undefined, // only include if a clinic is chosen
            },
          }
        );
        setData(res.data || []);
        setError(null);
      } catch (e) {
        console.error(e);
        setError('Failed to fetch sick leave data');
      } finally {
        setLoading(false);
      }
    })();
  }, [year, selectedClinic]);

  // normalize data into 12 fixed buckets (Jan..Dec)
  const chartData = useMemo(() => {
    const buckets = Array.from({ length: 12 }, (_, i) => ({ label: MONTHS_SHORT[i], count: 0 }));
    for (const d of data) {
      const idx = monthToIndex(d.month);
      buckets[idx].count = d.count ?? 0;
    }
    return buckets;
  }, [data]);

  const LINE_COLOR = '#3b82f6'; // blue-500

  return (
    <Card className="rounded-xl border shadow-sm">
      <CardHeader className="pb-3">
        <CardTitle className="text-lg">Sick Leave Issued Per Month</CardTitle>
        <CardDescription>Monthly count of consultations with sick leave</CardDescription>
      </CardHeader>

      {/* Filters */}
      <CardContent className="pt-0">
        <div className="mb-4 flex flex-col items-start gap-3 sm:flex-row sm:items-end sm:justify-end">
          <div className="w-full min-w-[200px] sm:w-auto">
            <Label className="mb-1 block text-sm">Clinic</Label>
            <Select
              value={selectedClinic !== null ? String(selectedClinic) : 'all'}
              onValueChange={(val) => setSelectedClinic(val === 'all' ? null : Number(val))}
            >
              <SelectTrigger className="w-full sm:w-[220px]">
                <SelectValue placeholder="All Clinics" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Clinics</SelectItem>
                {clinics.map((c) => (
                  <SelectItem key={c.id} value={String(c.id)}>
                    {c.name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="w-full min-w-[140px] sm:w-auto">
            <Label className="mb-1 block text-sm">Year</Label>
            <Select
              value={String(year)}
              onValueChange={(val) => setYear(Number(val))}
            >
              <SelectTrigger className="w-full sm:w-[140px]">
                <SelectValue placeholder="Year" />
              </SelectTrigger>
              <SelectContent>
                {Array.from({ length: 10 }, (_, i) => new Date().getFullYear() - i).map((y) => (
                  <SelectItem key={y} value={String(y)}>
                    {y}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* Chart area */}
        {loading ? (
          <div className="h-[350px]">
            <Skeleton className="mb-3 h-4 w-48" />
            <Skeleton className="h-[300px] w-full" />
          </div>
        ) : error ? (
          <Alert variant="destructive">
            <AlertTitle>Error</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : chartData.some((d) => d.count > 0) ? (
          <div className="h-[350px]">
            <ResponsiveContainer width="100%" height="100%">
              <LineChart data={chartData}>
                <CartesianGrid strokeDasharray="3 3" strokeOpacity={0.2} />
                <XAxis dataKey="label" />
                <YAxis allowDecimals={false} />
                <Tooltip
                  contentStyle={{
                    background: 'hsl(var(--popover))',
                    border: '1px solid hsl(var(--border))',
                    borderRadius: 8,
                  }}
                  labelStyle={{ color: 'hsl(var(--foreground))' }}
                  itemStyle={{ color: 'hsl(var(--foreground))' }}
                />
                <Legend />
                <Line
                  type="monotone"
                  dataKey="count"
                  name="Sick Leave Issued"
                  stroke={LINE_COLOR}
                  strokeWidth={2}
                  dot={{ r: 3, stroke: LINE_COLOR, strokeWidth: 1, fill: '#fff' }}
                  activeDot={{ r: 5 }}
                />
              </LineChart>
            </ResponsiveContainer>
          </div>
        ) : (
          <PlaceholderPattern />
        )}
      </CardContent>
    </Card>
  );
}
