// hooks/useInactivityTimer.ts
import { router } from '@inertiajs/react';
import { useEffect } from 'react';
import Swal from 'sweetalert2';

export function useInactivityTimer(timeoutMinutes: number = 2, warningSeconds: number = 30) {
    useEffect(() => {
        const timeoutMs = timeoutMinutes * 60 * 1000;
        const warningMs = (timeoutMinutes * 60 - warningSeconds) * 1000;

        let inactivityTimer: NodeJS.Timeout;
        let warningTimer: NodeJS.Timeout;
        let countdownInterval: NodeJS.Timeout | null = null;

        const handleLogout = () => {
            // Perform the POST request to log out
            router.post(
                '/logout',
                {},
                {
                    onSuccess: () => {
                        // Redirect to login page after successful logout
                        router.visit('/login');
                    },
                    onError: (error) => {
                        // Check if error.status is a number before comparing
                        console.log("Error", error)
                        if (typeof error.status === 'number' && error.status === 419) {
                            // Redirect to login page if status is 419
                            router.visit('/login');
                        } else {
                            // Show error message for other status codes
                            Swal.fire('Error', 'Failed to logout automatically', 'error');
                        }
                    },
                },
            );
        };

        const showWarning = () => {
            let secondsLeft = warningSeconds;

            Swal.fire({
                title: 'Session Timeout Warning',
                html: `You will be logged out in <strong>${secondsLeft}</strong> seconds due to inactivity.`,
                icon: 'warning',
                timer: warningSeconds * 1000,
                timerProgressBar: true,
                showConfirmButton: true,
                confirmButtonText: 'Stay Logged In',
                allowOutsideClick: false,
                allowEscapeKey: false,
                didOpen: () => {
                    countdownInterval = setInterval(() => {
                        secondsLeft--;
                        const htmlContainer = Swal.getHtmlContainer();
                        if (htmlContainer) {
                            htmlContainer.innerHTML = `You will be logged out in <strong>${secondsLeft}</strong> seconds due to inactivity.`;
                        }
                    }, 1000);
                },
                willClose: () => {
                    if (countdownInterval) {
                        clearInterval(countdownInterval);
                    }
                },
            }).then((result) => {
                if (result.isConfirmed) {
                    // User clicked "Stay Logged In"
                    resetTimers();
                } else if (result.dismiss === Swal.DismissReason.timer) {
                    // Timer expired - logout
                    handleLogout();
                }
            });
        };

        const resetTimers = () => {
            // Clear all timers
            clearTimeout(inactivityTimer);
            clearTimeout(warningTimer);
            if (countdownInterval) {
                clearInterval(countdownInterval);
            }

            // Close any active SweetAlert
            Swal.close();

            // Set new timers
            warningTimer = setTimeout(showWarning, warningMs);
            inactivityTimer = setTimeout(() => {
                handleLogout();
            }, timeoutMs);
        };

        // Events that reset the timer
        const events = ['mousemove', 'keydown', 'scroll', 'click', 'touchstart', 'mousedown'];

        events.forEach((event) => {
            window.addEventListener(event, resetTimers);
        });

        // Initialize timers
        resetTimers();

        // Cleanup
        return () => {
            clearTimeout(inactivityTimer);
            clearTimeout(warningTimer);
            if (countdownInterval) {
                clearInterval(countdownInterval);
            }
            events.forEach((event) => {
                window.removeEventListener(event, resetTimers);
            });
            Swal.close();
        };
    }, [timeoutMinutes, warningSeconds]);
}
