import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { FormEvent } from 'react';
import * as Yup from 'yup';

interface CentralStoreData {
    name: string;
    location: string;
    address: string;
    latitude: string;
    longitude: string;
    description: string;
    [key: string]: any;
}

export default function CreateCentralStore() {
    const { data, setData, post, processing, errors, reset } = useForm<CentralStoreData>({
        name: '',
        location: '',
        address: '',
        latitude: '',
        longitude: '',
        description: '',
    });

    const schema = Yup.object().shape({
        name: Yup.string().required('Name is required'),
        location: Yup.string().nullable(),
        address: Yup.string().nullable(),
        latitude: Yup.string().nullable(),
        longitude: Yup.string().nullable(),
        description: Yup.string().nullable(),
    });

    const handleSubmit = async (e: FormEvent) => {
        e.preventDefault();
        try {
            await schema.validate(data, { abortEarly: false });
            post(route('central-stores.store'), {
                preserveScroll: true,
                onSuccess: () => reset(),
            });
        } catch (validationError: any) {
            const fieldErrors: Record<string, string> = {};
            validationError.inner.forEach((err: any) => {
                if (err.path) fieldErrors[err.path] = err.message;
            });
            // Optionally handle manual error state
        }
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Central Stores', href: route('central-stores.index') },
        { title: 'Create Central Store', href: route('central-stores.create') },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Create Central Store" />

            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                <h1 className="text-foreground text-2xl font-bold">Create Central Store</h1>

                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                        {[
                            'name',
                            'location',
                            'address',
                            'latitude',
                            'longitude',
                            'description',
                        ].map((field) => (
                            <div key={field} className="grid gap-2">
                                <Label htmlFor={field}>
                                    {field.replace(/_/g, ' ').replace(/\b\w/g, (l) => l.toUpperCase())}
                                    {field === 'name' && <span className="text-red-500">*</span>}
                                </Label>
                                <Input
                                    id={field}
                                    name={field}
                                    value={data[field]}
                                    onChange={(e) => setData(field, e.target.value)}
                                />
                                <InputError message={errors[field]} />
                            </div>
                        ))}
                    </div>

                    <div className="flex justify-end gap-3">
                        <Button type="button" variant="outline" onClick={() => window.history.back()}>
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Saving...' : 'Save Central Store'}
                        </Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
