// resources/js/Pages/CentralStores/EditBatch.tsx

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Calendar } from '@/components/ui/calendar';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { cn } from '@/lib/utils';
import { format } from 'date-fns';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, Link, usePage, router } from '@inertiajs/react';
import { useEffect, useMemo, useState } from 'react';
import Select from 'react-select';
import { CalendarIcon, FileText, Upload, X, Save } from 'lucide-react';
import { useDropzone } from 'react-dropzone';

type OptionType = {
  value: number;
  label: string;
};

type FormData = {
  medication_id: string;
  supplier_id: string;
  batch_number: string;
  quantity: string;
  expiry_date: string;
  manufacture_date: string;
  received_date: string;
  invoice_number: string;
  price: string;
  invoice_file: File | null;
};

export default function EditBatch() {
  const { centralStore, batch, medications, suppliers } = usePage().props as any;

  // Dark mode detection to theme react-select
  const [isDarkMode, setIsDarkMode] = useState(false);
  useEffect(() => {
    const checkDarkMode = () => setIsDarkMode(document.documentElement.classList.contains('dark'));
    checkDarkMode();
    const observer = new MutationObserver(checkDarkMode);
    observer.observe(document.documentElement, { attributes: true, attributeFilter: ['class'] });
    return () => observer.disconnect();
  }, []);

  // Build options
  const medicationOptions: OptionType[] = useMemo(
    () =>
      medications.map((m: any) => {
        const dosage = m.dosage ?? '';
        const unit = m.unit ?? '';
        const form = m.form ?? '';
        const labelParts = [m.name, dosage, unit, form].filter(Boolean);
        const label = labelParts.join(' ').trim();
        return { value: m.id, label: label || ' ' };
      }),
    [medications]
  );

  const supplierOptions: OptionType[] = useMemo(
    () =>
      suppliers.map((s: any) => ({
        value: s.id,
        label: s.name,
      })),
    [suppliers]
  );

  // Initialize dates from batch
  const parseDate = (d?: string | null) => (d ? new Date(d) : undefined);
  const [expiryDate, setExpiryDate] = useState<Date | undefined>(parseDate(batch?.expiry_date));
  const [manufactureDate, setManufactureDate] = useState<Date | undefined>(parseDate(batch?.manufacture_date));
  const [receivedDate, setReceivedDate] = useState<Date | undefined>(parseDate(batch?.received_date));

  // Inertia form with prefilled values
  const { data, setData, post, processing, errors } = useForm<FormData>({
    medication_id: String(batch?.medication_id ?? ''),
    supplier_id: String(batch?.supplier_id ?? ''),
    batch_number: String(batch?.batch_number ?? ''),
    quantity: String(batch?.quantity ?? ''),
    expiry_date: batch?.expiry_date ?? '',
    manufacture_date: batch?.manufacture_date ?? '',
    received_date: batch?.received_date ?? '',
    invoice_number: String(batch?.invoice_number ?? ''),
    price: batch?.price != null ? String(batch.price) : '',
    invoice_file: null,
  });

  const selectStyles = {
    control: (base: any) => ({
      ...base,
      backgroundColor: isDarkMode ? '#000' : '#fff',
      borderColor: isDarkMode ? '#374151' : '#d1d5db',
      color: isDarkMode ? '#fff' : '#000',
    }),
    menu: (base: any) => ({
      ...base,
      backgroundColor: isDarkMode ? '#000' : '#fff',
    }),
    option: (base: any, { isFocused }: any) => ({
      ...base,
      backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
      color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
    }),
    singleValue: (base: any) => ({
      ...base,
      color: isDarkMode ? '#fff' : '#000',
    }),
    input: (base: any) => ({
      ...base,
      color: isDarkMode ? '#fff' : '#000',
    }),
    placeholder: (base: any) => ({
      ...base,
      color: isDarkMode ? '#9ca3af' : '#6b7280',
    }),
  };

  // Dropzone for optional invoice replacement
  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    accept: {
      'application/pdf': ['.pdf'],
      'image/*': ['.png', '.jpg', '.jpeg'],
    },
    maxFiles: 1,
    onDrop: (acceptedFiles: File[]) => {
      if (acceptedFiles.length > 0) {
        setData('invoice_file', acceptedFiles[0]);
      }
    },
  });

  const removeFile = () => setData('invoice_file', null);

  const handleDateChange = (date: Date | undefined, field: 'expiry_date' | 'manufacture_date' | 'received_date') => {
    const dateString = date ? format(date, 'yyyy-MM-dd') : '';
    setData(field, dateString);
    if (field === 'expiry_date') setExpiryDate(date);
    if (field === 'manufacture_date') setManufactureDate(date);
    if (field === 'received_date') setReceivedDate(date);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    // Build FormData to support file upload + PUT override
    const formData = new FormData();
    formData.append('medication_id', data.medication_id);
    formData.append('supplier_id', data.supplier_id);
    formData.append('batch_number', data.batch_number);
    formData.append('quantity', data.quantity);
    formData.append('expiry_date', data.expiry_date);
    formData.append('manufacture_date', data.manufacture_date);
    formData.append('received_date', data.received_date);
    formData.append('invoice_number', data.invoice_number);
    formData.append('price', data.price);
    if (data.invoice_file) formData.append('invoice_file', data.invoice_file);
    formData.append('_method', 'PUT');

    router.post(
      route('central-stores.received-batches.update', { centralStore: centralStore.id, batch: batch.id }),
      formData,
      {
        forceFormData: true,
        onSuccess: () => {
          // optional success handling
        },
      }
    );
  };

  const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Central Stores', href: route('central-stores.index') },
    { title: centralStore.name, href: route('central-stores.show', centralStore.id) },
    { title: `Edit Batch #${batch.batch_number}`, href: '#' },
  ];

  // Helper: build public URL for existing invoice file (if you store in 'public' disk)
  const existingInvoiceUrl = batch?.invoice_file ? `/storage/${batch.invoice_file}` : null;
  const existingInvoiceName =
    batch?.invoice_file ? String(batch.invoice_file).split('/').slice(-1)[0] : null;

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`Edit Batch #${batch.batch_number}`} />

      <div className="grid grid-cols-12 gap-8 px-4 py-6 sm:px-6">
        <form onSubmit={handleSubmit} className="col-span-12 space-y-6 lg:col-span-8">
          <div className="rounded-lg border bg-card p-6 shadow-sm dark:bg-neutral-900">
            <div className="mb-4 flex items-center justify-between">
              <h2 className="text-xl font-semibold">Edit Medication Batch</h2>
              <div className="flex gap-2">
                <Button asChild variant="outline">
                  <Link href={route('central-stores.received-batches.index', centralStore.id)}>
                    View Received Batches
                  </Link>
                </Button>
              </div>
            </div>

            {/* Note about uniqueness */}
            <div className="mb-4 rounded-md border border-amber-300 bg-amber-50 p-3 text-sm text-amber-800 dark:border-amber-600/50 dark:bg-amber-900/20 dark:text-amber-200">
              <strong>Note:</strong> The <em>Batch Number</em> should be <strong>unique</strong> for the same
              medication and supplier within this central store. Editing into an identity that already exists will be blocked.
            </div>

            <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
              {/* Medication & Supplier */}
              <div className="space-y-2">
                <Label htmlFor="medication_id">Medication</Label>
                <Select
                  id="medication_id"
                  options={medicationOptions}
                  value={medicationOptions.find((opt) => opt.value === Number(data.medication_id))}
                  onChange={(option) => setData('medication_id', option?.value.toString() ?? '')}
                  placeholder="Select medication..."
                  styles={selectStyles}
                />
                {errors.medication_id && <p className="text-sm text-red-500">{errors.medication_id}</p>}
              </div>

              <div className="space-y-2">
                <Label htmlFor="supplier_id">Supplier</Label>
                <Select
                  id="supplier_id"
                  options={supplierOptions}
                  value={supplierOptions.find((opt) => opt.value === Number(data.supplier_id))}
                  onChange={(option) => setData('supplier_id', option?.value.toString() ?? '')}
                  placeholder="Select supplier..."
                  styles={selectStyles}
                />
                {errors.supplier_id && <p className="text-sm text-red-500">{errors.supplier_id}</p>}
              </div>

              {/* Batch Number & Quantity */}
              <div className="space-y-2">
                <Label htmlFor="batch_number">Batch Number</Label>
                <Input
                  id="batch_number"
                  value={data.batch_number}
                  onChange={(e) => setData('batch_number', e.target.value)}
                  placeholder="e.g., BATCH-001-2025"
                />
                {errors.batch_number && <p className="text-sm text-red-500">{errors.batch_number}</p>}
              </div>

              <div className="space-y-2">
                <Label htmlFor="quantity">Quantity</Label>
                <Input
                  id="quantity"
                  type="number"
                  value={data.quantity}
                  onChange={(e) => setData('quantity', e.target.value)}
                />
                {errors.quantity && <p className="text-sm text-red-500">{errors.quantity}</p>}
              </div>

              {/* Dates */}
              <div className="space-y-2">
                <Label>Expiry Date</Label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      className={cn('w-full justify-start text-left font-normal', !expiryDate && 'text-muted-foreground')}
                    >
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {expiryDate ? format(expiryDate, 'PPP') : <span>Pick a date</span>}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0">
                    <Calendar
                      mode="single"
                      selected={expiryDate}
                      onSelect={(date) => handleDateChange(date, 'expiry_date')}
                      initialFocus
                    />
                  </PopoverContent>
                </Popover>
                {errors.expiry_date && <p className="text-sm text-red-500">{errors.expiry_date}</p>}
              </div>

              <div className="space-y-2">
                <Label>Manufacture Date</Label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      className={cn(
                        'w-full justify-start text-left font-normal',
                        !manufactureDate && 'text-muted-foreground'
                      )}
                    >
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {manufactureDate ? format(manufactureDate, 'PPP') : <span>Pick a date</span>}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0">
                    <Calendar
                      mode="single"
                      selected={manufactureDate}
                      onSelect={(date) => handleDateChange(date, 'manufacture_date')}
                      initialFocus
                    />
                  </PopoverContent>
                </Popover>
                {errors.manufacture_date && <p className="text-sm text-red-500">{errors.manufacture_date}</p>}
              </div>

              <div className="space-y-2">
                <Label>Received Date</Label>
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      variant="outline"
                      className={cn('w-full justify-start text-left font-normal', !receivedDate && 'text-muted-foreground')}
                    >
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {receivedDate ? format(receivedDate, 'PPP') : <span>Pick a date</span>}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0">
                    <Calendar
                      mode="single"
                      selected={receivedDate}
                      onSelect={(date) => handleDateChange(date, 'received_date')}
                      initialFocus
                    />
                  </PopoverContent>
                </Popover>
                {errors.received_date && <p className="text-sm text-red-500">{errors.received_date}</p>}
              </div>

              {/* Invoice Number & Price */}
              <div className="space-y-2">
                <Label htmlFor="invoice_number">Invoice Number</Label>
                <Input
                  id="invoice_number"
                  value={data.invoice_number}
                  onChange={(e) => setData('invoice_number', e.target.value)}
                />
                {errors.invoice_number && <p className="text-sm text-red-500">{errors.invoice_number}</p>}
              </div>

              <div className="space-y-2">
                <Label htmlFor="price">Price</Label>
                <Input
                  id="price"
                  type="number"
                  step="0.01"
                  value={data.price}
                  onChange={(e) => setData('price', e.target.value)}
                />
                {errors.price && <p className="text-sm text-red-500">{errors.price}</p>}
              </div>

              {/* Existing Invoice (if any) */}
              <div className="col-span-full space-y-2">
                <Label>Existing Invoice</Label>
                {existingInvoiceUrl ? (
                  <div className="flex items-center justify-between rounded-lg border bg-gray-50 p-3 dark:bg-gray-900">
                    <div className="flex items-center gap-2">
                      <FileText className="h-5 w-5 text-blue-500" />
                      <a
                        href={existingInvoiceUrl}
                        target="_blank"
                        rel="noreferrer"
                        className="text-sm underline"
                      >
                        {existingInvoiceName}
                      </a>
                    </div>
                    <span className="text-xs text-muted-foreground">
                      Uploading a new file will replace this one.
                    </span>
                  </div>
                ) : (
                  <p className="text-sm text-muted-foreground">No invoice file currently attached.</p>
                )}
              </div>

              {/* File Upload (optional replace) */}
              <div className="col-span-full space-y-2">
                <Label htmlFor="invoice_file">Replace Invoice File (optional)</Label>
                <div
                  {...getRootProps()}
                  className={cn(
                    'cursor-pointer rounded-lg border-2 border-dashed p-6 text-center transition-colors',
                    isDragActive ? 'border-blue-500 bg-blue-50 dark:bg-blue-950' : 'border-gray-300 dark:border-gray-600',
                    'hover:border-gray-400 dark:hover:border-gray-500'
                  )}
                >
                  <input {...getInputProps()} />
                  <Upload className="mx-auto mb-2 h-8 w-8 text-gray-400" />
                  <p className="text-sm text-gray-600 dark:text-gray-400">
                    {isDragActive ? 'Drop the file here' : 'Drag & drop a file here, or click to select'}
                  </p>
                  <p className="mt-1 text-xs text-gray-500 dark:text-gray-500">PDF, PNG, JPG up to 10MB</p>
                </div>

                {data.invoice_file && (
                  <div className="mt-3 flex items-center justify-between rounded-lg border bg-gray-50 p-3 dark:bg-gray-900">
                    <div className="flex items-center gap-2">
                      <FileText className="h-5 w-5 text-blue-500" />
                      <span className="text-sm">{data.invoice_file.name}</span>
                    </div>
                    <Button type="button" variant="ghost" size="sm" onClick={removeFile} className="h-8 w-8 p-0">
                      <X className="h-4 w-4" />
                    </Button>
                  </div>
                )}

                {errors.invoice_file && <p className="text-sm text-red-500">{errors.invoice_file}</p>}
              </div>
            </div>
          </div>

          <div className="flex justify-end gap-4">
            <Button type="button" variant="outline" onClick={() => window.history.back()}>
              Cancel
            </Button>
            <Button type="submit" disabled={processing} className="inline-flex items-center gap-2">
              <Save className="h-4 w-4" />
              {processing ? 'Saving…' : 'Save Changes'}
            </Button>
          </div>
        </form>

        <aside className="col-span-12 space-y-4 lg:col-span-4">
          <div className="rounded-lg border bg-card p-6 shadow-sm" style={{ marginTop: '10px' }}>
            <h2 className="mb-4 text-xl font-semibold">Actions</h2>

            <Button asChild className="mb-3 w-full">
              <Link href={route('central-stores.received-batches.index', centralStore.id)}>
                View Received Batches
              </Link>
            </Button>

            <Button
              asChild
              variant="outline"
              className="mb-3 w-full"
            >
              <Link href={route('central-stores.show', centralStore.id)}>
                View Central Store
              </Link>
            </Button>

            <Button
              asChild
              variant="secondary"
              className="mb-3 w-full"
            >
              <Link
                href={route('central-stores.transfer-stock-form', {
                  centralStore: centralStore.id,
                  batch: batch.id,
                })}
              >
                Transfer from this Batch
              </Link>
            </Button>

            <Button
              asChild
              variant="default"
              className="w-full"
            >
              <Link href={route('central-stores.stock-transfers.index', centralStore.id)}>
                Stock Transaction History
              </Link>
            </Button>
          </div>

          <div className="rounded-lg border bg-card p-6 shadow-sm dark:bg-neutral-900">
            <h2 className="mb-2 text-lg font-semibold">Central Store Info</h2>
            <p className="text-sm text-muted-foreground">Name: {centralStore.name}</p>
            <p className="text-sm text-muted-foreground">Location: {centralStore.location}</p>
          </div>
        </aside>
      </div>
    </AppLayout>
  );
}
