import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Plus, Search, Trash } from 'lucide-react';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import { Button } from '@/components/ui/button';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Central Stores', href: '/central-stores' }];

interface CentralStore {
    id: number;
    name: string;
    location: string;
    address: string;
    latitude: string;
    longitude: string;
    description: string;
}

interface PagePropsWithCentralStores extends PageProps {
    centralStores: {
        data: CentralStore[];
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    filters: {
        search?: string;
    };
}

export default function CentralStoreList() {
    const { centralStores, filters } = usePage<PagePropsWithCentralStores>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');

    const handleSearch = () => {
        router.get('/central-stores', { search: searchQuery }, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get('/central-stores', { page: selected.selected + 1, search: searchQuery }, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This central store will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('central-stores.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The central store has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Central Stores" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-6 flex flex-wrap items-center justify-between gap-3">
                    <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100">Central Stores</h1>

                    <div className="flex flex-wrap items-center gap-3">
                        {/* Example quick-links (optional / can be removed) */}
                        <Link
                            href={route('central-stores.received-batches.index', 1)}
                            className="inline-flex items-center gap-2 rounded-lg bg-yellow-400 px-4 py-2 text-sm font-semibold text-white shadow-sm transition-colors hover:bg-yellow-600 focus:outline-none focus:ring-2 focus:ring-green-400 focus:ring-offset-2"
                        >
                            <Eye className="h-4 w-4" />
                            <span>Manage Medication Batches</span>
                        </Link>
                        <Link
                            href={route('central-stores.receive-batch.form', 1)}
                            className="inline-flex items-center gap-2 rounded-lg bg-green-500 px-4 py-2 text-sm font-semibold text-white shadow-sm transition-colors hover:bg-green-600 focus:outline-none focus:ring-2 focus:ring-green-400 focus:ring-offset-2"
                        >
                            <Plus className="h-4 w-4" />
                            <span>Receive Medication</span>
                        </Link>

                        <Link
                            href={route('central-stores.create')}
                            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-4 py-2 text-sm font-semibold text-white shadow-sm transition-colors hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-400 focus:ring-offset-2"
                        >
                            <Plus className="h-4 w-4" />
                            <span>Add Store</span>
                        </Link>
                    </div>
                </div>

                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            placeholder="Search stores..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                    <button
                        onClick={handleSearch}
                        className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                    >
                        Search
                    </button>
                </div>

                {/* Card Layout */}
                <div className="grid grid-cols-1 gap-4">
                    {centralStores.data.map((store) => (
                        <div
                            key={store.id}
                            className="border-border rounded-xl border bg-white p-4 shadow-sm dark:bg-neutral-900"
                        >
                            <h2 className="mb-1 text-lg font-semibold">{store.name}</h2>

                            <p className="text-muted-foreground mb-1 text-sm">
                                <strong>Location:</strong> {store.location}
                            </p>
                            <p className="text-muted-foreground mb-1 text-sm">
                                <strong>Address:</strong> {store.address}
                            </p>
                            <p className="text-muted-foreground mb-1 text-sm">
                                <strong>Coordinates:</strong> {store.latitude}, {store.longitude}
                            </p>
                            <p className="text-muted-foreground mb-3 text-sm">
                                <strong>Description:</strong> {store.description}
                            </p>

                            {/* Map Preview */}
                            <div className="mb-3 h-40 w-full overflow-hidden rounded-md">
                                <iframe
                                    title={`Map of ${store.name}`}
                                    width="100%"
                                    height="100%"
                                    className="rounded-md border"
                                    loading="lazy"
                                    src={`https://www.openstreetmap.org/export/embed.html?bbox=${parseFloat(store.longitude) - 0.005},${parseFloat(store.latitude) - 0.003},${parseFloat(store.longitude) + 0.005},${parseFloat(store.latitude) + 0.003}&layer=mapnik&marker=${store.latitude},${store.longitude}`}
                                ></iframe>
                            </div>

                            <div className="flex flex-wrap gap-2">
                                <Button variant="secondary" asChild>
                                    <Link href={route('central-stores.show', store.id)} title="View">
                                        <Eye className="mr-2 h-4 w-4" />
                                        Manage Stock (Receive/Issue)
                                    </Link>
                                </Button>

                                {/* NEW: View Received Batches button (per store) */}
                                <Button asChild className="bg-indigo-600 hover:bg-indigo-700 text-white">
                                    <Link
                                        href={`/central-stores/${store.id}/received-batches`}
                                        title="View Received Batches"
                                    >
                                        <Eye className="mr-2 h-4 w-4" />
                                        View Received Batches
                                    </Link>
                                </Button>

                                <Button asChild className="bg-yellow-500 hover:bg-yellow-600">
                                    <Link href={route('central-stores.edit', store.id)} title="Edit">
                                        <Pencil className="mr-2 h-4 w-4" />
                                        Edit
                                    </Link>
                                </Button>

                                <Button
                                    onClick={() => handleDelete(store.id)}
                                    className="bg-red-600 hover:bg-red-700 text-white"
                                    title="Delete"
                                    style={{ cursor: 'pointer' }}
                                >
                                    <Trash className="mr-2 h-4 w-4" />
                                    Delete
                                </Button>
                            </div>
                        </div>
                    ))}
                </div>

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={centralStores.last_page}
                        forcePage={centralStores.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}
