import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Calendar } from '@/components/ui/calendar';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { cn } from '@/lib/utils';
import { format } from 'date-fns';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm, Link, router, usePage } from '@inertiajs/react';
import { useEffect, useState } from 'react';
import Select from 'react-select';
import { Plus, Upload, FileText, X, CalendarIcon } from 'lucide-react';
import { useDropzone } from 'react-dropzone';

type OptionType = {
    value: number;
    label: string;
};

type FormData = {
    medication_id: string;
    supplier_id: string;
    batch_number: string;
    quantity: string;
    expiry_date: string;
    manufacture_date: string;
    received_date: string;
    invoice_number: string;
    price: string;
    invoice_file: File | null;
};

export default function ReceiveBatch() {
    const { centralStore, medications, suppliers } = usePage().props as any;

    const { data, setData, post, processing, errors, reset } = useForm<FormData>({
        medication_id: '',
        supplier_id: '',
        batch_number: '',
        quantity: '',
        expiry_date: '',
        manufacture_date: '',
        received_date: '',
        invoice_number: '',
        price: '',
        invoice_file: null,
    });

    const [isDarkMode, setIsDarkMode] = useState(false);
    const [expiryDate, setExpiryDate] = useState<Date>();
    const [manufactureDate, setManufactureDate] = useState<Date>();
    const [receivedDate, setReceivedDate] = useState<Date>();

    useEffect(() => {
        const checkDarkMode = () => {
            setIsDarkMode(document.documentElement.classList.contains('dark'));
        };
        checkDarkMode();

        const observer = new MutationObserver(checkDarkMode);
        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class'],
        });

        return () => observer.disconnect();
    }, []);

    const medicationOptions: OptionType[] = medications.map((m: any) => {
        const dosage = m.dosage ?? '';
        const unit = m.unit ?? '';
        const form = m.form ?? '';
        const labelParts = [m.name, dosage, unit, form].filter(Boolean);
        const label = labelParts.join(' ').trim();
        return { value: m.id, label: label || ' ' };
    });

    const supplierOptions: OptionType[] = suppliers.map((s: any) => ({
        value: s.id,
        label: s.name,
    }));

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post(route('central-stores.receive-batch.store', centralStore.id), {
            forceFormData: true, // ensure file uploads work
            onSuccess: () => {
                // optional: reset();
            },
        });
    };

    const selectStyles = {
        control: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
            borderColor: isDarkMode ? '#374151' : '#d1d5db',
            color: isDarkMode ? '#fff' : '#000',
        }),
        menu: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#000' : '#fff',
        }),
        option: (base: any, { isFocused }: any) => ({
            ...base,
            backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
            color: isFocused ? '#fff' : isDarkMode ? '#fff' : '#000',
        }),
        singleValue: (base: any) => ({
            ...base,
            color: isDarkMode ? '#fff' : '#000',
        }),
    };

    const { getRootProps, getInputProps, isDragActive } = useDropzone({
        accept: {
            'application/pdf': ['.pdf'],
            'image/*': ['.png', '.jpg', '.jpeg'],
        },
        maxFiles: 1,
        onDrop: (acceptedFiles: File[]) => {
            if (acceptedFiles.length > 0) {
                setData('invoice_file', acceptedFiles[0]);
            }
        },
    });

    const removeFile = () => {
        setData('invoice_file', null);
    };

    const handleDateChange = (date: Date | undefined, field: 'expiry_date' | 'manufacture_date' | 'received_date') => {
        const dateString = date ? format(date, 'yyyy-MM-dd') : '';
        setData(field, dateString);
        if (field === 'expiry_date') setExpiryDate(date);
        if (field === 'manufacture_date') setManufactureDate(date);
        if (field === 'received_date') setReceivedDate(date);
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Central Stores', href: route('central-stores.index') },
        { title: 'Receive Batch', href: '#' },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Receive Medication Batch" />
            <div className="grid grid-cols-12 gap-8 px-4 py-6 sm:px-6">
                <form onSubmit={handleSubmit} className="col-span-12 space-y-6 lg:col-span-8">
                    <div className="rounded-lg border bg-card p-6 shadow-sm dark:bg-neutral-900">
                        <div className="mb-4 flex items-center justify-between">
                            <h2 className="text-xl font-semibold">Receive Medication Batch</h2>
                            <Link
                                href={route('medications.create', { from: 'receive-batch' })}
                                className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-3 py-1.5 text-sm font-medium text-white shadow-sm transition-colors hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-400 focus:ring-offset-2"
                            >
                                <Plus className="h-4 w-4" />
                                Add New Medication
                            </Link>
                        </div>

                        {/* Note about uniqueness */}
                        <div className="mb-4 rounded-md border border-amber-300 bg-amber-50 p-3 text-sm text-amber-800 dark:border-amber-600/50 dark:bg-amber-900/20 dark:text-amber-200">
                            <strong>Note:</strong> The <em>Batch Number</em> should be <strong>unique</strong> for the same medication and supplier within this central store.
                            If you submit details that exactly match an existing batch, the system will <strong>top up</strong> the quantity of that batch instead of creating a new one.
                        </div>

                        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                            {/* Medication and Supplier */}
                            <div className="space-y-2">
                                <Label htmlFor="medication_id">Medication</Label>
                                <Select
                                    id="medication_id"
                                    options={medicationOptions}
                                    value={medicationOptions.find((opt) => opt.value === Number(data.medication_id))}
                                    onChange={(option) => setData('medication_id', option?.value.toString() ?? '')}
                                    placeholder="Select medication..."
                                    styles={selectStyles}
                                />
                                {errors.medication_id && <p className="text-sm text-red-500">{errors.medication_id}</p>}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="supplier_id">Supplier</Label>
                                <Select
                                    id="supplier_id"
                                    options={supplierOptions}
                                    value={supplierOptions.find((opt) => opt.value === Number(data.supplier_id))}
                                    onChange={(option) => setData('supplier_id', option?.value.toString() ?? '')}
                                    placeholder="Select supplier..."
                                    styles={selectStyles}
                                />
                                {errors.supplier_id && <p className="text-sm text-red-500">{errors.supplier_id}</p>}
                            </div>

                            {/* Batch Number and Quantity */}
                            <div className="space-y-2">
                                <Label htmlFor="batch_number">Batch Number</Label>
                                <Input
                                    id="batch_number"
                                    value={data.batch_number}
                                    onChange={(e) => setData('batch_number', e.target.value)}
                                    placeholder="e.g., BATCH-001-2025"
                                />
                                <p className="text-xs text-muted-foreground">
                                    Must be unique for this store/medication/supplier. Exact duplicate details will top up the existing batch.
                                </p>
                                {errors.batch_number && <p className="text-sm text-red-500">{errors.batch_number}</p>}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="quantity">Quantity</Label>
                                <Input
                                    id="quantity"
                                    type="number"
                                    value={data.quantity}
                                    onChange={(e) => setData('quantity', e.target.value)}
                                />
                                {errors.quantity && <p className="text-sm text-red-500">{errors.quantity}</p>}
                            </div>

                            {/* Date Pickers */}
                            <div className="space-y-2">
                                <Label>Expiry Date</Label>
                                <Popover>
                                    <PopoverTrigger asChild>
                                        <Button
                                            variant="outline"
                                            className={cn('w-full justify-start text-left font-normal', !expiryDate && 'text-muted-foreground')}
                                        >
                                            <CalendarIcon className="mr-2 h-4 w-4" />
                                            {expiryDate ? format(expiryDate, 'PPP') : <span>Pick a date</span>}
                                        </Button>
                                    </PopoverTrigger>
                                    <PopoverContent className="w-auto p-0">
                                        <Calendar
                                            mode="single"
                                            selected={expiryDate}
                                            onSelect={(date) => handleDateChange(date, 'expiry_date')}
                                            initialFocus
                                        />
                                    </PopoverContent>
                                </Popover>
                                {errors.expiry_date && <p className="text-sm text-red-500">{errors.expiry_date}</p>}
                            </div>

                            <div className="space-y-2">
                                <Label>Manufacture Date</Label>
                                <Popover>
                                    <PopoverTrigger asChild>
                                        <Button
                                            variant="outline"
                                            className={cn('w-full justify-start text-left font-normal', !manufactureDate && 'text-muted-foreground')}
                                        >
                                            <CalendarIcon className="mr-2 h-4 w-4" />
                                            {manufactureDate ? format(manufactureDate, 'PPP') : <span>Pick a date</span>}
                                        </Button>
                                    </PopoverTrigger>
                                    <PopoverContent className="w-auto p-0">
                                        <Calendar
                                            mode="single"
                                            selected={manufactureDate}
                                            onSelect={(date) => handleDateChange(date, 'manufacture_date')}
                                            initialFocus
                                        />
                                    </PopoverContent>
                                </Popover>
                                {errors.manufacture_date && <p className="text-sm text-red-500">{errors.manufacture_date}</p>}
                            </div>

                            <div className="space-y-2">
                                <Label>Received Date</Label>
                                <Popover>
                                    <PopoverTrigger asChild>
                                        <Button
                                            variant="outline"
                                            className={cn('w-full justify-start text-left font-normal', !receivedDate && 'text-muted-foreground')}
                                        >
                                            <CalendarIcon className="mr-2 h-4 w-4" />
                                            {receivedDate ? format(receivedDate, 'PPP') : <span>Pick a date</span>}
                                        </Button>
                                    </PopoverTrigger>
                                    <PopoverContent className="w-auto p-0">
                                        <Calendar
                                            mode="single"
                                            selected={receivedDate}
                                            onSelect={(date) => handleDateChange(date, 'received_date')}
                                            initialFocus
                                        />
                                    </PopoverContent>
                                </Popover>
                                {errors.received_date && <p className="text-sm text-red-500">{errors.received_date}</p>}
                            </div>

                            {/* Invoice Number and Price */}
                            <div className="space-y-2">
                                <Label htmlFor="invoice_number">Invoice Number</Label>
                                <Input
                                    id="invoice_number"
                                    value={data.invoice_number}
                                    onChange={(e) => setData('invoice_number', e.target.value)}
                                />
                                {errors.invoice_number && <p className="text-sm text-red-500">{errors.invoice_number}</p>}
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="price">Price</Label>
                                <Input
                                    id="price"
                                    type="number"
                                    step="0.01"
                                    value={data.price}
                                    onChange={(e) => setData('price', e.target.value)}
                                />
                                {errors.price && <p className="text-sm text-red-500">{errors.price}</p>}
                            </div>

                            {/* File Upload */}
                            <div className="col-span-full space-y-2">
                                <Label htmlFor="invoice_file">Invoice File</Label>
                                <div
                                    {...getRootProps()}
                                    className={cn(
                                        'cursor-pointer rounded-lg border-2 border-dashed p-6 text-center transition-colors',
                                        isDragActive ? 'border-blue-500 bg-blue-50 dark:bg-blue-950' : 'border-gray-300 dark:border-gray-600',
                                        'hover:border-gray-400 dark:hover:border-gray-500'
                                    )}
                                >
                                    <input {...getInputProps()} />
                                    <Upload className="mx-auto mb-2 h-8 w-8 text-gray-400" />
                                    <p className="text-sm text-gray-600 dark:text-gray-400">
                                        {isDragActive ? 'Drop the file here' : 'Drag & drop a file here, or click to select'}
                                    </p>
                                    <p className="mt-1 text-xs text-gray-500 dark:text-gray-500">PDF, PNG, JPG up to 10MB</p>
                                </div>

                                {data.invoice_file && (
                                    <div className="mt-3 flex items-center justify-between rounded-lg border bg-gray-50 p-3 dark:bg-gray-900">
                                        <div className="flex items-center gap-2">
                                            <FileText className="h-5 w-5 text-blue-500" />
                                            <span className="text-sm">{data.invoice_file.name}</span>
                                        </div>
                                        <Button type="button" variant="ghost" size="sm" onClick={removeFile} className="h-8 w-8 p-0">
                                            <X className="h-4 w-4" />
                                        </Button>
                                    </div>
                                )}

                                {errors.invoice_file && <p className="text-sm text-red-500">{errors.invoice_file}</p>}
                            </div>
                        </div>
                    </div>

                    <div className="flex justify-end gap-4">
                        <Button type="button" variant="outline" onClick={() => window.history.back()}>
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Saving…' : 'Receive Batch'}
                        </Button>
                    </div>
                </form>

                <div className="col-span-12 space-y-4 lg:col-span-4">
                    <div className="rounded-lg border bg-card p-6 shadow-sm" style={{ marginTop: '10px' }}>
                        <h2 className="mb-4 text-xl font-semibold">Actions</h2>

                        <Button asChild className="mb-3 w-full">
                            <Link href={route('central-stores.received-batches.index', centralStore.id)}>View Received Batches</Link>
                        </Button>

                        <Button asChild variant="outline" className="mb-3 w-full">
                            <a href="">View Central Store Stock</a>
                        </Button>

                        <Button asChild variant="secondary" className="mb-3 w-full">
                            <Link href={route('central-stores.receive-batch.form', centralStore.id)}>Receive Medication Batches</Link>
                        </Button>

                        <Button asChild variant="default" className="w-full">
                            <Link href={route('central-stores.stock-transfers.index', centralStore.id)}>View Stock Transaction History</Link>
                        </Button>
                    </div>

                    <div className="rounded-lg border bg-card p-6 shadow-sm dark:bg-neutral-900">
                        <h2 className="mb-2 text-lg font-semibold">Central Store Info</h2>
                        <p className="text-sm text-muted-foreground">Name: {centralStore.name}</p>
                        <p className="text-sm text-muted-foreground">Location: {centralStore.location}</p>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
