import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Plus, Search, Trash, Loader2, Download } from 'lucide-react';
import moment from 'moment';
import { useMemo, useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import axios from 'axios';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Received Batches', href: '/received-batches' }];

interface PagePropsWithBatches extends PageProps {
  batches: {
    data: any[];
    total: number;
    per_page: number;
    current_page: number;
    last_page: number;
  };
  medications: Array<{ id: number; name: string }>;
  suppliers: Array<{ id: number; name: string }>;
  filters: {
    search?: string;
    medication_id?: string;
    supplier_id?: string;
    start_date?: string;
    end_date?: string;
    expired?: boolean | null;
    expiring_soon?: boolean | null;
  };
  centralStoreId: number;
}

export default function ReceivedBatchesIndex() {
  const { batches, medications, suppliers, filters, centralStoreId } = usePage<PagePropsWithBatches>().props;

  const [searchQuery, setSearchQuery] = useState(filters.search || '');
  const [medicationId, setMedicationId] = useState(filters.medication_id || '');
  const [supplierId, setSupplierId] = useState(filters.supplier_id || '');
  const [startDate, setStartDate] = useState(filters.start_date || '');
  const [endDate, setEndDate] = useState(filters.end_date || '');
  const [expired, setExpired] = useState<boolean | null>(filters.expired ?? null);
  const [expiringSoon, setExpiringSoon] = useState<boolean | null>(filters.expiring_soon ?? null);

  // Track which row is deleting (for disabling only that button)
  const [deletingId, setDeletingId] = useState<number | null>(null);

  // Dialog state for "View" (show)
  const [openView, setOpenView] = useState(false);
  const [viewBatch, setViewBatch] = useState<any | null>(null);

  const handleFilter = () => {
    router.get(
      route('central-stores.received-batches.index', { centralStore: centralStoreId }),
      {
        search: searchQuery,
        medication_id: medicationId,
        supplier_id: supplierId,
        start_date: startDate,
        end_date: endDate,
        expired: expired,
        expiring_soon: expiringSoon,
      },
      { preserveState: true }
    );
  };

  const handleResetFilters = () => {
    setSearchQuery('');
    setMedicationId('');
    setSupplierId('');
    setStartDate('');
    setEndDate('');
    setExpired(null);
    setExpiringSoon(null);
    router.get(route('central-stores.received-batches.index', { centralStore: centralStoreId }), {}, { preserveState: true });
  };

  const handlePageChange = (selected: { selected: number }) => {
    router.get(
      route('central-stores.received-batches.index', { centralStore: centralStoreId }),
      {
        page: selected.selected + 1,
        search: searchQuery,
        medication_id: medicationId,
        supplier_id: supplierId,
        start_date: startDate,
        end_date: endDate,
        expired: expired,
        expiring_soon: expiringSoon,
      },
      { preserveState: true }
    );
  };

  const handleDelete = (id: number) => {
    Swal.fire({
      title: '⚠️ Are you absolutely sure?',
      html: `
        <div style="text-align:left">
          <p>Deleting this <strong>received batch</strong> will:</p>
          <ul style="list-style-type:disc; padding-left:20px; margin:10px 0;">
            <li>Permanently remove the batch record</li>
            <li>Delete all associated <strong>stock transfers</strong></li>
            <li>Delete all linked <strong>clinic stock receipts</strong></li>
            <li>Break traceability for this batch in reports</li>
          </ul>
          <p class="text-red-600 font-semibold">This action cannot be undone.</p>
        </div>
      `,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonColor: '#e3342f',
      cancelButtonColor: '#6c757d',
      confirmButtonText: 'Yes, delete it!',
      cancelButtonText: 'Cancel',
      width: '650px',
    }).then((result) => {
      if (!result.isConfirmed) return;

      setDeletingId(id);

      // Show loading while the DELETE request is in flight
      Swal.fire({
        title: 'Deleting…',
        html: 'Please wait',
        allowOutsideClick: false,
        showConfirmButton: false,
        didOpen: () => {
          Swal.showLoading();
        },
      });

      router.delete(
        route('central-stores.received-batches.destroy', {
          centralStore: centralStoreId,
          batch: id,
        }),
        {
          preserveScroll: true,
          onSuccess: () => {
            Swal.fire({
              title: 'Deleted!',
              text: 'The batch and all related records have been permanently removed.',
              icon: 'success',
              timer: 2000,
              showConfirmButton: false,
            });
          },
          onError: (errors) => {
            Swal.fire({
              title: 'Error!',
              text:
                (errors as any)?.message ??
                'Something went wrong. The batch may already have linked records that prevent deletion.',
              icon: 'error',
            });
          },
          onFinish: () => {
            setDeletingId(null);
          },
        }
      );
    });
  };

  const openViewDialog = (batch: any) => {
    setViewBatch(batch);
    setOpenView(true);
  };

  // ----- Download server-generated Excel with current filters -----
  const downloadReport = async () => {
    try {
      Swal.fire({
        title: 'Preparing download…',
        text: 'Generating central store batches report',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading(),
        width: '420px',
      });

      const params = new URLSearchParams();
      if (searchQuery) params.append('search', searchQuery);
      if (medicationId) params.append('medication_id', String(medicationId));
      if (supplierId) params.append('supplier_id', String(supplierId));
      if (startDate) params.append('start_date', startDate);
      if (endDate) params.append('end_date', endDate);
      if (expired === true) params.append('expired', '1');
      if (expiringSoon === true) params.append('expiring_soon', '1');

      const url =
        route('central-stores.received-batches.report', { centralStore: centralStoreId }) +
        (params.toString() ? `?${params.toString()}` : '');

      const res = await axios.get(url, { responseType: 'blob' });

      let filename = 'CentralStore_Batches.xlsx';
      const dispo = res.headers['content-disposition'];
      if (dispo) {
        const match = /filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/.exec(dispo);
        if (match?.[1]) filename = decodeURIComponent(match[1].replace(/['"]/g, ''));
      }

      const blob = new Blob([res.data], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
      const blobUrl = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = blobUrl;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(blobUrl);

      Swal.close();
      Swal.fire({ icon: 'success', title: 'Download started', timer: 1200, showConfirmButton: false, width: '400px' });
    } catch (err: any) {
      Swal.close();
      const message = err?.response?.data?.message || err?.message || 'Failed to download the report.';
      Swal.fire({ icon: 'error', title: 'Error', text: message });
    }
  };

  const statusPill = useMemo(() => {
    if (!viewBatch) return null;
    const isExpired = moment(viewBatch.expiry_date).isBefore(moment());
    const isExpiringSoon = moment(viewBatch.expiry_date).isBetween(moment(), moment().add(30, 'days'));
    return (
      <span
        className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${isExpired
          ? 'bg-red-100 text-red-800'
          : isExpiringSoon
            ? 'bg-yellow-100 text-yellow-800'
            : 'bg-green-100 text-green-800'
          }`}
      >
        {isExpired ? 'Expired' : isExpiringSoon ? 'Expiring Soon' : 'Active'}
      </span>
    );
  }, [viewBatch]);

  const invoiceUrl = viewBatch?.invoice_file ? `/storage/${viewBatch.invoice_file}` : null;
  const invoiceName = viewBatch?.invoice_file ? String(viewBatch.invoice_file).split('/').slice(-1)[0] : null;

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title="Received Batches" />
      <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
        <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
          <h1 className="text-2xl font-bold">Received Batches</h1>

          <div className="flex items-center gap-2">
            <Button onClick={downloadReport} className="inline-flex items-center gap-2"
              style={{
                cursor: 'pointer'
              }}
            >
              <Download className="h-4 w-4" />
              Download Report
            </Button>

            <Link
              href={route('central-stores.receive-batch.form', centralStoreId)}
              className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700"
            >
              <Plus className="h-4 w-4" />
              Receive A Batch
            </Link>
          </div>
        </div>

        {/* Filter Section */}
        <div className="mb-6 rounded-lg border p-4">
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
            {/* Search */}
            <div>
              <label className="mb-1 block text-sm font-medium">Search</label>
              <div className="relative">
                <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  placeholder="Search..."
                  className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pl-10 pr-3 text-sm focus:outline-none focus:ring-2 focus:ring-blue-600"
                />
              </div>
            </div>

            {/* Date Range */}
            <div>
              <label className="mb-1 block text-sm font-medium">Received Date Range</label>
              <div className="grid grid-cols-2 gap-2">
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  className="border-border bg-background rounded-lg border p-2 text-sm"
                />
              </div>
            </div>

            {/* Medication Dropdown */}
            <div>
              <label className="mb-1 block text-sm font-medium">Medication</label>
              <select
                value={medicationId}
                onChange={(e) => setMedicationId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Medications</option>
                {medications.map((medication) => (
                  <option key={medication.id} value={medication.id}>
                    {medication.name}
                  </option>
                ))}
              </select>
            </div>

            {/* Supplier Dropdown */}
            <div>
              <label className="mb-1 block text-sm font-medium">Supplier</label>
              <select
                value={supplierId}
                onChange={(e) => setSupplierId(e.target.value)}
                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
              >
                <option value="">All Suppliers</option>
                {suppliers.map((supplier) => (
                  <option key={supplier.id} value={supplier.id}>
                    {supplier.name}
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Checkboxes */}
          <div className="mt-4 flex flex-wrap items-center justify-start gap-4">
            <div className="flex items-center space-x-2">
              <input
                type="checkbox"
                id="expired"
                checked={expired === true}
                onChange={(e) => setExpired(e.target.checked ? true : null)}
                className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
              />
              <label htmlFor="expired" className="text-sm font-medium">
                Expired Batches
              </label>
            </div>

            <div className="flex items-center space-x-2">
              <input
                type="checkbox"
                id="expiringSoon"
                checked={expiringSoon === true}
                onChange={(e) => setExpiringSoon(e.target.checked ? true : null)}
                className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
              />
              <label htmlFor="expiringSoon" className="text-sm font-medium">
                Expiring Soon (within 30 days)
              </label>
            </div>
          </div>

          <div className="mt-4 flex justify-end gap-2">
            <button
              onClick={handleResetFilters}
              className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
            >
              Reset Filters
            </button>
            <button
              onClick={handleFilter}
              className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
            >
              Apply Filters
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto rounded-xl border">
          <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
            <thead className="bg-muted">
              <tr>
                {[
                  'Batch Number',
                  'Medication',
                  'Supplier',
                  'Quantity',
                  'Expiry Date',
                  'Manufacture Date',
                  'Received Date',
                  'Status',
                  'Actions',
                ].map((header) => (
                  <th
                    key={header}
                    className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider"
                  >
                    {header}
                  </th>
                ))}
              </tr>
            </thead>
            <tbody className="divide-y">
              {batches.data.map((batch: any) => {
                const isExpired = moment(batch.expiry_date).isBefore(moment());
                const isExpiringSoon = moment(batch.expiry_date).isBetween(moment(), moment().add(30, 'days'));
                const deletingThis = deletingId === batch.id;

                return (
                  <tr key={batch.id} className="transition hover:bg-muted/50">
                    <td className="px-4 py-3">{batch.batch_number}</td>
                    <td className="px-4 py-3">{batch.medication?.name}</td>
                    <td className="px-4 py-3">{batch.supplier?.name}</td>
                    <td className="px-4 py-3">{batch.quantity}</td>
                    <td className="px-4 py-3">{moment(batch.expiry_date).format('MMMM Do YYYY')}</td>
                    <td className="px-4 py-3">
                      {batch.manufacture_date && moment(batch.manufacture_date, moment.ISO_8601, true).isValid()
                        ? moment(batch.manufacture_date).format('MMMM Do YYYY')
                        : '_'}
                    </td>
                    <td className="px-4 py-3">{moment(batch.received_date).format('MMMM Do YYYY')}</td>
                    <td className="px-4 py-3">
                      <span
                        className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${isExpired
                          ? 'bg-red-100 text-red-800'
                          : isExpiringSoon
                            ? 'bg-yellow-100 text-yellow-800'
                            : 'bg-green-100 text-green-800'
                          }`}
                      >
                        {isExpired ? 'Expired' : isExpiringSoon ? 'Expiring Soon' : 'Active'}
                      </span>
                    </td>
                    <td className="px-4 py-3">
                      <div className="flex items-center gap-2">
                        {/* View (Dialog) */}
                        <button
                          onClick={() => openViewDialog(batch)}
                          className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                          title="View"
                          style={{ cursor: 'pointer' }}
                        >
                          <Eye className="h-4 w-4" />
                        </button>

                        {/* Edit (navigate to Edit page) */}
                        <Link
                          href={route('central-stores.received-batches.edit', {
                            centralStore: centralStoreId,
                            batch: batch.id,
                          })}
                          className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                          title="Edit"
                        >
                          <Pencil className="h-4 w-4" />
                        </Link>

                        {/* Delete */}
                        <button
                          onClick={() => handleDelete(batch.id)}
                          disabled={deletingThis}
                          className={`inline-flex items-center justify-center rounded-full p-2 text-white ${deletingThis ? 'bg-red-400 cursor-not-allowed' : 'bg-red-600 hover:bg-red-700'
                            }`}
                          title="Delete"
                          style={{ cursor: 'pointer' }}
                        >
                          {deletingThis ? <Loader2 className="h-4 w-4 animate-spin" /> : <Trash className="h-4 w-4" />}
                        </button>

                        {/* Transfer */}
                        <Button asChild size="sm" variant="outline">
                          <Link
                            href={route('central-stores.transfer-stock-form', {
                              centralStore: centralStoreId,
                              batch: batch.id,
                            })}
                          >
                            Transfer
                          </Link>
                        </Button>
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="mt-6 flex justify-center">
          <ReactPaginate
            pageCount={batches.last_page}
            forcePage={batches.current_page - 1}
            onPageChange={handlePageChange}
            marginPagesDisplayed={1}
            pageRangeDisplayed={3}
            previousLabel="← Prev"
            nextLabel="Next →"
            breakLabel="..."
            containerClassName="flex items-center gap-2 text-sm"
            pageClassName="rounded border border-border px-3 py-1 hover:bg-muted"
            activeClassName="bg-blue-600 text-white"
            previousClassName="rounded border border-border px-3 py-1 hover:bg-muted"
            nextClassName="rounded border border-border px-3 py-1 hover:bg-muted"
            breakClassName="px-2"
          />
        </div>
      </div>

      {/* View Dialog */}
      <Dialog open={openView} onOpenChange={setOpenView}>
        <DialogContent className="sm:max-w-xl">
          <DialogHeader>
            <DialogTitle className="flex items-center justify-between">
              <span>Batch Details</span>
              {statusPill}
            </DialogTitle>
            <DialogDescription>Quick view of batch information.</DialogDescription>
          </DialogHeader>

          {viewBatch && (
            <div className="grid grid-cols-1 gap-3 text-sm sm:grid-cols-2">
              <div>
                <div className="text-muted-foreground">Batch Number</div>
                <div className="font-medium">{viewBatch.batch_number}</div>
              </div>
              <div>
                <div className="text-muted-foreground">Medication</div>
                <div className="font-medium">{viewBatch.medication?.name ?? '-'}</div>
              </div>
              <div>
                <div className="text-muted-foreground">Supplier</div>
                <div className="font-medium">{viewBatch.supplier?.name ?? '-'}</div>
              </div>
              <div>
                <div className="text-muted-foreground">Quantity</div>
                <div className="font-medium">{viewBatch.quantity}</div>
              </div>
              <div>
                <div className="text-muted-foreground">Expiry Date</div>
                <div className="font-medium">{moment(viewBatch.expiry_date).format('MMM D, YYYY')}</div>
              </div>
              <div>
                <div className="text-muted-foreground">Manufacture Date</div>
                <div className="font-medium">
                  {viewBatch.manufacture_date ? moment(viewBatch.manufacture_date).format('MMM D, YYYY') : '-'}
                </div>
              </div>
              <div>
                <div className="text-muted-foreground">Received Date</div>
                <div className="font-medium">
                  {viewBatch.received_date ? moment(viewBatch.received_date).format('MMM D, YYYY') : '-'}
                </div>
              </div>
              <div>
                <div className="text-muted-foreground">Invoice Number</div>
                <div className="font-medium">{viewBatch.invoice_number ?? '-'}</div>
              </div>
              <div>
                <div className="text-muted-foreground">Price</div>
                <div className="font-medium">{viewBatch.price != null ? viewBatch.price : '-'}</div>
              </div>
              <div className="col-span-full">
                <div className="text-muted-foreground">Invoice File</div>
                {viewBatch.invoice_file ? (
                  <a
                    href={`/storage/${viewBatch.invoice_file}`}
                    target="_blank"
                    rel="noreferrer"
                    className="font-medium underline"
                  >
                    {String(viewBatch.invoice_file).split('/').slice(-1)[0]}
                  </a>
                ) : (
                  <div className="font-medium">-</div>
                )}
              </div>
            </div>
          )}

          <DialogFooter className="mt-4 flex w-full justify-between gap-2">
            <Button variant="outline" onClick={() => setOpenView(false)}>
              Close
            </Button>

            <div className="flex gap-2">
              {viewBatch && (
                <>
                  <Button asChild variant="secondary">
                    <Link
                      href={route('central-stores.transfer-stock-form', {
                        centralStore: centralStoreId,
                        batch: viewBatch.id,
                      })}
                    >
                      Transfer
                    </Link>
                  </Button>
                  <Button asChild>
                    <Link
                      href={route('central-stores.received-batches.edit', {
                        centralStore: centralStoreId,
                        batch: viewBatch.id,
                      })}
                    >
                      Edit Batch
                    </Link>
                  </Button>
                </>
              )}
            </div>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </AppLayout>
  );
}
