import { Button } from '@/components/ui/button';
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, usePage } from '@inertiajs/react';
import { ArrowLeft, Pencil } from 'lucide-react';

interface CentralStore {
    id: number;
    name: string;
    location: string;
    address: string;
    latitude: string;
    longitude: string;
    description: string;
}

interface Batch {
    id: number;
    batch_number: string;
    quantity: number;
    expiry_date: string;
    manufacture_date: string;
    received_date: string;
    medication: {
        name: string;
    };
    supplier: {
        name: string;
    };
}

interface PagePropsWithStore extends PageProps {
    centralStore: CentralStore;
    batches: {
        data: Batch[];
        links: any;
    };
}

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Central Stores', href: `${API}/central-stores` },
    { title: 'Details', href: '#' },
];

export default function CentralStoreShow() {
    const { centralStore, batches } = usePage<PagePropsWithStore>().props;

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title={`Central Store - ${centralStore.name}`} />
            <div className="grid grid-cols-12 gap-8">
                <div className="col-span-8 space-y-6">
                    <div className="bg-background text-foreground mx-8 my-6 space-y-6 rounded-xl p-6 shadow-sm">
                        <div className="mb-4 flex items-center justify-between">
                            <div>
                                <h1 className="text-2xl font-bold">Central Store Details</h1>
                                <p className="text-muted-foreground text-sm">{centralStore.name}</p>
                            </div>
                            <div className="flex gap-2">
                                <Link
                                    href={route('central-stores.index')}
                                    className="hover:bg-muted inline-flex items-center gap-2 rounded-lg border px-4 py-2 text-sm font-medium"
                                >
                                    <ArrowLeft className="h-4 w-4" />
                                    Back
                                </Link>
                                <Link
                                    href={route('central-stores.edit', centralStore.id)}
                                    className="inline-flex items-center gap-2 rounded-lg bg-yellow-500 px-4 py-2 text-sm font-medium text-white hover:bg-yellow-600"
                                >
                                    <Pencil className="h-4 w-4" />
                                    Edit
                                </Link>
                                <Link
                                    href={route('central-stores.receive-batch.form', centralStore.id)}
                                    className="inline-flex items-center gap-2 rounded-lg bg-green-500 px-4 py-2 text-sm font-medium text-white hover:bg-green-600"
                                >
                                    <Pencil className="h-4 w-4" />
                                    Receive Medication Batches
                                </Link>
                            </div>
                        </div>

                        <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                            <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                                <div>
                                    <p className="text-muted-foreground mb-1 text-sm">Name</p>
                                    <p className="text-base font-medium">{centralStore.name}</p>
                                </div>
                                <div>
                                    <p className="text-muted-foreground mb-1 text-sm">Location</p>
                                    <p className="text-base font-medium">{centralStore.location}</p>
                                </div>
                                <div>
                                    <p className="text-muted-foreground mb-1 text-sm">Address</p>
                                    <p className="text-base font-medium">{centralStore.address}</p>
                                </div>
                                <div>
                                    <p className="text-muted-foreground mb-1 text-sm">Coordinates</p>
                                    <p className="text-base font-medium">
                                        {centralStore.latitude}, {centralStore.longitude}
                                    </p>
                                </div>
                                <div className="md:col-span-2">
                                    <p className="text-muted-foreground mb-1 text-sm">Description</p>
                                    <p className="text-base font-medium">{centralStore.description}</p>
                                </div>
                            </div>

                            {/* Map */}
                            <div className="mt-4">
                                <h2 className="mb-2 text-lg font-semibold">Map Preview</h2>
                                <div className="h-64 w-full overflow-hidden rounded-md border">
                                    <iframe
                                        title={`Map of ${centralStore.name}`}
                                        width="100%"
                                        height="100%"
                                        className="rounded-md"
                                        loading="lazy"
                                        src={`https://www.openstreetmap.org/export/embed.html?bbox=${parseFloat(centralStore.longitude) - 0.005},${parseFloat(centralStore.latitude) - 0.003},${parseFloat(centralStore.longitude) + 0.005},${parseFloat(centralStore.latitude) + 0.003}&layer=mapnik&marker=${centralStore.latitude},${centralStore.longitude}`}
                                    ></iframe>
                                </div>
                            </div>
                        </div>

                        {/* Batches Close to Expiry */}
                        <div className="mt-6">
                            <h2 className="text-destructive mb-3 text-lg font-semibold">Batches Close to Expiry</h2>

                            {batches.data.length > 0 ? (
                                <div className="border-border overflow-x-auto rounded-xl border shadow-sm">
                                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                                        <thead className="bg-muted text-muted-foreground text-xs font-medium uppercase">
                                            <tr>
                                                <th className="px-4 py-2 text-left">Batch Number</th>
                                                <th className="px-4 py-2 text-left">Medication</th>
                                                <th className="px-4 py-2 text-left">Supplier</th>
                                                <th className="px-4 py-2 text-left">Quantity</th>
                                                <th className="px-4 py-2 text-left">Expiry Date</th>
                                                <th className="px-4 py-2 text-left">Manufacture Date</th>
                                                <th className="px-4 py-2 text-left">Received Date</th>
                                            </tr>
                                        </thead>
                                        <tbody className="divide-y">
                                            {batches.data.map((batch) => (
                                                <tr key={batch.id} className="hover:bg-muted/50 transition">
                                                    <td className="px-4 py-3">{batch.batch_number}</td>
                                                    <td className="px-4 py-3">{batch.medication.name}</td>
                                                    <td className="px-4 py-3">{batch.supplier.name}</td>
                                                    <td className="px-4 py-3">{batch.quantity}</td>
                                                    <td className="px-4 py-3 font-medium text-red-600">{batch.expiry_date}</td>
                                                    <td className="px-4 py-3">{batch.manufacture_date}</td>
                                                    <td className="px-4 py-3">{batch.received_date}</td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>
                                </div>
                            ) : (
                                <p className="text-muted-foreground text-sm italic">No batches are close to expiring.</p>
                            )}
                        </div>
                    </div>
                </div>
                <div className="col-span-4" style={{
                    marginTop: '10px',
                }}>
                    <div className="space-y-4">
                        <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px' }}>
                            <h2 className="mb-4 text-xl font-semibold">Actions</h2>

                            <Button asChild className="mb-3 w-full">
                                <Link href={route('central-stores.received-batches.index', centralStore.id)}>View Received Batches</Link>
                            </Button>

                            <Button asChild variant="outline" className="mb-3 w-full">
                                <a href="">View Central Store Stock</a>
                            </Button>

                            <Button asChild variant="secondary" className="mb-3 w-full">
                                <Link href={route('central-stores.receive-batch.form', centralStore.id)}>Receive Medication Batches</Link>
                            </Button>

                            <Button asChild variant="default" className="w-full">
                                <Link href={route('central-stores.stock-transfers.index', centralStore.id)}>View Stock Transaction History</Link>
                            </Button>
                        </div>
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}
