import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps } from '@inertiajs/core';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { Eye, Pencil, Plus, Search, Trash } from 'lucide-react';
import moment from 'moment';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import * as XLSX from 'xlsx';

const breadcrumbs: BreadcrumbItem[] = [{ title: 'Transfer History', href: '/central-stores/transfer-history' }];

interface PagePropsWithTransfers extends PageProps {
    stockTransfers: {
        data: any[];
        total: number;
        per_page: number;
        current_page: number;
        last_page: number;
    };
    clinics: Array<{ id: number; name: string }>;
    users: Array<{ id: number; name: string }>;
    centralStore: { id: number; name: string };
    filters: {
        clinic_id?: string;
        batch_number?: string;
        start_date?: string;
        end_date?: string;
        transferred_by?: string;
    };
}

export default function TransferHistory() {
    const { stockTransfers, clinics, users, centralStore, filters } = usePage<PagePropsWithTransfers>().props;
    const [clinicId, setClinicId] = useState(filters.clinic_id || '');
    const [batchNumber, setBatchNumber] = useState(filters.batch_number || '');
    const [startDate, setStartDate] = useState(filters.start_date || '');
    const [endDate, setEndDate] = useState(filters.end_date || '');
    const [transferredBy, setTransferredBy] = useState(filters.transferred_by || '');

    const handleFilter = () => {
        router.get(
            route('central-stores.stock-transfers.index', { centralStore: centralStore.id }),
            {
                clinic_id: clinicId,
                batch_number: batchNumber,
                start_date: startDate,
                end_date: endDate,
                transferred_by: transferredBy,
            },
            { preserveState: true },
        );
    };

    const handleResetFilters = () => {
        setClinicId('');
        setBatchNumber('');
        setStartDate('');
        setEndDate('');
        setTransferredBy('');
        router.get(route('central-stores.stock-transfers.index', { centralStore: centralStore.id }), {}, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(
            route('central-stores.stock-transfers.index', { centralStore: centralStore.id }),
            {
                page: selected.selected + 1,
                clinic_id: clinicId,
                batch_number: batchNumber,
                start_date: startDate,
                end_date: endDate,
                transferred_by: transferredBy,
            },
            { preserveState: true },
        );
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This transfer will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('stock-transfers.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The transfer has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    const exportToExcel = () => {
        if (!stockTransfers?.data?.length) return;

        // Prepare data with safe null handling
        const data = stockTransfers.data.map((transfer) => ({
            'Batch Number': transfer.medication_batch?.batch_number ?? '',
            Medication: transfer.medication_batch?.medication?.name ?? '',
            'From Central Store': transfer.from_central_store?.name ?? '',
            'To Clinic': transfer.to_clinic?.name ?? '',
            'Transferred Quantity': transfer.quantity ?? '',
            'Transferred By': transfer.transferred_by?.name ?? '',
            'Transfer Date': transfer.transfer_date ? moment(transfer.transfer_date).format('MMMM Do YYYY') : '',
        }));

        // Create worksheet from JSON
        const worksheet: XLSX.WorkSheet = XLSX.utils.json_to_sheet(data, { cellStyles: true });
        const workbook: XLSX.WorkBook = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(workbook, worksheet, 'Stock Transfers');

        // Apply styles: borders + bold headers
        const range = XLSX.utils.decode_range(worksheet['!ref']!); // non-null assertion for safe range access

        // Style header row
        for (let C = range.s.c; C <= range.e.c; ++C) {
            const cellAddress = XLSX.utils.encode_cell({ r: 0, c: C });
            const cell = worksheet[cellAddress];
            if (cell) {
                cell.s = {
                    font: { bold: true },
                    border: {
                        top: { style: 'thin', color: { rgb: '000000' } },
                        bottom: { style: 'thin', color: { rgb: '000000' } },
                        left: { style: 'thin', color: { rgb: '000000' } },
                        right: { style: 'thin', color: { rgb: '000000' } },
                    },
                };
            }
        }

        // Style data cells
        for (let R = 1; R <= range.e.r; ++R) {
            for (let C = range.s.c; C <= range.e.c; ++C) {
                const cellAddress = XLSX.utils.encode_cell({ r: R, c: C });
                const cell = worksheet[cellAddress];
                if (cell) {
                    cell.s = {
                        border: {
                            top: { style: 'thin', color: { rgb: '000000' } },
                            bottom: { style: 'thin', color: { rgb: '000000' } },
                            left: { style: 'thin', color: { rgb: '000000' } },
                            right: { style: 'thin', color: { rgb: '000000' } },
                        },
                    };
                }
            }
        }

        // Download the file
        XLSX.writeFile(workbook, 'stock-transfers.xlsx', { cellStyles: true });
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Transfer History" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Transfer History</h1>
                    <div className="flex gap-2">
                        <Link
                            href={route('central-stores.receive-batch.form', centralStore.id)}
                            className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700"
                        >
                            <Plus className="h-4 w-4" />
                            Receive Medication
                        </Link>
                        <button
                            onClick={exportToExcel}
                            className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-5 py-2 text-sm font-medium text-white hover:bg-green-700"
                        >
                            Export to Excel
                        </button>
                    </div>
                </div>

                <div className="mb-6 rounded-lg border p-4">
                    <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-4">
                        <div>
                            <label className="mb-1 block text-sm font-medium">Batch Number</label>
                            <div className="relative">
                                <Search className="text-muted-foreground absolute top-2.5 left-3 h-5 w-5" />
                                <input
                                    type="text"
                                    value={batchNumber}
                                    onChange={(e) => setBatchNumber(e.target.value)}
                                    placeholder="Batch Number"
                                    className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                                />
                            </div>
                        </div>

                        <div>
                            <label className="mb-1 block text-sm font-medium">Transfer Date Range</label>
                            <div className="grid grid-cols-2 gap-2">
                                <input
                                    type="date"
                                    value={startDate}
                                    onChange={(e) => setStartDate(e.target.value)}
                                    className="border-border bg-background rounded-lg border p-2 text-sm"
                                />
                                <input
                                    type="date"
                                    value={endDate}
                                    onChange={(e) => setEndDate(e.target.value)}
                                    className="border-border bg-background rounded-lg border p-2 text-sm"
                                />
                            </div>
                        </div>

                        <div>
                            <label className="mb-1 block text-sm font-medium">Clinic</label>
                            <select
                                value={clinicId}
                                onChange={(e) => setClinicId(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Clinics</option>
                                {clinics.map((clinic) => (
                                    <option key={clinic.id} value={clinic.id}>
                                        {clinic.name}
                                    </option>
                                ))}
                            </select>
                        </div>

                        <div>
                            <label className="mb-1 block text-sm font-medium">Transferred By</label>
                            <select
                                value={transferredBy}
                                onChange={(e) => setTransferredBy(e.target.value)}
                                className="border-border bg-background w-full rounded-lg border p-2 text-sm"
                            >
                                <option value="">All Users</option>
                                {users.map((user) => (
                                    <option key={user.id} value={user.id}>
                                        {user.name}
                                    </option>
                                ))}
                            </select>
                        </div>
                    </div>

                    <div className="mt-4 flex justify-end gap-2">
                        <button
                            onClick={handleResetFilters}
                            className="rounded-lg bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300"
                        >
                            Reset Filters
                        </button>
                        <button onClick={handleFilter} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                            Apply Filters
                        </button>
                    </div>
                </div>

                <div className="overflow-x-auto rounded-xl border">
                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {[
                                    'Batch Number',
                                    'Medication',
                                    'From Central Store',
                                    'To Clinic',
                                    'Transfered Quantity',
                                    'Transferred By',
                                    'Transfer Date',
                                    'Actions',
                                ].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {stockTransfers.data.map((transfer: any) => (
                                <tr key={transfer.id} className="hover:bg-muted/50 transition">
                                    <td className="px-4 py-3">{transfer.medication_batch?.batch_number}</td>
                                    <td className="px-4 py-3">{transfer.medication_batch?.medication?.name}</td>
                                    <td className="px-4 py-3">{transfer.from_central_store?.name}</td>
                                    <td className="px-4 py-3">
                                        <span className="inline-block rounded-full bg-blue-100 px-3 py-1 text-xs font-medium text-blue-800">
                                            {transfer.to_clinic?.name || '—'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3">{transfer.quantity || '—'}</td>
                                    <td className="px-4 py-3">
                                        <span className="inline-block rounded-full bg-green-100 px-3 py-1 text-xs font-medium text-green-800">
                                            {transfer.transferred_by?.name || '—'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3">{moment(transfer.transfer_date).format('MMMM Do YYYY')}</td>
                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <Link
                                            href={''}
                                            className="inline-flex items-center justify-center rounded-full bg-blue-600 p-2 text-white hover:bg-blue-700"
                                            title="View"
                                        >
                                            <Eye className="h-4 w-4" />
                                        </Link>
                                        <Link
                                            href={''}
                                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                            title="Edit"
                                        >
                                            <Pencil className="h-4 w-4" />
                                        </Link>
                                        <button
                                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                            title="Delete"
                                            onClick={() => handleDelete(transfer.id)}
                                        >
                                            <Trash className="h-4 w-4" />
                                        </button>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={stockTransfers.last_page}
                        forcePage={stockTransfers.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>
        </AppLayout>
    );
}
