// resources/js/Pages/CentralStores/TransferStock.tsx
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { API } from '@/config';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import { Head, useForm, Link } from '@inertiajs/react';
import { useEffect, useState } from 'react';
import Select from 'react-select';
import Swal from 'sweetalert2';

// Define your custom page props interface
interface TransferStockPageProps extends InertiaPageProps {
    centralStore: {
        id: number;
        name: string;
        // other central store properties
    };
    batch: {
        id: number;
        batch_number: string;
        quantity: number;
        expiry_date: string;
        medication: {
            id: number;
            name: string;
        };
        // other batch properties
    };
    clinics: Array<{
        id: number;
        name: string;
    }>;
    currentStock: number;
}

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Central Stores', href: `${API}/central-stores` },
    { title: 'Transfer Stock', href: '' },
];

export default function TransferStock({ centralStore, batch, clinics, currentStock }: TransferStockPageProps) {
    const { data, setData, post, errors, processing } = useForm({
        clinic_id: '',
        quantity: '',
        transfer_date: new Date().toISOString().split('T')[0],
    });

    const [selectedClinic, setSelectedClinic] = useState<any>(null);
    const [isDarkMode, setIsDarkMode] = useState(false);

    useEffect(() => {
        const checkDarkMode = () => {
            setIsDarkMode(document.documentElement.classList.contains('dark'));
        };
        checkDarkMode();

        const observer = new MutationObserver(checkDarkMode);
        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class'],
        });

        return () => observer.disconnect();
    }, []);

    const clinicOptions = clinics.map((clinic) => ({
        value: clinic.id,
        label: clinic.name,
    }));

    const handleClinicChange = (selectedOption: any) => {
        setSelectedClinic(selectedOption);
        setData('clinic_id', selectedOption ? selectedOption.value : '');
    };

    const selectStyles = {
        control: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#1e1e1e' : '#ffffff',
            borderColor: isDarkMode ? '#374151' : '#d1d5db',
            color: isDarkMode ? '#ffffff' : '#000000',
            '&:hover': {
                borderColor: isDarkMode ? '#4b5563' : '#9ca3af',
            },
        }),
        menu: (base: any) => ({
            ...base,
            backgroundColor: isDarkMode ? '#1e1e1e' : '#ffffff',
            color: isDarkMode ? '#ffffff' : '#000000',
        }),
        option: (base: any, { isFocused }: any) => ({
            ...base,
            backgroundColor: isFocused ? (isDarkMode ? '#1e40af' : '#3b82f6') : 'transparent',
            color: isFocused ? '#ffffff' : isDarkMode ? '#ffffff' : '#000000',
            '&:active': {
                backgroundColor: isDarkMode ? '#1e40af' : '#3b82f6',
            },
        }),
        singleValue: (base: any) => ({
            ...base,
            color: isDarkMode ? '#ffffff' : '#000000',
        }),
        input: (base: any) => ({
            ...base,
            color: isDarkMode ? '#ffffff' : '#000000',
        }),
        placeholder: (base: any) => ({
            ...base,
            color: isDarkMode ? '#9ca3af' : '#6b7280',
        }),
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();

        Swal.fire({
            title: 'Transferring Stock...',
            text: 'Please wait while the transfer is being processed.',
            allowOutsideClick: false,
            allowEscapeKey: false,
            didOpen: () => {
                Swal.showLoading();
            },
        });

        post(route('central-stores.transfer-stock', [centralStore.id, batch.id]), {
            onSuccess: () => {
                Swal.fire({
                    icon: 'success',
                    title: 'Stock Transferred',
                    text: 'The stock was transferred successfully!',
                    timer: 1500,
                    showConfirmButton: false,
                }).then(() => {
                    // window.location.href = route('central-stores.received-batches.index', centralStore.id);
                });
            },
            onError: () => {
                Swal.fire({
                    icon: 'error',
                    title: 'Transfer Failed',
                    text: 'An error occurred while transferring stock.',
                });
            },
        });
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Transfer Stock" />
            <div className="w-full px-4 py-6 sm:px-6">
                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid gap-6 sm:grid-cols-12">
                        {/* Main form content */}
                        <div className="col-span-12 space-y-6 sm:col-span-8">
                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Transfer Medication Stock</h2>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                                    <div>
                                        <Label>Medication</Label>
                                        <Input value={batch.medication.name} readOnly />
                                    </div>
                                    <div>
                                        <Label>Batch Number</Label>
                                        <Input value={batch.batch_number} readOnly />
                                    </div>
                                    <div>
                                        <Label>Current Stock</Label>
                                        <Input value={currentStock} readOnly />
                                    </div>
                                    <div>
                                        <Label>Expiry Date</Label>
                                        <Input value={batch.expiry_date} readOnly />
                                    </div>
                                </div>
                            </div>

                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h2 className="mb-4 text-xl font-semibold">Transfer Details</h2>
                                <div className="grid grid-cols-1 gap-4 sm:grid-cols-2">
                                    <div>
                                        <Label htmlFor="clinic_id">Clinic</Label>
                                        <Select
                                            options={clinicOptions}
                                            value={selectedClinic}
                                            onChange={handleClinicChange}
                                            placeholder="Select a clinic..."
                                            styles={selectStyles}
                                            className="react-select-container"
                                            classNamePrefix="react-select"
                                            isSearchable
                                            required
                                        />
                                        {errors.clinic_id && <p className="mt-1 text-sm text-red-500">{errors.clinic_id}</p>}
                                    </div>
                                    <div>
                                        <Label htmlFor="quantity">Quantity to Transfer</Label>
                                        <Input
                                            type="number"
                                            id="quantity"
                                            value={data.quantity}
                                            onChange={(e) => setData('quantity', e.target.value)}
                                            min="1"
                                            max={currentStock}
                                            required
                                        />
                                        {errors.quantity && <p className="mt-1 text-sm text-red-500">{errors.quantity}</p>}
                                    </div>
                                    <div>
                                        <Label htmlFor="transfer_date">Transfer Date</Label>
                                        <Input
                                            type="date"
                                            id="transfer_date"
                                            value={data.transfer_date}
                                            onChange={(e) => setData('transfer_date', e.target.value)}
                                            required
                                        />
                                        {errors.transfer_date && <p className="mt-1 text-sm text-red-500">{errors.transfer_date}</p>}
                                    </div>
                                </div>
                            </div>

                            <div className="flex flex-col justify-end gap-4 sm:flex-row">
                                <Button type="button" variant="outline" onClick={() => window.history.back()}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={processing}>
                                    {processing ? 'Transferring...' : 'Transfer Stock'}
                                </Button>
                            </div>
                        </div>

                        {/* Optional sidebar area */}
                        <div className="col-span-12 space-y-6 sm:col-span-4">
                            <div className="bg-card rounded-lg border p-6 shadow-sm" style={{ marginTop: '10px' }}>
                                <h2 className="mb-4 text-xl font-semibold">Actions</h2>

                                <Button asChild className="mb-3 w-full">
                                    <Link href={route('central-stores.received-batches.index', centralStore.id)}>View Received Batches</Link>
                                </Button>

                                <Button asChild variant="outline" className="mb-3 w-full">
                                    <a href="">View Central Store Stock</a>
                                </Button>

                                <Button asChild variant="secondary" className="mb-3 w-full">
                                    <Link href={route('central-stores.receive-batch.form', centralStore.id)}>Receive Medication Batches</Link>
                                </Button>

                                <Button asChild variant="default" className="w-full">
                                    <Link href={route('central-stores.stock-transfers.index', centralStore.id)}>View Stock Transaction History</Link>
                                </Button>
                            </div>


                            <div className="bg-card rounded-lg border p-6 shadow-sm dark:bg-neutral-900">
                                <h3 className="mb-2 text-lg font-medium">Transfer Guidelines</h3>
                                <ul className="text-muted-foreground space-y-2 text-sm">
                                    <li>• Ensure the receiving clinic has storage capacity</li>
                                    <li>• Verify medication expiry dates before transfer</li>
                                    <li>• Document any special handling requirements</li>
                                </ul>
                            </div>


                        </div>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
