// resources/js/Pages/Chronics/Dispensations/PatientHistory.tsx
import { Head, Link, router, usePage } from '@inertiajs/react';
import React, { useMemo, useRef, useState } from 'react';
import moment from 'moment';
import { QRCodeSVG } from 'qrcode.react';
import { useReactToPrint } from 'react-to-print';

import AppLayout from '@/layouts/app-layout';
import { type BreadcrumbItem } from '@/types';

import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Card, CardHeader, CardContent, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';

import {
    ImageIcon,
    Pill,
    CalendarClock,
    User,
    MapPin,
    ChevronLeft,
    ChevronRight,
    Search,
    Download,
    Printer,
    CornerDownLeft,
} from 'lucide-react';

type TimelineItem = {
    id: number;
    dispensed_at?: string | null;
    origin?: 'clinic' | 'nurse' | null;
    clinic_name?: string | null;
    quantity: number;
    frequency?: string | null;
    route?: string | null;
    duration?: string | number | null;
    notes?: string | null;
    medication?: {
        name?: string | null;
        dosage?: string | null;
        form?: string | null;
        unit?: string | null;
    } | null;
    batch_number?: string | null;
    dispensed_by?: { id: number; name: string; email?: string | null } | null;
    show_url: string;
};

type PageProps = {
    patient: {
        id: number;
        full_name: string;
        employee_number?: string | null;
        gender?: string | null;
        date_of_birth?: string | null;
        image_url?: string | null;
    };
    timeline: {
        data: TimelineItem[];
        meta: {
            current_page: number;
            last_page: number;
            per_page: number;
            total: number;
        };
    };
    filters: {
        search?: string | null;
        start_date?: string | null;
        end_date?: string | null;
        per_page?: number | null;
    };
};

const logoUrl = 'https://providencehumancapital.com/headoffice/assets/images/PHC_Logo.png';

export default function PatientHistory() {
    const { patient, timeline, filters } = usePage<PageProps>().props;

    // ── Breadcrumbs
    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Chronic Database', href: route('chronics.patients.index') },
        { title: patient.full_name, href: route('patients.show', patient.id) },
        { title: 'Medication History', href: '#' },
    ];

    // ── Filters UI State
    const [search, setSearch] = useState(filters.search ?? '');
    const [startDate, setStartDate] = useState(filters.start_date ?? '');
    const [endDate, setEndDate] = useState(filters.end_date ?? '');
    const [perPage, setPerPage] = useState<number>(filters.per_page ?? 20);

    const baseHref = route('chronics.patients.dispensations.history', patient.id);

    const applyFilters = (page = 1) => {
        router.get(
            baseHref,
            { page, search, start_date: startDate, end_date: endDate, per_page: perPage },
            { preserveState: true, replace: true },
        );
    };

    const clearFilters = () => {
        setSearch('');
        setStartDate('');
        setEndDate('');
        setPerPage(20);
        router.get(baseHref, {}, { preserveState: true, replace: true });
    };

    // ── Derived patient fields
    const ageText = useMemo(() => {
        if (!patient.date_of_birth) return '—';
        const years = moment().diff(moment(patient.date_of_birth), 'years');
        return `${moment(patient.date_of_birth).format('YYYY-MM-DD')} (${years} yrs)`;
    }, [patient.date_of_birth]);

    // ── Print (simple cover/summary for archive)
    const contentRef = useRef<HTMLDivElement>(null);
    const reactToPrintFn = useReactToPrint({ contentRef });

    // ── Helpers
    const originChip = (o?: 'clinic' | 'nurse' | null) => {
        if (!o) return null;
        const label = o === 'nurse' ? 'Nurse Stock' : 'Clinic Stock';
        const cls =
            o === 'nurse'
                ? 'bg-amber-100 text-amber-800 border-amber-200'
                : 'bg-emerald-100 text-emerald-800 border-emerald-200';
        return <span className={`rounded border px-2 py-0.5 text-xs ${cls}`}>{label}</span>;
    };

    const medLine = (it: TimelineItem) => {
        if (!it.medication) return '—';
        const parts = [
            it.medication.name,
            it.medication.dosage,
            it.medication.unit,
            it.medication.form,
        ].filter(Boolean);
        return parts.join(' ');
    };

    const formatDuration = (d?: string | number | null) => {
        if (!d) return '—';
        const n = Number(d);
        if (!isNaN(n)) return `${n} day${n === 1 ? '' : 's'}`;
        return String(d);
    };

    const timeOf = (it: TimelineItem) =>
        it.dispensed_at ? moment(it.dispensed_at).format('HH:mm') : '—';

    // ── Group by date (YYYY-MM-DD), newest first
    const grouped = useMemo(() => {
        const byDate: Record<string, TimelineItem[]> = {};
        for (const it of timeline.data) {
            const key = it.dispensed_at ? moment(it.dispensed_at).format('YYYY-MM-DD') : 'Unknown';
            if (!byDate[key]) byDate[key] = [];
            byDate[key].push(it);
        }
        // sort items in each day by time desc
        Object.values(byDate).forEach((arr) =>
            arr.sort((a, b) => {
                const ta = a.dispensed_at ? +new Date(a.dispensed_at) : 0;
                const tb = b.dispensed_at ? +new Date(b.dispensed_at) : 0;
                return tb - ta;
            }),
        );
        // sort days desc
        const entries = Object.entries(byDate).sort(
            (a, b) => +new Date(b[0]) - +new Date(a[0]),
        );
        return entries;
    }, [timeline.data]);

    // ── Export (JSON) — simple client-side dump of current page
    const handleExport = () => {
        const blob = new Blob([JSON.stringify(timeline.data, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = Object.assign(document.createElement('a'), { href: url, download: `chronic-history-${patient.id}.json` });
        a.click();
        URL.revokeObjectURL(url);
    };

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Chronic Medication History" />

            <div className="bg-background mx-8 my-6 rounded-xl p-6 ">
                {/* Hidden printable summary */}
                <div style={{ display: 'none' }}>
                    <div ref={contentRef} className="p-6 font-sans text-sm text-gray-800">
                        <style>
                            {`
                @page { size: auto; margin: 15mm 10mm; }
                @media print {
                  body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                }
              `}
                        </style>

                        <div className="flex items-center justify-between border-b-2 border-gray-200 pb-4">
                            <img src={logoUrl} alt="Logo" style={{ height: '3.5rem' }} />
                            <div className="text-center">
                                <div className="text-lg font-bold">Chronic Medication History</div>
                                <div>{patient.full_name}</div>
                            </div>
                            <QRCodeSVG value={`Patient:${patient.id} • ChronicHistory`} size={60} />
                        </div>

                        <div className="mt-4">
                            <div><b>Patient:</b> {patient.full_name}</div>
                            <div><b>Employee #:</b> {patient.employee_number ?? '—'}</div>
                            <div><b>DOB (Age):</b> {ageText}</div>
                            <div><b>Generated:</b> {moment().format('MMM D, YYYY • HH:mm')}</div>
                        </div>
                    </div>
                </div>

                {/* Header: Avatar + Name + Actions */}
                <div className="mb-6 flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                    <div className="flex items-center gap-4">
                        {patient.image_url ? (
                            <img src={patient.image_url} alt="Patient" className="h-24 w-24 rounded-full border object-cover" />
                        ) : (
                            <div className="bg-muted text-muted-foreground flex h-24 w-24 items-center justify-center rounded-full border">
                                <ImageIcon className="h-8 w-8" />
                            </div>
                        )}
                        <div>
                            <h1 className="text-foreground text-2xl font-bold">{patient.full_name}</h1>
                            <div className="mt-1 flex flex-wrap items-center gap-2 text-sm text-muted-foreground">
                                <span>Employee #: {patient.employee_number ?? '—'}</span>
                                <Separator orientation="vertical" className="h-4" />
                                <span>Gender: {patient.gender ?? '—'}</span>
                                <Separator orientation="vertical" className="h-4" />
                                <span>DOB: {ageText}</span>
                            </div>
                        </div>
                    </div>

                    <div className="flex flex-wrap gap-2">
                        <Button variant="outline" onClick={() => window.history.back()}>
                            <CornerDownLeft className="mr-2 h-4 w-4" />
                            Back
                        </Button>
                        <Button asChild>
                            <Link href={route('chronics.chronics.patients.dispense', patient.id)}>
                                Dispense Medication
                            </Link>
                        </Button>
                        <Button variant="secondary" onClick={reactToPrintFn}>
                            <Printer className="mr-2 h-4 w-4" />
                            Print Summary
                        </Button>
                        <Button variant="destructive" onClick={handleExport}>
                            <Download className="mr-2 h-4 w-4" />
                            Export JSON
                        </Button>
                        <Button asChild variant="outline">
                            <Link href={route('patients.show', patient.id)}>View Patient</Link>
                        </Button>
                    </div>
                </div>

                <div className="grid grid-cols-12 gap-8">
                    {/* Left: Patient snapshot & Filters */}
                    <div className="col-span-12 lg:col-span-4 space-y-6">
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <User className="h-5 w-5 text-blue-600" />
                                    Patient Snapshot
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="grid grid-cols-2 gap-3">
                                <KV label="Full Name" value={patient.full_name} />
                                <KV label="Employee #" value={patient.employee_number ?? '—'} />
                                <KV label="Gender" value={patient.gender ?? '—'} />
                                <KV label="DOB (Age)" value={ageText} />
                            </CardContent>
                        </Card>

                        <Card>
                            <CardHeader>
                                <CardTitle>Filters</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-3">
                                <div>
                                    <Label htmlFor="search">Search medication</Label>
                                    <div className="relative mt-1">
                                        <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                        <Input
                                            id="search"
                                            value={search ?? ''}
                                            onChange={(e) => setSearch(e.target.value)}
                                            placeholder="e.g. Metformin"
                                            className="pl-9"
                                        />
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-3">
                                    <div>
                                        <Label htmlFor="start">Start date</Label>
                                        <Input id="start" type="date" className="mt-1" value={startDate ?? ''} onChange={(e) => setStartDate(e.target.value)} />
                                    </div>
                                    <div>
                                        <Label htmlFor="end">End date</Label>
                                        <Input id="end" type="date" className="mt-1" value={endDate ?? ''} onChange={(e) => setEndDate(e.target.value)} />
                                    </div>
                                </div>

                                <div>
                                    <Label htmlFor="per">Per page</Label>
                                    <Input
                                        id="per"
                                        type="number"
                                        min={5}
                                        className="mt-1"
                                        value={perPage}
                                        onChange={(e) => setPerPage(Math.max(5, Number(e.target.value || 20)))}
                                    />
                                </div>

                                <div className="flex gap-2 pt-1">
                                    <Button onClick={() => applyFilters(1)}>Apply</Button>
                                    <Button variant="outline" onClick={clearFilters}>Reset</Button>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Right: Timeline of per-day tables */}
                    <div className="col-span-12 lg:col-span-8">
                        <Card className="mb-4">
                            <CardHeader>
                                <CardTitle>Chronic Medication History</CardTitle>
                            </CardHeader>
                            <CardContent className="pt-0">
                                {timeline.data.length === 0 ? (
                                    <div className="py-10 text-center text-sm text-muted-foreground">
                                        No chronic dispensations found.
                                    </div>
                                ) : (
                                    <div className="relative">
                                        {/* Vertical timeline line */}
                                        <div className="absolute left-4 top-0 bottom-0 w-px bg-border" aria-hidden />
                                        <div className="space-y-6 pt-4">
                                            {grouped.map(([dateKey, items]) => {
                                                const friendly = dateKey === 'Unknown'
                                                    ? 'Unknown Date'
                                                    : moment(dateKey).format('dddd, MMM D, YYYY');

                                                return (
                                                    <div key={dateKey} className="relative pl-10">
                                                        {/* Dot */}
                                                        <div
                                                            className="absolute left-3 top-3 h-2.5 w-2.5 -translate-x-1/2 rounded-full bg-primary"
                                                            aria-hidden
                                                        />
                                                        <Card>
                                                            <CardHeader className="pb-2">
                                                                <CardTitle className="flex items-center justify-between gap-2">
                                                                    <div className="flex items-center gap-2">
                                                                        <CalendarClock className="h-5 w-5 text-blue-600" />
                                                                        <span className="font-semibold">{friendly}</span>
                                                                    </div>
                                                                    <Badge variant="secondary">{items.length} item{items.length === 1 ? '' : 's'}</Badge>
                                                                </CardTitle>
                                                            </CardHeader>
                                                            <CardContent className="pt-0">
                                                                <div className="overflow-x-auto">
                                                                    <table className="min-w-full divide-y text-sm">
                                                                        <thead className="bg-muted">
                                                                            <tr>
                                                                                <th className="px-3 py-2 text-left">Time</th>
                                                                                <th className="px-3 py-2 text-left">Medication</th>
                                                                                <th className="px-3 py-2 text-left">Qty</th>
                                                                                <th className="px-3 py-2 text-left">Freq</th>
                                                                                <th className="px-3 py-2 text-left">Route</th>
                                                                                <th className="px-3 py-2 text-left">Duration</th>
                                                                                <th className="px-3 py-2 text-left">Batch</th>
                                                                                <th className="px-3 py-2 text-left">Origin</th>
                                                                                <th className="px-3 py-2 text-left">By</th>
                                                                                <th className="px-3 py-2 text-left"></th>
                                                                            </tr>
                                                                        </thead>
                                                                        <tbody className="divide-y">
                                                                            {items.map((it) => (
                                                                                <tr key={it.id} className="hover:bg-muted/50">

                                                                                    <td className="px-3 py-2 whitespace-nowrap">{timeOf(it)}</td>
                                                                                    <td className="px-3 py-2">
                                                                                        <div className="flex items-center gap-2">
                                                                                            <span className="inline-flex items-center gap-1 rounded-full bg-blue-100 px-3 py-1 text-xs font-semibold text-blue-800 dark:bg-blue-800/20 dark:text-blue-300">
                                                                                                <Pill className="h-3.5 w-3.5" /> {medLine(it)}
                                                                                            </span>
                                                                                        </div>
                                                                                    </td>
                                                                                    <td className="px-3 py-2">{it.quantity}</td>
                                                                                    <td className="px-3 py-2">{it.frequency ?? '—'}</td>
                                                                                    <td className="px-3 py-2">{it.route ?? '—'}</td>
                                                                                    <td className="px-3 py-2">{formatDuration(it.duration)}</td>
                                                                                    <td className="px-3 py-2">{it.batch_number ?? '—'}</td>
                                                                                    <td className="px-3 py-2">
                                                                                        <div className="flex items-center gap-2">
                                                                                            {originChip(it.origin)}
                                                                                            <span className="text-xs text-muted-foreground">
                                                                                                {it.clinic_name ?? ''}
                                                                                            </span>
                                                                                        </div>
                                                                                    </td>
                                                                                    <td className="px-3 py-2">{it.dispensed_by?.name ?? '—'}</td>
                                                                                    <td className="px-3 py-2">
                                                                                        <Button asChild size="sm" variant="outline">
                                                                                            <Link href={it.show_url}>View</Link>
                                                                                        </Button>
                                                                                    </td>
                                                                                </tr>
                                                                            ))}
                                                                        </tbody>
                                                                    </table>
                                                                </div>
                                                            </CardContent>
                                                        </Card>
                                                    </div>
                                                );
                                            })}
                                        </div>
                                    </div>
                                )}
                            </CardContent>
                        </Card>

                        {/* Pagination */}
                        {timeline.meta.last_page > 1 && (
                            <div className="mt-2 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Page {timeline.meta.current_page} of {timeline.meta.last_page} • {timeline.meta.total} total
                                </div>
                                <div className="flex gap-2">
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        disabled={timeline.meta.current_page <= 1}
                                        onClick={() => applyFilters(timeline.meta.current_page - 1)}
                                    >
                                        <ChevronLeft className="mr-1 h-4 w-4" />
                                        Prev
                                    </Button>
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        disabled={timeline.meta.current_page >= timeline.meta.last_page}
                                        onClick={() => applyFilters(timeline.meta.current_page + 1)}
                                    >
                                        Next
                                        <ChevronRight className="ml-1 h-4 w-4" />
                                    </Button>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </AppLayout>
    );
}

function KV({
    label,
    value,
    icon,
}: {
    label: string;
    value?: string | number | null;
    icon?: React.ReactNode;
}) {
    if (value == null || value === '') return null;
    return (
        <div className="rounded-md border p-3">
            <div className="flex items-center gap-2">
                {icon}
                <div>
                    <div className="text-xs font-semibold text-muted-foreground">{label}</div>
                    <div className="text-sm font-medium">{String(value)}</div>
                </div>
            </div>
        </div>
    );
}
