// resources/js/Pages/Chronics/Dispensations/Show.tsx
import AppLayout from '@/layouts/app-layout';
import { Head, usePage, router } from '@inertiajs/react';
import moment from 'moment';
import { Clock, ClipboardList, Pill, Syringe, ArrowLeft, User2, Factory, Hash } from 'lucide-react';

import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';
import { Alert, AlertDescription } from '@/components/ui/alert';

type Disp = {
  id: number;
  origin: 'clinic' | 'nurse';
  clinic_name?: string | null;

  quantity: number;
  frequency?: string | null;
  route?: string | null;
  duration?: string | null; // keep string to match backend validation
  notes?: string | null;

  dispensed_at?: string | null;
  dispensed_by?: { id: number; name: string; email: string } | null;

  batch_number?: string | null;
  expiry_date?: string | null;

  medication?: { name: string; dosage?: string | null; form?: string | null; unit?: string | null } | null;

  patient: {
    id: number;
    full_name: string;
    employee_no?: string | null;
    gender?: string | null;
    dob?: string | null;
  };
};

type PageProps = {
  dispensation: Disp;
  links: { patient_history: string; patient_show: string };
};

export default function Show() {
  const { dispensation, links } = usePage<PageProps>().props;

  const medName = dispensation.medication?.name ?? '—';
  const medDosage = dispensation.medication?.dosage ?? '';
  const medForm = dispensation.medication?.form ?? '';
  const medUnit = dispensation.medication?.unit ?? '';
  const medLine = [
    medDosage ? `${medDosage}${medUnit ? ` ${medUnit}` : ''}` : '',
    medForm || '',
  ]
    .filter(Boolean)
    .join(' • ');

  const dispensedAt = dispensation.dispensed_at
    ? moment(dispensation.dispensed_at).format('YYYY-MM-DD HH:mm')
    : '—';

  const breadcrumbs = [
    { title: 'Chronics', href: route('chronics.dashboard') },
    { title: dispensation.patient.full_name, href: links.patient_show },
    { title: `Dispensation #${dispensation.id}`, href: '#' },
  ];

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`Chronic Dispensation #${dispensation.id}`} />

      <div className="mx-6 my-6 space-y-6">
        {/* Page header */}
        <div className="flex flex-col justify-between gap-3 sm:flex-row sm:items-center">
          <div>
            <h1 className="text-2xl font-semibold tracking-tight">
              Chronic Dispensation #{dispensation.id}
            </h1>
            <p className="text-muted-foreground mt-1 text-sm">
              Recorded {dispensedAt} {dispensation.dispensed_by?.name ? `• ${dispensation.dispensed_by.name}` : ''}
            </p>
          </div>

          <div className="flex flex-wrap gap-2">
            <Button variant="outline" onClick={() => router.visit(links.patient_show)}>
              <ArrowLeft className="mr-2 h-4 w-4" />
              Patient Profile
            </Button>
            <Button onClick={() => router.visit(links.patient_history)}>
              View Patient History
            </Button>
          </div>
        </div>

        {/* Summary alert */}
        <Alert className="border-blue-200 bg-blue-50 dark:border-blue-900/40 dark:bg-blue-900/20">
          <AlertDescription className="flex items-center gap-2">
            <Pill className="h-4 w-4" />
            <span>
              {dispensation.quantity} x <strong>{medName}</strong>
              {medLine ? ` • ${medLine}` : ''} {dispensation.frequency ? ` • ${dispensation.frequency}` : ''}{' '}
              {dispensation.route ? ` • ${dispensation.route}` : ''}{' '}
              {dispensation.duration ? ` • ${dispensation.duration}` : ''}
            </span>
          </AlertDescription>
        </Alert>

        {/* Content grid */}
        <div className="grid grid-cols-1 gap-6 lg:grid-cols-3">
          {/* Left column */}
          <div className="space-y-6 lg:col-span-2">
            {/* Patient & Medication */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User2 className="h-5 w-5" />
                  Patient & Medication
                </CardTitle>
                <CardDescription>Core details of this dispensation</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {/* Patient */}
                <div>
                  <div className="text-sm text-muted-foreground">Patient</div>
                  <div className="mt-1 flex flex-wrap items-center gap-2">
                    <span className="font-medium">{dispensation.patient.full_name}</span>
                    {dispensation.patient.employee_no ? (
                      <Badge variant="secondary">Emp #{dispensation.patient.employee_no}</Badge>
                    ) : null}
                    {dispensation.patient.gender ? (
                      <Badge variant="outline">{dispensation.patient.gender}</Badge>
                    ) : null}
                    {dispensation.patient.dob ? (
                      <Badge variant="outline">
                        DOB {moment(dispensation.patient.dob).format('YYYY-MM-DD')}
                      </Badge>
                    ) : null}
                  </div>
                </div>

                <Separator />

                {/* Medication */}
                <div className="grid grid-cols-1 gap-4 md:grid-cols-2">
                  <div>
                    <div className="text-sm text-muted-foreground">Medication</div>
                    <div className="mt-1 font-medium">{medName}</div>
                    {medLine ? (
                      <div className="text-muted-foreground text-sm">{medLine}</div>
                    ) : null}
                  </div>

                  <div>
                    <div className="text-sm text-muted-foreground">Batch • Expiry</div>
                    <div className="mt-1 flex flex-wrap items-center gap-2">
                      <Badge variant="outline" className="flex items-center gap-1">
                        <Hash className="h-3.5 w-3.5" /> {dispensation.batch_number ?? '—'}
                      </Badge>
                      <Badge variant="outline">
                        {dispensation.expiry_date ? `Exp ${dispensation.expiry_date}` : '—'}
                      </Badge>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Administration details */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Syringe className="h-5 w-5" />
                  Administration
                </CardTitle>
                <CardDescription>How this medication should be taken</CardDescription>
              </CardHeader>
              <CardContent className="grid grid-cols-1 gap-4 md:grid-cols-2">
                <div>
                  <div className="text-sm text-muted-foreground">Quantity</div>
                  <div className="mt-1 font-medium">{dispensation.quantity}</div>
                </div>

                <div>
                  <div className="text-sm text-muted-foreground">Frequency</div>
                  <div className="mt-1 font-medium">{dispensation.frequency ?? '—'}</div>
                </div>

                <div>
                  <div className="text-sm text-muted-foreground">Route</div>
                  <div className="mt-1 font-medium">{dispensation.route ?? '—'}</div>
                </div>

                <div>
                  <div className="text-sm text-muted-foreground">Duration</div>
                  <div className="mt-1 font-medium">{dispensation.duration ?? '—'}</div>
                </div>
              </CardContent>
            </Card>

            {/* Notes */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <ClipboardList className="h-5 w-5" />
                  Notes
                </CardTitle>
                <CardDescription>Additional information captured at dispensing</CardDescription>
              </CardHeader>
              <CardContent>
                {dispensation.notes ? (
                  <div className="rounded-md border p-3 text-sm">{dispensation.notes}</div>
                ) : (
                  <div className="text-muted-foreground text-sm">No notes added.</div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Right column */}
          <div className="space-y-6">
            {/* Source & timing */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Factory className="h-5 w-5" />
                  Source & Timing
                </CardTitle>
                <CardDescription>Where and when this was dispensed</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <div className="text-sm text-muted-foreground">Origin</div>
                  <div className="mt-1 flex items-center gap-2">
                    <Badge className="capitalize">{dispensation.origin}</Badge>
                    {dispensation.clinic_name ? (
                      <span className="text-sm">{dispensation.clinic_name}</span>
                    ) : null}
                  </div>
                </div>

                <div>
                  <div className="text-sm text-muted-foreground">Dispensed At</div>
                  <div className="mt-1 flex items-center gap-2">
                    <Clock className="h-4 w-4 text-muted-foreground" />
                    <span className="font-medium">{dispensedAt}</span>
                  </div>
                </div>

                <div>
                  <div className="text-sm text-muted-foreground">Dispensed By</div>
                  <div className="mt-1">{dispensation.dispensed_by?.name ?? '—'}</div>
                  {dispensation.dispensed_by?.email ? (
                    <div className="text-muted-foreground text-sm">{dispensation.dispensed_by.email}</div>
                  ) : null}
                </div>

                <Separator />

                <div className="flex flex-col gap-2">
                  <Button onClick={() => router.visit(links.patient_history)}>Back to History</Button>
                  <Button variant="outline" onClick={() => router.visit(links.patient_show)}>
                    Patient Profile
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
