// resources/js/Pages/Chronics/Medications/MedicationDispense.tsx
import AppLayout from '@/layouts/app-layout';
import { Button } from '@/components/ui/button';
import { Head, router, usePage } from '@inertiajs/react';
import { PageProps as InertiaPageProps } from '@inertiajs/core';
import React, { useEffect, useMemo, useState } from 'react';
import { Info as InfoIcon, Pill, Search, Trash2 } from 'lucide-react';
import moment from 'moment';
import Select, { GroupBase, MultiValue, SingleValue, StylesConfig } from 'react-select';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';

/** ===================== Types ===================== */
type MaybePaginated<T> =
  | T[]
  | {
      data: T[];
      current_page: number;
      last_page: number;
      links: Array<{ url: string | null; label: string; active: boolean }>;
    };

type Medication = {
  id: number;
  name: string;
  dosage?: string | null;
  unit?: string | null;
  form?: string | null;
};

type MedicationBatch = {
  id: number;
  batch_number?: string | null;
  expiry_date?: string | null;
  medication: Medication;
};

type ClinicMedicationStock = {
  id: number;
  clinic_id?: number;
  quantity: number;
  medication_batch: MedicationBatch;
};

type NurseMedicationStock = {
  id: number;
  nurse_id?: number;
  clinic_id?: number;
  quantity: number;
  medication_batch: MedicationBatch;
};

type DispensedBy = { id: number; name: string };

type ChronicDispensationItem = {
  id: number;
  patient_id: number;
  clinic_medication_stock?: ClinicMedicationStock | null;
  nurse_medication_stock?: NurseMedicationStock | null;
  quantity: number;
  frequency?: string | null;
  route?: string | null;
  duration?: string | null; // <-- string to match backend validation
  notes?: string | null;
  dispensed_at?: string | null;
  dispensed_by?: DispensedBy | null;
};

type Patient = {
  id: number;
  first_name: string;
  middle_name?: string | null;
  surname: string;
  employee_number?: string | null;
  company?: { id: number; name: string } | null;
};

interface PageProps extends InertiaPageProps {
  patient: Patient;
  clinicMedicationStocks: MaybePaginated<ClinicMedicationStock>;
  nurseMedicationStocks: MaybePaginated<NurseMedicationStock>;
  dispensations: ChronicDispensationItem[];
  frequencyOptions: Record<string, string>;
  routeOptions: Record<string, string>;
  filters: { search?: string | null };
}

/** ===================== Utils ===================== */
function isPaginated<T>(v: MaybePaginated<T>): v is Extract<MaybePaginated<T>, { data: T[] }> {
  return !!v && typeof v === 'object' && 'data' in v && Array.isArray((v as any).data);
}

const uuid = () =>
  (crypto as any)?.randomUUID
    ? (crypto as any).randomUUID()
    : `${Date.now()}-${Math.random().toString(16).slice(2)}`;

function useThemeTokens() {
  const read = (name: string, fallback: string) =>
    typeof window !== 'undefined'
      ? getComputedStyle(document.documentElement).getPropertyValue(name).trim() || fallback
      : fallback;

  const bg = read('--background', '#fff');
  const fg = read('--foreground', '#0a0a0a');
  const border = read('--border', '220 13% 91%');
  const muted = read('--muted', '220 14% 96%');
  const ring = read('--ring', '221 83% 53%');

  const hsl = (triple: string) => (triple.includes('%') ? `hsl(${triple})` : triple);
  return { bg, fg, border: hsl(border), muted: hsl(muted), ring: hsl(ring) };
}

function makeSelectStyles<
  T,
  M extends boolean = false,
  G extends GroupBase<T> = GroupBase<T>
>(tokens: ReturnType<typeof useThemeTokens>): StylesConfig<T, M, G> {
  return {
    container: (base) => ({ ...base, width: '100%' }),
    control: (base, state) => ({
      ...base,
      minHeight: 36,
      borderRadius: 8,
      backgroundColor: tokens.bg,
      borderColor: state.isFocused ? tokens.ring : tokens.border,
      boxShadow: state.isFocused ? `0 0 0 2px ${tokens.ring}22` : 'none',
      ':hover': { borderColor: state.isFocused ? tokens.ring : tokens.border },
    }),
    valueContainer: (b) => ({ ...b, paddingBlock: 2, paddingInline: 8 }),
    input: (b) => ({ ...b, color: tokens.fg }),
    singleValue: (b) => ({ ...b, color: tokens.fg }),
    placeholder: (b) => ({ ...b, color: `${tokens.fg}99` }),
    menu: (b) => ({
      ...b,
      zIndex: 60,
      backgroundColor: tokens.bg,
      border: `1px solid ${tokens.border}`,
      boxShadow: `0 8px 24px rgba(0,0,0,.12)`,
      overflow: 'hidden',
      borderRadius: 8,
    }),
    menuList: (b) => ({ ...b, paddingBlock: 4 }),
    option: (b, s) => ({
      ...b,
      backgroundColor: s.isSelected ? `${tokens.ring}22` : s.isFocused ? `${tokens.muted}` : 'transparent',
      color: tokens.fg,
      cursor: 'pointer',
    }),
  };
}

/** ===================== Component ===================== */
export default function ChronicMedicationDispense() {
  const {
    patient,
    clinicMedicationStocks,
    nurseMedicationStocks,
    filters,
    dispensations,
    frequencyOptions,
    routeOptions,
  } = usePage<PageProps>().props;

  const tokens = useThemeTokens();
  const selectStyles = makeSelectStyles(tokens);

  const [searchQuery, setSearchQuery] = useState(filters?.search || '');

  // Flatten current stocks depending on source
  const clinicStocks = useMemo<ClinicMedicationStock[]>(
    () => (isPaginated(clinicMedicationStocks) ? clinicMedicationStocks.data : clinicMedicationStocks),
    [clinicMedicationStocks]
  );
  const nurseStocks = useMemo<NurseMedicationStock[]>(
    () => (isPaginated(nurseMedicationStocks) ? nurseMedicationStocks.data : nurseMedicationStocks),
    [nurseMedicationStocks]
  );

  // Choose stock source
  const stockTypeOptions = [
    { value: 'clinic', label: 'Clinic Stock' },
    { value: 'nurse', label: 'Nurse Stock' },
  ];
  const [globalStockType, setGlobalStockType] = useState<'clinic' | 'nurse'>('nurse');
  const currentStocks =
    globalStockType === 'clinic'
      ? (clinicStocks as ClinicMedicationStock[])
      : (nurseStocks as NurseMedicationStock[]);

  // Select medications (multi)
  type MedOpt = { value: string; label: string };
  const medicationOptions: MedOpt[] = useMemo(
    () =>
      currentStocks.map((s) => ({
        value: String(s.id),
        label: `${s.medication_batch.medication.name} (${s.quantity} avail) — ${s.medication_batch.medication.dosage ?? ''}${s.medication_batch.medication.unit ?? ''}`,
      })),
    [currentStocks]
  );
  const [selectedMeds, setSelectedMeds] = useState<MedOpt[]>([]);

  // Row fields per selected stock
  type RowFields = {
    quantity: string;
    frequency: string;
    route: string;
    duration: string; // keep as string in UI
    notes: string;
  };
  const [rowsByStockId, setRowsByStockId] = useState<Record<string, RowFields>>({});

  // On source change, reset selection
  const onChangeGlobalStockType = (opt: SingleValue<{ value: string; label: string }>) => {
    const value = (opt?.value || 'nurse') as 'clinic' | 'nurse';
    setGlobalStockType(value);
    setSelectedMeds([]);
    setRowsByStockId({});
  };

  // Keep rows aligned with selection
  useEffect(() => {
    setRowsByStockId((prev) => {
      const next: Record<string, RowFields> = {};
      selectedMeds.forEach((m) => {
        next[m.value] =
          prev[m.value] || { quantity: '', frequency: '', route: '', duration: '', notes: '' };
      });
      return next;
    });
  }, [selectedMeds]);

  const updateRow = (stockId: string, patch: Partial<RowFields>) =>
    setRowsByStockId((prev) => ({ ...prev, [stockId]: { ...prev[stockId], ...patch } }));

  const removeSelectedMed = (stockId: string) =>
    setSelectedMeds((prev) => prev.filter((m) => m.value !== stockId));

  const findStock = (stock_id: string) =>
    currentStocks.find((s) => String(s.id) === stock_id);

  /** ===== Search & paging ===== */
  const gotoSelf = (params: Record<string, any> = {}, only?: string[]) =>
    router.get(
      route('chronics.patients.dispense', patient.id),
      params,
      {
        preserveState: true,
        preserveScroll: true,
        replace: true,
        ...(only ? { only } : {}),
      }
    );

  const handleSearch = () =>
    gotoSelf({ search: searchQuery }, ['clinicMedicationStocks', 'nurseMedicationStocks', 'filters']);

  const handleResetSearch = () => {
    setSearchQuery('');
    gotoSelf({}, ['clinicMedicationStocks', 'nurseMedicationStocks', 'filters']);
  };

  const handlePageChange = (selected: { selected: number }) =>
    gotoSelf(
      { page: selected.selected + 1, search: searchQuery },
      ['clinicMedicationStocks', 'nurseMedicationStocks', 'filters']
    );

  /** ===== Submit batch ===== */
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleSubmitBatch = async (e: React.FormEvent) => {
    e.preventDefault();
    if (isSubmitting) return;

    const items = selectedMeds
      .map((m) => {
        const f = rowsByStockId[m.value] || {};
        const quantity = f.quantity === '' ? 0 : parseInt(f.quantity, 10);
        const duration = f.duration === '' ? null : f.duration.trim(); // <<< send as string

        return {
          stock_type: globalStockType, // 'clinic' | 'nurse'
          stock_id: Number(m.value),
          quantity,
          frequency: f.frequency || null,
          route: f.route || null,
          duration, // string | null
          notes: f.notes || null,
          request_id: uuid(),
        };
      })
      .filter((it) => it.stock_id && it.quantity && it.quantity > 0);

    if (items.length === 0) {
      return Swal.fire('Nothing to dispense', 'Pick medication(s) and enter valid quantity.', 'warning');
    }

    // Validate against current stock and duration > 0 if numeric
    for (const it of items) {
      const stock = currentStocks.find((s) => s.id === it.stock_id);
      if (!stock) return Swal.fire('Invalid selection', 'One of the selected stocks no longer exists.', 'error');
      if (stock.quantity < (it.quantity ?? 0)) {
        return Swal.fire(
          'Insufficient stock',
          `Only ${stock.quantity} available for ${stock.medication_batch.medication.name}.`,
          'warning'
        );
      }
      if (it.duration && /^\d+$/.test(it.duration) && parseInt(it.duration, 10) <= 0) {
        return Swal.fire('Invalid duration', 'Duration must be greater than 0 days.', 'warning');
      }
    }

    setIsSubmitting(true);
    Swal.fire({
      title: 'Dispensing…',
      text: 'Please wait while we record all medications.',
      allowOutsideClick: false,
      allowEscapeKey: false,
      didOpen: () => Swal.showLoading(),
    });

    router.post(
      route('chronics.dispensations.store.batch', patient.id),
      { patient_id: patient.id, items },
      {
        preserveScroll: true,
        onSuccess: async () => {
          Swal.close();
          await Swal.fire('Success', 'Medications dispensed successfully.', 'success');
          setSelectedMeds([]);
          setRowsByStockId({});
          router.reload({
            only: ['dispensations', 'clinicMedicationStocks', 'nurseMedicationStocks'],
            preserveScroll: true,
          });
        },
        onError: async (errors) => {
          Swal.close();
          const msg = (errors && typeof errors === 'object')
            ? Object.values(errors as Record<string, any>).join('\n')
            : 'Failed to dispense medications.';
          await Swal.fire('Error', msg || 'Failed to dispense medications.', 'error');
        },
        onFinish: () => setIsSubmitting(false),
      }
    );
  };

  /** ===== Close & go to history ===== */
  const handleCloseAndGoHistory = async () => {
    const res = await Swal.fire({
      title: 'Finish dispensing?',
      text: 'You will be taken to this patient’s chronic dispensation history.',
      icon: 'question',
      showCancelButton: true,
      confirmButtonText: 'Close & View History',
      cancelButtonText: 'Stay Here',
      reverseButtons: true,
      focusCancel: true,
    });
    if (res.isConfirmed) {
      router.visit(route('chronics.patients.dispensations.history', patient.id));
    }
  };

  /** ===== UI ===== */
  const fullName = [patient.first_name, patient.middle_name, patient.surname].filter(Boolean).join(' ');
  const breadcrumbs = [
    { title: 'Chronics', href: route('chronics.dashboard') },
    { title: fullName, href: route('patients.show', patient.id) },
    { title: 'Dispense Medication', href: '#' },
  ];

  const showClinicPagination = isPaginated(clinicMedicationStocks);
  const showNursePagination = isPaginated(nurseMedicationStocks);
  const stockBannerText = globalStockType === 'clinic' ? 'Clinic Stock' : 'Nurse Stock';

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <Head title={`Chronic Dispense — ${fullName}`} />

      <div className="bg-background text-foreground mx-8 my-6 rounded-xl p-6 shadow-sm">
        <div className="mb-6 flex items-center justify-between">
          <h1 className="text-2xl font-bold">
            Dispense Medication to <span style={{ textTransform: 'capitalize' }}>{fullName}</span>
            {patient.company?.name ? (
              <span className="text-muted-foreground ml-2 text-base">• {patient.company.name}</span>
            ) : null}
          </h1>
        </div>

        <div className="grid grid-cols-12 gap-8">
          {/* LEFT: Form + lists */}
          <div className="col-span-12 lg:col-span-8 flex flex-col gap-4">
            {/* Banner */}
            <div className="flex items-start gap-3 rounded-md border border-blue-200 bg-blue-50 p-3 text-sm dark:border-blue-900/50 dark:bg-blue-900/20">
              <InfoIcon className="mt-0.5 h-4 w-4" />
              <div>
                Chronic dispensing is <strong>not linked to a consultation</strong>. You are dispensing from{' '}
                <strong>{stockBannerText}</strong>. Choose medications once, then fill their details below.
              </div>
            </div>

            {/* Search */}
            <div className="mb-2 flex gap-2">
              <div className="relative w-72">
                <Search className="text-muted-foreground absolute left-3 top-2.5 h-5 w-5" />
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => {
                    const v = e.target.value;
                    setSearchQuery(v);
                    if (v === '') handleResetSearch();
                  }}
                  onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                  placeholder="Search medications..."
                  className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:outline-none focus:ring-2 focus:ring-blue-600"
                />
              </div>
              <button
                onClick={handleSearch}
                className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
              >
                Search
              </button>
              <button
                onClick={handleResetSearch}
                className="rounded-lg border px-4 py-2 text-sm font-medium hover:bg-muted"
              >
                Reset
              </button>
            </div>

            {/* Stock source + meds multi-select */}
            <div className="grid grid-cols-1 gap-3 sm:grid-cols-2">
              <div className="max-w-xs">
                <label className="mb-1 block text-sm font-medium">Stock Source</label>
                <Select
                  instanceId="global-stock-type"
                  styles={selectStyles}
                  options={stockTypeOptions}
                  value={stockTypeOptions.find((o) => o.value === globalStockType) || null}
                  onChange={onChangeGlobalStockType}
                  classNamePrefix="rs"
                  menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                  menuPosition="fixed"
                />
              </div>

              <div>
                <label className="mb-1 block text-sm font-medium">Medications to dispense (multi-select)</label>
                <Select
                  instanceId="meds-multi"
                  isMulti
                  styles={selectStyles}
                  options={medicationOptions}
                  value={selectedMeds}
                  onChange={(vals: MultiValue<MedOpt>) => setSelectedMeds(vals as MedOpt[])}
                  classNamePrefix="rs"
                  placeholder="Select one or more medications…"
                  menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                  menuPosition="fixed"
                />
              </div>
            </div>

            {/* Compact rows (one row per selected med) */}
            <form onSubmit={handleSubmitBatch} className="space-y-3">
              <div className="overflow-x-auto">
                <table className="min-w-full border text-sm">
                  <thead className="bg-muted">
                    <tr>
                      <th className="px-3 py-2 text-left">Medication</th>
                      <th className="px-3 py-2 text-left">Qty</th>
                      <th className="px-3 py-2 text-left">Freq</th>
                      <th className="px-3 py-2 text-left">Route</th>
                      <th className="px-3 py-2 text-left">Days</th>
                      <th className="px-3 py-2 text-left">Notes</th>
                      <th className="px-3 py-2"></th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {selectedMeds.map((med) => {
                      const stock = findStock(med.value) as ClinicMedicationStock | NurseMedicationStock | undefined;
                      const row = rowsByStockId[med.value] || {
                        quantity: '',
                        frequency: '',
                        route: '',
                        duration: '',
                        notes: '',
                      };

                      const batch = stock?.medication_batch?.batch_number ?? '-';
                      const avail = stock?.quantity ?? 0;

                      return (
                        <tr key={med.value} className="align-top">
                          <td className="px-2 py-2 min-w-[260px]">
                            <div className="font-medium">{med.label.split(' — ')[0]}</div>
                            <div className="mt-1 text-xs text-muted-foreground">
                              Available: <strong>{avail}</strong> — batch {batch}
                            </div>
                          </td>

                          <td className="w-[90px] px-2 py-2">
                            <input
                              type="text"
                              value={row.quantity}
                              onChange={(e) => {
                                const v = e.target.value;
                                if (v === '' || /^[0-9\b]+$/.test(v)) updateRow(med.value, { quantity: v });
                              }}
                              className="w-full rounded-md border px-2 py-1"
                              placeholder="0"
                            />
                          </td>

                          <td className="w-[160px] px-2 py-2">
                            <Select
                              instanceId={`freq-${med.value}`}
                              styles={selectStyles}
                              options={Object.entries(frequencyOptions).map(([value, label]) => ({ value, label }))}
                              value={
                                row.frequency
                                  ? { value: row.frequency, label: frequencyOptions[row.frequency] ?? row.frequency }
                                  : null
                              }
                              onChange={(opt) => updateRow(med.value, { frequency: opt?.value || '' })}
                              isClearable
                              classNamePrefix="rs"
                              menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                              menuPosition="fixed"
                            />
                          </td>

                          <td className="w-[160px] px-2 py-2">
                            <Select
                              instanceId={`route-${med.value}`}
                              styles={selectStyles}
                              options={Object.entries(routeOptions).map(([value, label]) => ({ value, label }))}
                              value={row.route ? { value: row.route, label: routeOptions[row.route] ?? row.route } : null}
                              onChange={(opt) => updateRow(med.value, { route: opt?.value || '' })}
                              isClearable
                              classNamePrefix="rs"
                              menuPortalTarget={typeof document !== 'undefined' ? document.body : undefined}
                              menuPosition="fixed"
                            />
                          </td>

                          <td className="w-[90px] px-2 py-2">
                            <input
                              type="text"
                              value={row.duration}
                              onChange={(e) => {
                                const v = e.target.value;
                                if (v === '' || /^[0-9\b]+$/.test(v)) updateRow(med.value, { duration: v });
                              }}
                              className="w-full rounded-md border px-2 py-1"
                              placeholder="0"
                            />
                          </td>

                          <td className="px-2 py-2">
                            <input
                              type="text"
                              value={row.notes}
                              onChange={(e) => updateRow(med.value, { notes: e.target.value })}
                              className="w-full rounded-md border px-2 py-1"
                              placeholder="Notes"
                            />
                          </td>

                          <td className="w-[46px] px-2 py-2">
                            <button
                              type="button"
                              onClick={() => removeSelectedMed(med.value)}
                              className="inline-flex items-center rounded border px-2 py-1 text-xs hover:bg-muted"
                              title="Remove"
                            >
                              <Trash2 className="h-4 w-4" />
                            </button>
                          </td>
                        </tr>
                      );
                    })}
                    {selectedMeds.length === 0 && (
                      <tr>
                        <td colSpan={7} className="px-3 py-6 text-center text-sm text-muted-foreground">
                          Select one or more medications above to start dispensing.
                        </td>
                      </tr>
                    )}
                  </tbody>
                </table>
              </div>

              <div className="flex items-center justify-end">
                <Button
                  type="submit"
                  disabled={isSubmitting}
                  className="inline-flex items-center gap-2"
                  style={{ cursor: isSubmitting ? 'not-allowed' : 'pointer' }}
                >
                  <Pill className="h-4 w-4" />
                  {isSubmitting ? 'Processing…' : 'Dispense Selected'}
                </Button>
              </div>
            </form>

            {/* Dispensed list (chronic) */}
            <div className="bg-card rounded-lg border p-4 shadow-sm" style={{ marginTop: 10, marginBottom: 10 }}>
              <h2 className="mb-4 flex items-center gap-2 text-lg font-semibold">
                <Pill className="text-muted-foreground h-5 w-5" />
                Chronic Dispensed Medications
              </h2>
              {Array.isArray(dispensations) && dispensations.length > 0 ? (
                <div className="overflow-x-auto">
                  <table className="min-w-full divide-y text-sm">
                    <thead className="bg-muted">
                      <tr>
                        <th className="px-3 py-2 text-left">Medication</th>
                        <th className="px-3 py-2 text-left">Qty</th>
                        <th className="px-3 py-2 text-left">Frequency</th>
                        <th className="px-3 py-2 text-left">Route</th>
                        <th className="px-3 py-2 text-left">Duration</th>
                        <th className="px-3 py-2 text-left">Batch</th>
                        <th className="px-3 py-2 text-left">Dispensed By</th>
                        <th className="px-3 py-2 text-left">Date</th>
                      </tr>
                    </thead>
                    <tbody className="divide-y">
                      {dispensations.map((d) => {
                        const medObj =
                          d.clinic_medication_stock?.medication_batch?.medication ||
                          d.nurse_medication_stock?.medication_batch?.medication ||
                          ({} as Medication);
                        const medName = medObj?.name ?? '—';
                        const dosage = medObj?.dosage ?? '';
                        const unit = medObj?.unit ?? '';
                        const form = medObj?.form ?? '';
                        const batch =
                          d.clinic_medication_stock?.medication_batch?.batch_number ??
                          d.nurse_medication_stock?.medication_batch?.batch_number ??
                          '—';

                        return (
                          <tr key={d.id} className="hover:bg-muted/50">
                            <td className="px-3 py-2">
                              <span className="inline-block rounded-full bg-blue-100 px-3 py-1 text-xs font-semibold text-blue-800 dark:bg-blue-800/20 dark:text-blue-300">
                                {medName}
                              </span>
                              <div className="mt-1 text-xs text-muted-foreground">
                                {(dosage || unit) ? `${dosage}${unit ? ` ${unit}` : ''}` : '-'}
                                {form ? ` • ${form}` : ''}
                              </div>
                            </td>
                            <td className="px-3 py-2">{d.quantity}</td>
                            <td className="px-3 py-2">{d.frequency || '—'}</td>
                            <td className="px-3 py-2">{d.route || '—'}</td>
                            <td className="px-3 py-2">
                              {d.duration != null && d.duration !== ''
                                ? `${d.duration}${/^\d+$/.test(String(d.duration)) ? ` day${Number(d.duration) === 1 ? '' : 's'}` : ''}`
                                : '—'}
                            </td>
                            <td className="px-3 py-2">{batch}</td>
                            <td className="px-3 py-2">{d.dispensed_by?.name ?? '—'}</td>
                            <td className="px-3 py-2">
                              {d.dispensed_at ? moment(d.dispensed_at).format('ddd D MMM YYYY • HH:mm') : '—'}
                            </td>
                          </tr>
                        );
                      })}
                    </tbody>
                  </table>
                </div>
              ) : (
                <p className="text-muted-foreground text-sm">No chronic medications dispensed yet.</p>
              )}
            </div>

            {/* Stock table + pagination */}
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y border text-sm">
                <thead className="bg-muted">
                  <tr>
                    {['Medication', 'Dosage', 'Form', 'Available Qty', 'Batch', 'Expiry'].map((h) => (
                      <th key={h} className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase">{h}</th>
                    ))}
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {currentStocks.map((stock) => (
                    <tr key={stock.id} className="transition hover:bg-muted/50">
                      <td className="px-4 py-3">{stock.medication_batch.medication.name}</td>
                      <td className="px-4 py-3">{stock.medication_batch.medication.dosage || '-'}</td>
                      <td className="px-4 py-3">{stock.medication_batch.medication.form || '-'}</td>
                      <td className="px-4 py-3">{stock.quantity}</td>
                      <td className="px-4 py-3">{stock.medication_batch.batch_number || '-'}</td>
                      <td className="px-4 py-3">{stock.medication_batch.expiry_date || '-'}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>

            {globalStockType === 'clinic' && showClinicPagination && (
              <div className="mt-6 flex justify-center">
                <ReactPaginate
                  pageCount={(clinicMedicationStocks as any).last_page}
                  forcePage={(clinicMedicationStocks as any).current_page - 1}
                  onPageChange={handlePageChange}
                  marginPagesDisplayed={1}
                  pageRangeDisplayed={3}
                  previousLabel="← Prev"
                  nextLabel="Next →"
                  breakLabel="..."
                  containerClassName="flex items-center gap-2 text-sm"
                  pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  activeClassName="bg-blue-600 text-white"
                  previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  breakClassName="px-2"
                />
              </div>
            )}
            {globalStockType === 'nurse' && showNursePagination && (
              <div className="mt-6 flex justify-center">
                <ReactPaginate
                  pageCount={(nurseMedicationStocks as any).last_page}
                  forcePage={(nurseMedicationStocks as any).current_page - 1}
                  onPageChange={handlePageChange}
                  marginPagesDisplayed={1}
                  pageRangeDisplayed={3}
                  previousLabel="← Prev"
                  nextLabel="Next →"
                  breakLabel="..."
                  containerClassName="flex items-center gap-2 text-sm"
                  pageClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  activeClassName="bg-blue-600 text-white"
                  previousClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  nextClassName="px-3 py-1 rounded border border-border hover:bg-muted"
                  breakClassName="px-2"
                />
              </div>
            )}
          </div>

          {/* RIGHT: Actions */}
          <div className="col-span-12 lg:col-span-4 space-y-4">
            <div className="bg-card rounded-lg border p-6 shadow-sm">
              <h2 className="mb-4 text-xl font-semibold">Actions</h2>
              <Button className="w-full" onClick={handleCloseAndGoHistory} style={{ cursor: 'pointer' }}>
                Close & View History
              </Button>
              <Button
                variant="outline"
                className="mt-2 w-full"
                onClick={() => router.visit(route('patients.show', patient.id))}
                style={{ cursor: 'pointer' }}
              >
                Back to Patient
              </Button>
            </div>
          </div>
        </div>
      </div>
    </AppLayout>
  );
}
