import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { FormEvent } from 'react';
import * as Yup from 'yup';

interface ClinicData {
    id: number;
    name: string;
    address: string;
    city: string;
    afhoz_number: string;
    contact_person: string;
    manager: string;
    country: string;
    email_address: string;
    [key: string]: any;
}

export default function EditClinic({ clinic }: { clinic: ClinicData }) {
    const { data, setData, put, processing, errors } = useForm<ClinicData>({ ...clinic });

    const schema = Yup.object().shape({
        name: Yup.string().required('Clinic name is required'),
        email_address: Yup.string().email('Invalid email').nullable(),
    });

    const handleSubmit = async (e: FormEvent) => {
        e.preventDefault();
        try {
            await schema.validate(data, { abortEarly: false });
            put(route('clinics.update', clinic.id), {
                preserveScroll: true,
            });
        } catch (validationError: any) {
            const fieldErrors: Record<string, string> = {};
            validationError.inner.forEach((err: any) => {
                if (err.path) fieldErrors[err.path] = err.message;
            });
        }
    };

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Clinics', href: route('clinics.index') },
        { title: 'Edit Clinic', href: route('clinics.edit', clinic.id) },
    ];

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Edit Clinic" />

            <div className="w-full space-y-8 px-4 py-10 sm:px-6">
                <h1 className="text-foreground text-2xl font-bold">Edit Clinic</h1>

                <form onSubmit={handleSubmit} className="space-y-8">
                    <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                        {['name', 'address', 'city', 'afhoz_number', 'contact_person', 'manager', 'country', 'email_address'].map((name) => (
                            <div key={name} className="grid gap-2">
                                <Label htmlFor={name}>
                                    {name.replace(/_/g, ' ').replace(/\b\w/g, (l) => l.toUpperCase())}
                                    {name === 'name' && <span className="text-red-500">*</span>}
                                </Label>
                                <Input
                                    id={name}
                                    name={name}
                                    type={name === 'email_address' ? 'email' : 'text'}
                                    value={data[name]}
                                    onChange={(e) => setData(name, e.target.value)}
                                />
                                <InputError message={errors[name]} />
                            </div>
                        ))}
                    </div>

                    <div className="flex justify-end gap-3">
                        <Button type="button" variant="outline" onClick={() => window.history.back()}>
                            Cancel
                        </Button>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Updating...' : 'Update Clinic'}
                        </Button>
                    </div>
                </form>
            </div>
        </AppLayout>
    );
}
