import { Head, Link, router, usePage } from '@inertiajs/react';
import AppLayout from '@/layouts/app-layout';
import { BreadcrumbItem } from '@/types';
import { Eye, Pencil, Trash, Download } from 'lucide-react';
import { useState } from 'react';
import ReactPaginate from 'react-paginate';
import Swal from 'sweetalert2';
import { PagePropsWithMedicationRequests } from '@/types';
import { Modal, Form, Input, Button, message } from 'antd';
import { useDarkMode } from '@/hooks/useDarkMode'; // Ensure you have a hook for dark mode
import moment from 'moment';

import * as XLSX from 'xlsx';


const breadcrumbs: BreadcrumbItem[] = [{ title: 'Clinics', href: '/clinics' }];

export default function MedicationRequests() {
    const { clinic, requests, filters } = usePage<PagePropsWithMedicationRequests>().props;
    const [searchQuery, setSearchQuery] = useState(filters.search || '');
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editRequestId, setEditRequestId] = useState<number | null>(null);
    const [form] = Form.useForm();

    const handleSearch = () => {
        router.get(route('clinics.medication-requests.index', { clinic: clinic.id }), { search: searchQuery }, { preserveState: true });
    };

    const handlePageChange = (selected: { selected: number }) => {
        router.get(route('clinics.medication-requests.index', { clinic: clinic.id }), { page: selected.selected + 1, search: searchQuery }, { preserveState: true });
    };

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Are you sure?',
            text: 'This medication request will be permanently deleted.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e3342f',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Yes, delete it!',
            width: '350px',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('medication-requests.destroy', id), {
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'The medication request has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false,
                        });
                    },
                    onError: () => {
                        Swal.fire({
                            title: 'Error!',
                            text: 'Something went wrong.',
                            icon: 'error',
                        });
                    },
                });
            }
        });
    };

    const handleEdit = (id: number) => {
        setEditRequestId(id);
        setIsModalOpen(true);
        const request = requests.data.find((req) => req.id === id);
        if (request) {
            form.setFieldsValue({
                quantity: request.quantity,
                status: request.status,
            });
        }
    };

    const handleModalCancel = () => {
        setIsModalOpen(false);
        form.resetFields();
        setEditRequestId(null);
    };

    const handleModalOk = () => {
        form.validateFields().then((values) => {
            if (editRequestId) {
                // Show loading spinner
                const loading = Swal.fire({
                    title: 'Updating...',
                    allowOutsideClick: false,
                    allowEscapeKey: false,
                    didOpen: () => {
                        Swal.showLoading();
                    },
                });

                router.put(route('medication-requests.update', editRequestId), values, {
                    onSuccess: () => {
                        // Hide loading spinner and show success message
                        Swal.close();
                        message.success('Medication request updated successfully.');
                        setIsModalOpen(false);
                        form.resetFields();
                        setEditRequestId(null);
                    },
                    onError: (errors) => {
                        // Hide loading spinner and show error message
                        Swal.close();
                        message.error('Something went wrong.');
                        console.error(errors); // Log errors for debugging
                    },
                });
            }
        });
    };

    const { isDarkMode } = useDarkMode();


    const exportToExcel = () => {
        const data = requests.data.map((request: any) => ({
            'Date of Request': request.request_date || '-',
            'Requested Medication': request.medication?.name || '',
            'Dosage': request.medication?.dosage || '',
            'Unit': request.medication?.unit || '',
            'Quantity': request.quantity || '',

        }))

        const worksheet = XLSX.utils.json_to_sheet(data);

        // Get the keys from the first data object (all column headers)
        const keys = data.length > 0 ? Object.keys(data[0]) : [];

        // Calculate max length for each column dynamically
        const cols = keys.map((key) => {
            const maxLength = Math.max(
                key.length,
                ...data.map((row: { [x: string]: { toString: () => { (): any; new(): any; length: any; }; }; }) => (row[key] ? row[key].toString().length : 0))
            );
            return { wch: maxLength + 2 };
        });

        // Assign column widths to worksheet
        worksheet['!cols'] = cols;

        // Create workbook and append worksheet
        const workbook = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(workbook, worksheet, 'Consultations');

        // Generate file and trigger download
        XLSX.writeFile(workbook, 'Medication_Request.xlsx', {
            compression: true,
        });
    }

    return (
        <AppLayout breadcrumbs={breadcrumbs}>
            <Head title="Medication Requests" />
            <div className="bg-background text-foreground mx-8 my-6 flex flex-1 flex-col gap-4 rounded-xl p-6 shadow-sm">
                <div className="mb-4 flex flex-wrap items-center justify-between gap-4">
                    <h1 className="text-2xl font-bold">Medication Requests for {clinic.name}</h1>
                    <div className="flex items-center gap-3">
                        <button
                            onClick={exportToExcel}
                            className="inline-flex items-center gap-2 rounded-lg bg-green-600 px-4 py-2 text-sm font-medium text-white hover:bg-green-700"
                            style={{
                                cursor: 'pointer'
                            }}
                        >
                            <Download className="h-4 w-4" />
                            Export Excel
                        </button>

                    </div>
                </div>

                <div className="mb-4 flex gap-2">
                    <div className="relative w-72">
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                            placeholder="Search medication requests..."
                            className="border-border bg-background placeholder-muted-foreground w-full rounded-lg border py-2 pr-3 pl-10 text-sm focus:ring-2 focus:ring-blue-600 focus:outline-none"
                        />
                    </div>
                    <button onClick={handleSearch} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700">
                        Search
                    </button>
                </div>

                <div className="overflow-x-auto rounded-xl border">
                    <table className="min-w-full divide-y overflow-hidden rounded-xl bg-white text-sm dark:bg-neutral-900">
                        <thead className="bg-muted">
                            <tr>
                                {['Medication', 'Quantity', 'Request Date', 'Status', 'Actions'].map((header) => (
                                    <th
                                        key={header}
                                        className="text-muted-foreground px-4 py-3 text-left text-xs font-semibold tracking-wider uppercase"
                                    >
                                        {header}
                                    </th>
                                ))}
                            </tr>
                        </thead>
                        <tbody className="divide-y">
                            {requests.data.map((request) => (
                                <tr key={request.id} className="hover:bg-muted/50 transition">
                                    <td className="px-4 py-3">{request.medication.name}</td>
                                    <td className="px-4 py-3">{request.quantity}</td>
                                    <td className="px-4 py-3">
                                        {moment(request.request_date).format('MMM D, YYYY')}
                                    </td>
                                    <td className="px-4 py-3">
                                        <span
                                            className={`inline-block rounded-full px-3 py-1 text-sm font-semibold ${request.status === 'approved'
                                                ? 'bg-green-100 text-green-800'
                                                : request.status === 'pending'
                                                    ? 'bg-yellow-100 text-yellow-800'
                                                    : request.status === 'rejected'
                                                        ? 'bg-red-100 text-red-800'
                                                        : 'bg-gray-100 text-gray-800'
                                                }`}
                                            style={{ cursor: 'pointer', textTransform: "uppercase" }}
                                        >
                                            {request.status}
                                        </span>
                                    </td>

                                    <td className="flex items-center gap-2 px-4 py-3">
                                        <button
                                            onClick={() => handleEdit(request.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-yellow-500 p-2 text-white hover:bg-yellow-600"
                                            title="Edit"
                                            style={{ cursor: 'pointer' }}
                                        >
                                            <Pencil className="h-4 w-4" />
                                        </button>
                                        <button
                                            onClick={() => handleDelete(request.id)}
                                            className="inline-flex items-center justify-center rounded-full bg-red-600 p-2 text-white hover:bg-red-700"
                                            title="Delete"
                                            style={{ cursor: 'pointer' }}
                                        >
                                            <Trash className="h-4 w-4" />
                                        </button>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                <div className="mt-6 flex justify-center">
                    <ReactPaginate
                        pageCount={requests.last_page}
                        forcePage={requests.current_page - 1}
                        onPageChange={handlePageChange}
                        marginPagesDisplayed={1}
                        pageRangeDisplayed={3}
                        previousLabel="← Prev"
                        nextLabel="Next →"
                        breakLabel="..."
                        containerClassName="flex items-center gap-2 text-sm"
                        pageClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        activeClassName="bg-blue-600 text-white"
                        previousClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        nextClassName="px-3 py-1 border border-border rounded hover:bg-muted"
                        breakClassName="px-2"
                    />
                </div>
            </div>

            <Modal
                title="Edit Medication Request"
                open={isModalOpen}
                onCancel={handleModalCancel}
                footer={[
                    <Button key="back" onClick={handleModalCancel}>
                        Cancel
                    </Button>,
                    <Button key="submit" type="primary" onClick={handleModalOk}>
                        Update
                    </Button>,
                ]}
                className={isDarkMode ? 'dark-modal' : ''}
            >
                <Form form={form} layout="vertical">
                    <Form.Item label="Quantity" name="quantity" rules={[{ required: true, message: 'Please enter the quantity!' }]}>
                        <Input type="number" min={1} />
                    </Form.Item>
                    <Form.Item label="Status" name="status" rules={[{ required: true, message: 'Please enter the status!' }]}>
                        <Input />
                    </Form.Item>
                </Form>
            </Modal>
        </AppLayout>
    );
}